#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Log;

use strict;
use warnings;

use TSH::Utility qw(Error OpenFile);

=pod

=head1 NAME

TSH::Log - Record displayed information in text and HTML format

=head1 SYNOPSIS

  my $log = new TSH::Log($tournament, $division, $type, $round, \%options);
  $log->Write($text, $html);
  $log->Close();

=head1 ABSTRACT

This class manages the copying of information displayed on the
console to text and HTML files.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$;$$);
sub Close($);
sub new ($$$$;$$);
sub MakeHTMLIndex ($);
sub ReportHeader ($$$$);
sub ReportTrailer ();
sub Write ($$$);
sub Write3 ($$$$);

=item $log->Close()

Close a log and all files associated with it.

=cut

sub Close ($) {
  my $this = shift;
  $this->Write('', "</table>\n");
  {
    my ($text, $html) = ReportTrailer;
    $this->Write($text, $html);
  }
  for my $fh (qw(text_fh html_fh subhtml_fh)) {
    if ($this->{$fh}) {
      close($this->{$fh});
      $this->{$fh} = undef;
      }
    }
  $this->MakeHTMLIndex();
  }

=item $log->initialise($tournament, $division, $filename, $round, \%options)

Initialise the log object, create files.
Options (and defaults): 

noconsole (0): do not write to console

notitle (0): do not display title except in HTML <title>

title (ucfirst $filename): human-readable title

=cut

sub initialise ($$$$;$$) {
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;
  my $filename = lc shift;
  my $round = lc (shift||'');
  my $optionsp = $this->{'options'} = shift;

  my $config = $tournament->Config();

  $this->{'tournament'} = $tournament;

  my ($fh, $fn, %fhs);
  my $round_000 = 
    $round =~ /^\d+$/ ? sprintf("-%03d", $round) 
    : $round =~ /^(\d+)-(\d+)$/ ? sprintf("-%03d-%03d", $1, $2) 
    : '';

  my $dname = '';
  my $dname_hyphen = '';
  if ($dp) {
    $dname = $dp->Name();
    $dname_hyphen = "$dname-";
    }
  unless ($config->Value('no_text_files')) {
    $fn = $config->MakeRootPath("$dname_hyphen$filename.doc");
    $fh = OpenFile('>', $fn) or Error "Can't create $fn: $!\n"; 
    $this->{'text_fh'} = $fh;
    }

  $fn = $config->MakeRootPath("$dname_hyphen$filename.html");
  $fh = OpenFile('>', $fn) or Error "Can't create $fn: $!\n";
  $this->{'html_fh'} = $fh;

  $fn = $config->MakeHTMLPath("$dname_hyphen$filename$round_000.html");
  $fh = OpenFile('>', $fn) or Error "Can't create $fn: $!\n";
  $this->{'subhtml_fh'} = $fh;

  my $title = $optionsp->{'title'};
  unless (defined $optionsp->{'title'}) {
    $title = ucfirst $filename;
    $title = "Round $round $title" if $round =~ /^\d+$/;
    if ($tournament->CountDivisions() > 1) { 
      $title = $dp ? "Division $dname $title": $title;
      };
    }
  my ($text, $html) = $this->ReportHeader($title, $filename, $optionsp);
  $this->Write($text, $html);

  my $filename_ = $filename; $filename_ =~ s/-/_/g;
  $this->Write('', <<"EOF");
<table class="$filename_" align=center cellspacing=0>
<tr class=top1>
EOF

  return $this;
  }

=item $log->MakeHTMLIndex();

Used internally to update the HTML index of HTML files.

=cut

sub MakeHTMLIndex ($) {
  my $this = shift;
  my $dir;
  my $tournament = $this->{'tournament'};
  my $config = $tournament->Config();
  my $dn = $config->MakeHTMLPath('');
  unless (opendir($dir, $dn)) {
    TSH::Utility::Error "Can't open $dn: $!\n";
    return;
    }
  my @rounds;
  my %divisions;
  for my $file (readdir $dir) {
    next unless $file =~ /^(\w+)-([-a-z]+)(?:-(\d+))?(?:-(\d+))?\.html$/i;
    next if $file =~ /scorecard\.html/;
    my $div = lc $1;
    my $type = $2;
    my $round = ($3 || 0);
    my $lastround = $4;
    next unless $tournament->GetDivisionByName($div);
    $divisions{$div}++;
    $type = "$type $round-$lastround" if $lastround;
    $rounds[$round]{$div}{$type} = $file;
    }
  closedir $dir;
  my (@divisions) = sort keys %divisions;
  
  my $fh;
  my $fn = $config->MakeHTMLPath('index.html');
  unless (open $fh, ">$fn") {
    Error "Can't create HTML index file: $!\n";
    return;
    }
  print $fh ($this->ReportHeader("Event Coverage Index", 'index', {}))[1];
  print $fh "<table class=index align=center>\n";
  if (@divisions > 1) {
    print $fh "<tr><th class=empty>&nbsp;</th>";
    for my $div (@divisions) {
      print $fh "<th class=division>Div. \U$div\E</th>\n";
      }
    print $fh "</tr>\n";
    }
  for my $round (0..$#rounds) {
    my $rp = $rounds[$round];
    next unless $rp;
    if ($round) {
      print $fh "<tr><td class=round>Round $round</td>\n";
      }
    else {
      print $fh "<tr><td class=round>&nbsp;</td>\n";
      }
    for my $div (@divisions) {
      print $fh "<td class=links>";
      my $rdp = $rp->{$div};
      if ($rdp) {
	for my $type (sort keys %$rdp) {
	  print $fh qq(<div class=link><a href="$rdp->{$type}">$type</a></div>\n);
	  }
        }
      print $fh "</td>";
      }
    print $fh "</tr>\n";
    }
  print $fh "</table>\n";
  print $fh ReportTrailer;
  close $fh;
  }

=item $d = new Log;

Create a new Log object.  

=cut

sub new ($$$$;$$) { return TSH::Utility::new(@_); }

=item ($text, $html) = $log->ReportHeader($title, $type, \%options);

See initialise() for options.

=cut

sub ReportHeader ($$$$) {
  my $this = shift;
  my $title = shift;
  my $type = shift;
  my $optionsp = shift;
  my $config = $this->{'tournament'}->Config();
  my $html = '';
  my $type_ = $type; $type =~ s/-/_/;
  $html .= <<"EOF";
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN"
	"http://www.w3.org/TR/REC-html40/loose.dtd">
<html><head><title>$title</title>
<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">
<link rel="stylesheet" href="tsh.css" type="text/css">
</head>
<body class="$type_">
EOF
  my $extra_top = $config->Value('html_top');
  $html .= $extra_top if $extra_top;
  my $htmltitle = $title; $htmltitle =~ s/(\d)-(\d)/$1&ndash;$2/g;
  return $optionsp->{'notitle'}
    ? ('', $html) : ("$title\n\n", "$html<h1>$htmltitle</h1>\n"); 
  }

=item ($text, $html) = ReportTrailer $title, $type, $bare

Render a standard report trailer.

=cut

sub ReportTrailer () {
  return ('', "<p class=notice>This report was generated using <cite>tsh</cite> version $main::gkVersion.  For more information about <cite>tsh</cite>, please ask John Chew.</p></body></html>");
  }

=item $log->Write($text,$html)

Write text to logs and console.

=cut

sub Write ($$$) {
  my $this = shift;
  my $text = shift;
  my $html = shift;
  my $fh;
  print $text unless $this->{'options'}{'noconsole'};
  $fh = $this->{'text_fh'}; print $fh $text if $fh;
  $fh = $this->{'html_fh'}; print $fh $html if $fh;
  $fh = $this->{'subhtml_fh'}; print $fh $html if $fh;
  }

=back

=cut

=head1 BUGS

It's arguable that Close() should be part of the object destructor.

=cut

1;
