#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Report::CSC;

use strict;
use warnings;
use Carp;

=pod

=head1 NAME

TSH::Report::CSC - utility for generating a contestant scorecard

=head1 SYNOPSIS

  $c = new TSH::Report::CSC($player);
  $s = $c->GetHTML(\%options);
  $s = $c->GetText();

=head1 ABSTRACT

This Perl module is used to create contestant scorecards.

=head1 DESCRIPTION

A CSC has (at least) the following member fields, none of which
ought to be accessed directly from outside the class.

  player  reference to TSH::Player object

The following member functions are currently defined.

=over 4

=cut

sub GetHTML ($$);
sub GetText ($$);
sub new ($$);

=item $html = $csc->GetHTML(\%options);

Return HTML rendering of the CSC as the rows of a table,
omitting initial C(<tr>).
No options are currently supported.

=cut

sub GetHTML ($$) {
  my $this = shift;
  my $optionsp = shift;
  my $p = $this->{'player'};
  my $html = '';
  my $dp = $p->Division();
  my $tables = $config::tables{$dp->Name()};

  # TODO: should optionally display activity and off-spread?
  $html .= "<td class=number colspan=3>" . $p->FullID() . "</td>\n";
  my $clean_name = $p->Name();
  $clean_name =~ s/,$//; # kludge to allow names ending in digits
  $html .= "<td class=name colspan="
    . ($config::entry eq 'spread' ? 3 : 5) 
    . "><span class=label>Name:</span> $clean_name</td>\n";
  $html .= "<td class=rating colspan=2><span class=label>Rating:</span> "
    . $p->Rating() . "</td>\n";
  $html .= "</tr>\n";

  $html .= "<tr class=top2>"
    . "<th class=round>Round</th>"
    . "<th class=board>" . ($tables ? $config::table_title : 'Board') . "</th>"
    . "<th class=onum>Opp. #</th>"
    . "<th class=orat>Rating</th>"
    . "<th class=onam>Opponent Name</th>"
    . "<th class=won>Won</th><th class=lost>Lost</th>";
  $html .= "<th class=for>Player<br>Score</th>"
    . "<th class=against>Opponent<br>Score</th>"
    if $config::entry ne 'spread';
  $html .= "<th class=spread>Spread</th>"
    . "</tr>\n";

  # print one row for each paired or played round
  my $maxround1 = $config::max_rounds || 
    ($p->CountOpponents() > $p->CountScores() 
    ? $p->CountOpponents() : $p->CountScores());
  my $cume = 0;
  my $games = 0;
  my $wins = 0;
  for my $r1 (1..$maxround1) {
    my $r0 = $r1 - 1;
    my $oid = $p->OpponentID($r0);
    unless (defined $oid) { # no data yet for round
      $html .= "<tr class=round1>"
	. "<td class=round>$r1</td>"
	. "<td class=board rowspan=2>&nbsp;</td>"
	. "<td class=onum rowspan=2>&nbsp;</td>"
	. "<td class=orat rowspan=2>&nbsp;</td>"
	. "<td class=onam rowspan=2>&nbsp;</td>"
	. "<td class=won rowspan=2>&nbsp;</td>"
	. "<td class=lost rowspan=2>&nbsp;</td>"
	. ($config::entry ne 'spread' &&
	    "<td class=for rowspan=2>&nbsp;</td>"
	  . "<td class=against>&nbsp;</td>"
	  )
	. "<td class=spread>&nbsp;</td>"
	. "</tr>\n"
        . "<tr class=round2>"
	. "<td class=p12>1st 2nd</td>"
        . ($config::entry ne 'spread' &&
	  "<td class=cumelabel><span class=label>cumulative</span></td>")
	. "<td class=cume>&nbsp;</td>"
	. "</tr>\n";
      next;
      }
    my $opp = $p->Opponent($r0);
    if ($oid && !$opp) {
      $dp->Tournament()->TellUser('ebadtpn', $p->ID(), $oid);
      return;
      }
    my $score = $p->Score($r0);
    $html .= "<tr class=round1" . ($oid ? '' : 'bye') . '>'
      . "<td class=round>$r1</td>";
    my $p12 = $config::track_firsts && $p->First($r0);
    my $bort = $p->Board($r0);
    if ($bort && $tables) {
      $bort = $tables->[$bort-1];
      }
    $bort ||= '-';
    $html .= "<td class=board rowspan=2>$bort</td>";
    if ($oid) {
      $html .= "<td class=onum rowspan=2>" . $opp->ID() . "</td>"
	. "<td class=orat rowspan=2>" . $opp->Rating() . "</td>"
	. "<td class=onam rowspan=2>" . $opp->Name() . "</td>";
      }
    else {
      $html .= "<td class=onum rowspan=2>-</td>"
	. "<td class=orat rowspan=2>-</td>"
	. "<td class=onam rowspan=2>bye</td>";
      }
    if (defined $score) {
      $games++;
      my $oscore = $p->OpponentScore($r0) || 0;
      my $spread = $score - $oscore;
      $cume += $spread;
      $wins += (1+($spread <=> 0))/2 if $oid || $score;
      my $losses = $games - $wins;
      $html .= "<td class=won rowspan=2>$wins</td>"
        . "<td class=lost rowspan=2>$losses</td>";
      $html .= "<td class=for rowspan=2>$score</td>"
        . "<td class=against>$oscore</td>"
        unless $config::entry eq 'spread';
      $html .= sprintf("<td class=spread>%+d</td>", $spread);
      }
    else {
      $html .= "<td class=won rowspan=2>&nbsp;</td>"
        . "<td class=lost rowspan=2>&nbsp;</td>"
        . "<td class=for rowspan=2>&nbsp;</td>"
	. ($config::entry ne 'spread' &&
	    "<td class=against>&nbsp;</td>"
	  . "<td class=spread>&nbsp;</td>");
      }
    $html .= "</tr>\n" . "<tr class=round2" . ($oid ? '' : 'bye') . '>';
    if ($config::track_firsts) {
      $html .= "<td class=p12>"
	. (!defined $p12 ? '' : ('-','1st&nbsp;&nbsp;&nbsp;&nbsp;',
	  '&nbsp;&nbsp;&nbsp;&nbsp;2nd','1st 2nd','1st 2nd')[$p12])
	. '</td>';
      }
    else { 
      $html .= "<td class=p12>1st 2nd</td>";
      }
    if (defined $score) {
      $html .= ($config::entry ne 'spread' && "<td class=cumelabel><span class=label>cumulative</span><span class=strut>&nbsp;</span></td>")
        . sprintf("<td class=cume>%+d</td>", $cume);
      }
    else {
      $html .= ($config::entry ne 'spread' && "<td class=cumelabel><span class=label>cumulative</span><span class=strut>&nbsp;</span></td>")
	. "<td class=cume>&nbsp;</td>";
      }
    $html .= "</tr>\n";
    } # for $r0

  $html .= "<tr class=bottom><td colspan="
    . ($config::entry eq 'spread' ? 9 : 10) . ">&nbsp;</td></tr>\n";
  return $html;
  }

=item $html = $csc->GetText(\%options);

Return HTML rendering of the CSC as text.
No options are currently supported.

=cut

sub GetText ($$) {
  my $this = shift;
  my $optionsp = shift;
  my $p = $this->{'player'};
  my $text = '';
  my $dp = $p->Division();
  my $tables = $config::tables{$dp->Name()};

  $text .= 'Player Scorecard: ';
  $text .= $p->TaggedName();
  $text .= " (" . $p->Rating() . ")";
  if ($config::track_firsts) {
    my $firsts = $p->Firsts()|| 0;
    my $seconds = $p->Seconds() || 0;
    $text .= " (S${firsts}:R$seconds)";
    }
  if (!$p->Active()) {
    my $spread = $p->OffSpread();
    $text .= " INACTIVE";
    $text .= "[$spread]" unless $spread == -50;
    }
  $text .= "\n";
  my $clean_name = $p->Name();
  $clean_name =~ s/,$//; # kludge to allow names ending in digits
  if ($p->{'etc'}{'time'}) {
    my $age = int((time - $p->{'etc'}{'time'}[0])/60);
    if ($age < 200) {
      my $s = $age == 1 ? '' : 's';
      $text .= "- Last score was entered $age minute$s ago.\n";
      }
    }
  $text .= 'Rnd ';
  $text .= '1/2 ' if $config::track_firsts;
  $text .= sprintf(
    "$config::table_format Opp Rtng $config::name_format Won-Lost ",
    ($tables ? 'Tbl' : 'Brd'),
    'Opponent Name');
  $text .= 'For Agn ' if $config::entry ne 'spread';
  $text .= "Sprd Cumul\n";

  # print one line for each paired or played round
  my $maxround1 = $config::max_rounds || 
    ($p->CountOpponents() > $p->CountScores() 
    ? $p->CountOpponents() : $p->CountScores());
  my $cume = 0;
  my $games = 0;
  my $wins = 0;
  for my $r1 (1..$maxround1) {
    $text .= sprintf("%3d ", $r1);
    my $r0 = $r1 - 1;
    my $oid = $p->OpponentID($r0);
    unless (defined $oid) { $text .= "\n"; next; }
    my $opp = $p->Opponent($r0);
    if ($oid && !$opp) {
      $dp->Tournament()->TellUser('ebadtpn', $p->ID(), $oid);
      return;
      }
    my $score = $p->Score($r0);
    my $p12 = $p->First($r0);
    if ($config::track_firsts) {
      $text .= !defined $p12 ? '    ' 
	  : ' ' . ('- ',"$p12 ","$p12 ",'? ', '??')[$p12] . ' ';
      }
    my $bort = $p->Board($r0);
    if ($bort && $tables) {
      $bort = $tables->[$bort-1];
      }
    $bort ||= '';
    $text .= sprintf($config::table_format, $bort) . ' ';
    if ($oid) {
      $text .= sprintf("%3d %4d $config'name_format",
	$opp->ID(), $opp->Rating(), $opp->Name());
      }
    else {
      $text .= sprintf("%3s %4s $config'name_format", '', 'bye', '');
      }
    if (defined $score) {
      $games++;
      my $oscore = $p->OpponentScore($r0) || 0;
      my $spread = $score - $oscore;
      $cume += $spread;
      $wins += (1+($spread <=> 0))/2 if $oid || $score;
      my $losses = $games - $wins;
      $text .= sprintf(" %4.1f-%4.1f", $wins, $losses);
      $text .= sprintf(" %3d %3d", $score, $oscore)
        unless $config::entry eq 'spread';
      $text .= sprintf(" %+4d %+5d", $spread, $cume);
      }
    $text .= "\n";
    } # for $r0
  return $text;
  }

sub initialise ($$) {
  my $this = shift;
  my $p = shift;
  $this->{'player'} = $p;
  }

sub new ($$) { return TSH::Utility::new(@_); }

=back

=cut

=head1 BUGS

None known.

=cut

1;
