#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ChewPair;

use strict;
use warnings;

use TSH::Utility qw(Debug DebugOn DebugOff DebugDumpPairings);
use TSH::PairingCommand;
use TSH::Player;

our (@ISA) = qw(TSH::PairingCommand);

=pod

=head1 NAME

TSH::Command::ChewPair - implement the C<tsh> ChewPair command

=head1 SYNOPSIS

  my $command = new TSH::Command::ChewPair;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ChewPair is a subclass of TSH::Command.

=cut

=head1 SUBROUTINES

=over 4

=cut

sub CalculateBPFs ($);
sub CalculateBestPossibleFinish ($$);
sub CountContenders ($$$$$);
sub CountGibsons ($$);
sub GetPrizeBand($$$);
sub GibsonEquivalent($$);
sub initialise ($$$$);
sub MaxSpread ($);
sub new ($);
sub PairAllGibsons ($);
sub PairContenders ($$);
sub PairEvenGibsons ($$);
sub PairLeaders ($$$);
sub PairNonLeaders ($$);
sub PairOneGibson ($$$$);
sub Run ($$@);
sub RunInit ($$$);
sub SplitContenders ($$$);
sub Swiss ($$$$$);

=item CalculateBPFs($psp)

Calculate best possible finishes for everyone in @$psp.

=cut

sub CalculateBPFs($) {
  my $psp = shift;

  my $toprank = $psp->[0]->RoundRank($config::max_rounds);
  for my $i (0..$#$psp) {
    my $r = CalculateBestPossibleFinish $psp, $i;
    $psp->[$i]->MaxRank($r+$toprank-1);
    }
  }

=item CalculateBestPossibleFinish($psp, $index);

Calculate what the highest possible finishing rank is for the
${index}th player in $psp.  
The rank is relative to the players in @$psp.

=cut

sub CalculateBestPossibleFinish ($$) {
  my $psp = shift;
  my $index = shift;

  return undef if $index > $#$psp;
  return 1 if @$psp == 1;
  my $me = $psp->[$index];
  my $config = $me->Division()->Tournament()->Config();
  my $bye_spread = $config->Value('bye_spread');
  $bye_spread = 50 unless defined $bye_spread; # should not happen

# DebugOn('CBPF');
  # set scratch variables
  for my $p (@$psp) {
    # current wins
    $p->{'xw'} = $p->Wins();
    # current spread
    $p->{'xs'} = $p->Spread();
    # maximum spread per round, number of games left to play
    $p->{'xms'} = MaxSpread ($p->{'xleft'} = $p->UnscoredGames());
    }
  my $nrounds = 0;
  for my $p (@$psp) { $nrounds = $p->{'xleft'} if $nrounds < $p->{'xleft'} }
# Debug 'CBPF', "np=%d nrounds=%d.", $#$psp+1, $nrounds;
  # if no games left to play, rank won't change
  return 1+$index if $nrounds == 0;

  my $my_final_wins = $me->{'xw'} + $me->{'xleft'};
  my $my_final_spread = $me->{'xs'} + $me->{'xleft'} * $me->{'xms'};
  my @ps;
  # assign wins as favourably to me as possible
  for my $r (1..$nrounds) {
#   Debug 'CBPF', 'Round %d', $config::max_rounds-$nrounds+$r;
    # everyone ranked up to the cap must play each other
    my $rounds_left = $nrounds - $r + 1; # a rough approximation
    my $cap = &$config::flight_cap($rounds_left) - 1; # 0-based
    $cap = $#$psp if $cap > $#$psp;
    my $r0 = $config::max_rounds - $rounds_left; # +1 -1 = 0
    @ps = sort { $b->{'xw'}<=>$a->{'xw'} || $b->{'xs'}<=>$a->{'xs'}||($a eq $me ? -1 : $b eq $me ? 1 : 0)} @$psp;
    # divide players into five groups: preferred winners above and below
    # the cap line, losers (ditto) and me
    my $myrank = undef;
    my @hi_winners;
    my @lo_winners;
    my @hi_losers;
    my @lo_losers;
    for my $i (0..$#$psp) { 
      my $p = $ps[$i];
      if ($me eq $p) { $myrank = $i; next; } 
      next if defined $p->Score($r0); # already has a score this round
      my $w = $p->{'xw'}; 
      # If ranked below me
      if ($w < $my_final_wins - 1 
	|| ($w == $my_final_wins - 1 && $p->{'xs'} < $my_final_spread)) {
	# I want them to win
	if ($i <= $cap) { unshift(@hi_winners, $p); }
	else { unshift(@lo_winners, $p); }
        }
      # else ranked above me
      else { # build these lists in the other order.
	# I want them to lose
	if ($i <= $cap) { push(@hi_losers, $p); }
	else { push(@lo_losers, $p); }
        }
      }
#   Debug 'CBPF', 'MR: %d. HW: %s. LW: %s. HL: %s. LL: %s.', $myrank, (join(',',map {$_->{'name'}} @hi_winners)), (join(',',map {$_->{'name'}} @lo_winners)), (join(',',map {$_->{'name'}} @hi_losers)), (join(',',map {$_->{'name'}} @lo_losers));
    # if I can play, I always win with the largest possible spread
    if (!defined $me->Score($r0)) { 
      # find me an opponent
      # preference ranking for opponent category, according to where we stand
      my $grouppp = $myrank <= $cap 
	?  [ \@hi_losers, \@hi_winners, \@lo_losers, \@lo_winners]
	:  [ \@lo_losers, \@lo_winners, \@hi_losers, \@hi_winners];
      my $opp;
      for my $groupp (@$grouppp) 
        { if (@$groupp) { $opp = shift @$groupp; last; } }
      if ($opp) {
	$me->{'xw'}++; $me->{'xs'} += $me->{'xms'}; 
	# following is an approximation, should check when 
	# $me->{'xms'} != $opp->{'xms'} (TODO)
	# See several similar cases below. 
	$opp->{'xs'} += $opp->{'xms'}; 
        }
      else {
	$me->{'xw'}++; 
	$me->{'xs'} += $bye_spread;
	$my_final_spread += $bye_spread - $me->{'xms'};
        }
      }
    
    for my $group_data_p ((
      [\@hi_winners,[\@hi_losers,\@hi_winners,\@lo_winners,\@lo_losers]],
      [\@hi_losers,[\@hi_winners,\@hi_losers,\@lo_losers,\@lo_winners]],
      [\@lo_winners,[\@lo_losers,\@lo_winners,\@hi_losers,\@hi_winners]],
      [\@lo_losers,[\@lo_winners,\@lo_losers,\@hi_winners,\@hi_losers]],
      )) {
      my $tbps = $group_data_p->[0];
      my $p = shift @$tbps;
      next unless defined $p;
      my $grouppp = $group_data_p->[1];
      my $opp;
      for my $groupp (@$grouppp) 
        { if (@$groupp) { $opp = shift @$groupp; last; } }
      if (!$opp) { $p->{'xw'}++; $p->{'xs'} += $bye_spread; next; }
      # make sure $p is higher-ranked than $opp
      if (($p->{'xw'}<=>$opp->{'xw'}||$p->{'xs'}<=>$opp->{'xs'})<0)
        { ($p, $opp) = ($opp, $p); }
      # $p and $opp have more wins than me
      if ($opp->{'xw'} > $my_final_wins) 
        # who cares: give opp a big win
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
      # $p has more wins than me
      elsif ($p->{'xw'} > $my_final_wins) 
	# write $p off: give him a big win
        { $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
      # $p and $opp both have my wins
      elsif ($opp->{'xw'}== $my_final_wins) 
	# write $p off: give him a big win
	{ $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
      # $p has <= mine; $opp has mine-1
      elsif ($opp->{'xw'} == $my_final_wins-1) {
	# if $opp would overtake me by winning
	if ($opp->{'xs'} >= $my_final_spread) {
	  # give a big win to whichever one has more spread already
	  if ($p->{'xs'} > $opp->{'xs'}) 
        { $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
	  else 
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
	  }
	# give $opp as big a win as possible without overtaking
	else {
	  my $spread = $my_final_spread - $opp->{'xs'};
	  $opp->{'xw'}++; $opp->{'xs'} += $spread; $p->{'xs'} -= $spread;
	  }
	}
      # $p has my wins, $opp <= mine-2
      elsif ($p->{'xw'} == $my_final_wins) {
	my $delta = $p->{'xs'} - $my_final_spread;
	# $p is ahead on spread, but maybe we can bring him down
	if ($delta > 0 && $delta <= $p->{'xms'} * $rounds_left) 
	  # give opp a big win
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
	else 
	# write $p off: give him a big win
	{ $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
        }
      # $p has <= mine-1, $opp <= mine-2
      else 
	# give opp a big win
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
      }
    }
  
  my $rank = undef;
  @ps = sort { $b->{'xw'}<=>$a->{'xw'} || $b->{'xs'}<=>$a->{'xs'}||($a eq $me ? -1 : $b eq $me ? 1 : 0)} @$psp;
  for my $i (0..$#ps) {
    my $p = $ps[$i];
    next if $p ne $me;
    $rank = $i + 1;
#   my $behind = $i > 0 ? sprintf("%s %g %+d", $ps[$i-1]->TaggedName(), $ps[$i-1]->{'xw'}, $ps[$i-1]->{'xs'}) : 'nobody'; my $after = $i < $#ps ? sprintf("%s %g %+d", $ps[$i-1]->TaggedName(), $ps[$i+1]->{'xw'}, $ps[$i+1]->{'xs'}) : 'nobody'; Debug 'CBPF', '%s can finish ranked %d, %g %+d, behind %s, ahead of %s.', $me->TaggedName(), $rank, $my_final_wins, $my_final_spread, $behind, $after;
    last;
    }
  Debug 'CP', "%3d=>%3d%5.1f%+5d %s #%d", $index+1, $rank, $my_final_wins, $my_final_spread, $me->Name(), $me->ID();
    return $rank;
  }

=item $n = CountContenders($maxrank, $psp, $sr0, $rounds_left)

Count the number of players in contention for a prize band
based on results up to $sr0 with $rounds_left remaining.
Round up to an even number.
Cap at four times the number of rounds remaining.

=cut

sub CountContenders ($$$$$) {
  my $minrank = shift;
  my $maxrank = shift;
  my $psp = shift;
  my $sr0 = shift;
  my $rounds_left = shift;

  Debug 'CP', 'Contending rnks %d-%d?  Plyrs left: %d.  Rds left: %d as of Rd %d.', $minrank, $maxrank, scalar(@$psp), $rounds_left, $sr0+1;
  Debug 'CP', 'Now=>Fnl Wins Sprd Player (theoretical best finishes)';
  CalculateBPFs $psp;
  
  my $max_flight_size = 0;
  for my $i (0..$#$psp) {
    my $r = $psp->[$i]->MaxRank();
#   Debug 'CP', '%s maxrank is %d', $psp->[$i]->TaggedName(), $r;
    if ($r <= $maxrank) {
      $max_flight_size = $i;
#     Debug 'CP', 'CC: %d <= %d, mfs=%d', $r, $maxrank, $i;
      }
    }
  TSH::Utility::Error "Assertion failed: only one contender" 
    if $max_flight_size == 0;
  $max_flight_size++; # now one-based
  Debug 'CP', 'There are %d contenders.', $max_flight_size;
  $max_flight_size++ if $max_flight_size % 2; # round up to even
  # if contenders are more than four times rounds left, cap to that amount
  # see also CalculateBestPossibleFinish
  my $cap = &$config::flight_cap($rounds_left);
  if ($minrank == 1 && $max_flight_size > $cap && $sr0 >= 0) {
    $max_flight_size = $cap;
    Debug 'CP', 'Capping to %d contenders.', $max_flight_size;
    }
  return $max_flight_size;
  }

=item $n = CountGibsons($sr0, $psp)

Count the number of Gibsons in @$psp based on round $sr0 standings.
You must call Division::ComputeRanks($sr0) before calling this sub.
Gibsonization is done based on wins and spread, with the assumption
that a player can catch up 500 points of spread in one game, 700 in
two games, and 300 points per round in three or more games.

=cut

sub CountGibsons ($$) {
  my $sr0 = shift;
  my $psp = shift;
  my $sr = $sr0 + 1;
  my $rounds_left = $config::max_rounds - $sr;
  my $spread_left = 
  my $round_spread = 2 * MaxSpread($rounds_left);
  my $spread_allowed = $rounds_left * $round_spread;
  unless (@$psp) {
    Debug 'CP', 'CountGibsons called with empty list';
    return 0;
    }

  my $dname = $psp->[0]->Division()->Name();
  my $gibson_equivalent = $config::gibson_equivalent{$dname};
  my $ngibsons = 1;
  if ($gibson_equivalent) {
    # $gibson_equivalent tells us which ranks are equivalent,
    # usually either #1 and #2 or none
    my $rank = $psp->[0]->RoundRank($config::max_rounds);
    my $here = $gibson_equivalent->[$rank] || $rank;
    if ($here) {
      while (1) {
	my $next = $gibson_equivalent->[++$rank];
	last unless $next && $next == $here;
	$ngibsons++;
        }
      }
    }
  Debug 'CP', 'Gibson-equivalent ranks: %d starting at %d', $ngibsons, $psp->[0]->RoundRank($config::max_rounds);
  while ($ngibsons > 0) {
    my $win_diff = $psp->[$ngibsons-1]->RoundWins($sr0)
      - $psp->[$ngibsons]->RoundWins($sr0);
    if ($win_diff > $rounds_left) { last; }
    elsif ($win_diff < $rounds_left) { next; }
    my $spread_diff = $psp->[$ngibsons-1]->RoundSpread($sr0)
      - $psp->[$ngibsons]->RoundSpread($sr0);
    if ($spread_diff > $spread_allowed) { last; }
    }
  continue { $ngibsons--; }
  Debug 'CP', 'Gibsons found: %d', $ngibsons;
  return $ngibsons;
  }

=item ($minrank,$maxrank) = $parserp->GetPrizeBand($prize_bands, $psp, $sr0)

Find the prize band that contains the given rank.

=cut

sub GetPrizeBand ($$$) {
  my $prize_bands = shift;
  my $psp = shift;
  my $sr0 = shift;
  die "Assertion failed" unless @$psp;
  my $rank = $psp->[0]->RoundRank($config::max_rounds);
  my (@splits) = (0, @$prize_bands, $psp->[-1]->RoundRank($config::max_rounds));
# Debug 'CP', "GetPB: $rank, @splits";
  for my $i (1..$#splits) {
    my $minrank = $splits[$i-1]+1;
    my $maxrank = $splits[$i];
    if ($rank >= $minrank && $rank <= $maxrank) {
      return ($minrank, $maxrank);
      }
    }
  die "Assertion failed with rank=$rank, splits=@splits";
  }

=item $parserp->GibsonEquivalent($dname, $rank)

Return the highest rank equivalent for Gibson purposes to $rank.

=cut

sub GibsonEquivalent($$) {
  my $dname = shift;
  my $rank = shift;

  return ($config::gibson_equivalent{$dname}[$rank] || $rank);
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to compute Chew pairings for a division.
EOF
  $this->{'names'} = [qw(cp chewpair)];
  $this->{'argtypes'} = [qw(BasedOnRound Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  DebugOn('CP');
  return $this;
  }

=item $s = MaxSpread ($nr);

Return the maximum likely spread that a player can earn per round
over $nr rounds.

NSA rules are 500 points of spread in 1 round, 700 in 2, 900 in 3.
We generalise to 300*n for n>=3.

=cut

sub MaxSpread ($) {
  my $nr = shift;
  return $nr == 1 ? 250 : $nr == 2 ? 175 : 150;
  }

sub new ($) { 
  return TSH::Utility::new(@_); 
  }

=item $parser->PairAllGibsons($psp);

Pair all Gibsons at the top of $this->{'cp_tobepaired'}.
Pair all Gibsons at the top of @$psp.

=cut

sub PairAllGibsons ($) {
  my $this = shift;
  my $tobepaired = shift;
  my $sr0 = $this->{'cp_sr0'};
  my $rounds_left = $this->{'cp_rounds_left'};
  while (my $ngibsons = CountGibsons($sr0, $tobepaired)) {
#     Debug 'CP', '%d gibson%s.', $ngibsons, ($ngibsons == 1 ? '' : 's');
    # if there is an even number of gibsons
    if ($ngibsons % 2 == 0) {
      # then pair them KOTH but minimize rematches
      PairEvenGibsons($tobepaired, $ngibsons); 
      }
    # else there is an odd number of gibsons
    else { 
      # pair all but the lowest ranked gibson KOTH minimizing rematches
      PairEvenGibsons($tobepaired, $ngibsons-1);
      # pair the last one with a low-ranked victim
      PairOneGibson($tobepaired, $this->{'cp_current_prize_bands'}[-1], 
	$sr0, $rounds_left);
      }
    }
  }

=item $parserp->PairContenders($dp);

Pair all contenders at the top of $this->{'cp_tobepaired'};

=cut

sub PairContenders ($$) {
  my $this = shift;
  my $dp = shift;
  my $prize_bands = $this->{'cp_current_prize_bands'};
  my $round0 = $this->{'cp_round0'};
  my $sr0 = $this->{'cp_sr0'};
  my $rounds_left = $this->{'cp_rounds_left'};
  my $tobepaired = $this->{'cp_tobepaired'};
  # find the highest remaining prize band
  my ($minrank, $maxrank) = GetPrizeBand($prize_bands, $tobepaired, $sr0);
# Debug 'CP', 'prize_band: [%d,%d]', $minrank, $maxrank;
  # count prize band contenders
  my $ncontenders 
    = CountContenders($minrank,$maxrank,$tobepaired,$sr0,$rounds_left); 
  die "assertion failed" if $ncontenders > @$tobepaired;
  if ($ncontenders == @$tobepaired - 2) {
    my $repeats = $tobepaired->[-2]->Repeats($tobepaired->[-1]->ID());
    if ($repeats > 0) {
      $ncontenders += 2;
      Debug 'CP', 'Adding back last two players, their repeats=%d', $repeats;
      }
    }
  # compute the smallest number of repeats 'minrep' needed to pair band
  my $minrep = 0;
  my (@flight) = @$tobepaired[0..$ncontenders-1];
  until (TSH::Player::CanBePaired $minrep, \@flight) { $minrep++; }
  Debug 'CP', 'Flight can be paired with repeats=%d.', $minrep;
  # if everyone is still a contender, Swiss-pair them
  if ($ncontenders == @$tobepaired) { Debug 'CP', 'Everyone is a contender, will pair Swiss.';
    Swiss $dp, \@flight, $minrep, $sr0, $round0;
    }
  # else there are some contenders and some noncontenders
  else {
    # find the highest split into two groups pairable in minrep
    my $split = $this->SplitContenders(\@flight, $minrep);
#     DebugOn('GRT');
    # pair the flight leaders 
    PairLeaders [@flight[0..$split-1]], $minrep, $sr0;
    # pair the rest Swiss
# OLD PairNonLeaders [@flight[$split..$#flight]], $minrep if $split <= $#flight;
    if ($split <= $#flight) {
      Debug 'CP', 'Pairing %d non-leaders.', $#flight-$split+1;
      Swiss $dp, [@flight[$split..$#flight]], $minrep, $sr0, $round0;
      }
#     DebugOff('GRT');
    }
  splice(@$tobepaired, 0, $ncontenders);
  }

=item PairEvenGibsons($psp, $ngibsons)

Pair an even number of gibsons with each other,
with decreasing priority given to

  - minimizing rematches
  - avoiding consecutive rematches (where possible)
  - pairing each player with the one ranked closest possible

=cut

sub PairEvenGibsons ($$) {
  my $psp = shift;
  my $ngibsons = shift;
  return unless $ngibsons && @$psp;
  Debug 'CP', 'Pairing %d gibsons.', scalar($ngibsons);
  my $r0 = $psp->[0]->CountOpponents();
  if (
    TSH::Player::PairGRT([@$psp[0..$ngibsons-1]],
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs($_[1]-$_[2]);
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d svr=%d dist=%d', $pid, $oid, $repeats, $sameopp, $pairsvr, $distance;
	pack('NCCNN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance, # prefer closer-ranked opponents
	  $_[2], # to recover player ID
	  )
        },
      # allowed opponent filter
      sub {
	# allow any
	1,
        },
      [],
      $r0,
      )
    ) # if
    {
    DebugDumpPairings 'CP', $psp->[0]->CountOpponents()-1, 
      [splice(@$psp, 0, $ngibsons)];
    }
  else
    {
    TSH::Utility::Error "Assertion failed: can't resolve even gibson pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairLeaders($$$)

Pair an even number of flight leaders with decreasing priority given to
with decreasing priority given to

  - matching leaders with opponents who could catch up to them
  - not exceeding minrep (ever)
  - minimizing repeats
  - avoiding consecutive rematches (where possible)
  - pairing highest with lowest

=cut

sub PairLeaders ($$$) {
  my $psp = shift;
  my $minrep = shift;
  my $sr0 = shift;
  Debug 'CP', 'Pairing %d leaders.', scalar(@$psp);
  return unless @$psp;
  my $r0 = $psp->[0]->CountOpponents();
  unless (
    TSH::Player::PairGRT($psp,
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $ketchup = ($o->MaxRank() > $p->RoundRank($config::max_rounds));
#	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

#	Debug 'GRT', 'pref %d-%d rep=%d prev=%d can=%d (svr=%d)', $pid, $oid, $repeats, $sameopp, $ketchup, $pairsvr;
 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d can=%d', $pid, $oid, $repeats, $sameopp, $ketchup;
	pack('NCCNN',
	  $ketchup, # prefer those who can catch up
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
#	  $pairsvr, # pair those due to start vs those due to reply
	  999999-$_[2], # prefer lower-ranked opponents
	  $_[2], # for ID recovery
	  )
        },
      # allowed opponent filter
      sub {
 	Debug 'GRT', 'filter: %s vs %s %d ?<= %d', $_[0][$_[1]]->TaggedName(), $_[0][$_[2]]->TaggedName(), $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()), $minrep;
	# do not exceed minrep
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $minrep
        },
      [],
      $r0,
      )
    ) # unless
    {
    TSH::Utility::Error "Assertion failed: can't resolve leader pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairNonLeaders($$)

This routine is not currently used, as nonleaders are now paired Swiss.

Pair an even number of flight non-leaders with decreasing priority given to
with decreasing priority given to

  - not exceeding minrep (ever)
  - minimizing rematches
  - avoiding consecutive rematches
  - matching players due to go first with those due to go second
  - keeping rank differences close to half the group size

=cut

sub PairNonLeaders ($$) {
  my $psp = shift;
  my $minrep = shift;
  Debug 'CP', 'Pairing %d non-leaders.', scalar(@$psp);
  return unless @$psp;
  my $r0 = $psp->[0]->CountOpponents();
  unless (
    TSH::Player::PairGRT($psp,
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs(@{$_[0]}-abs(2*($_[1]-$_[2])));
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d dist=%d svr=%d', $pid, $oid, $repeats, $sameopp, $distance, $pairsvr;
	pack('NCCNN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance,# prefer opponents close to half the group size away in rank
	  # provide index for GRT to extract, effectively prefer higher opps
	  $_[2],
	  )
        },
      # allowed opponent filter
      sub {
#	Debug 'GRT', 'filter %d %d', $_[1], $_[2];
  	Debug 'GRT', 'filter: %s vs %s %d ?<= %d', $_[0][$_[1]]->TaggedName(), $_[0][$_[2]]->TaggedName(), $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()), $minrep;
	# do not exceed minrep
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $minrep
        },
      [],
      $r0,
      )
    ) # unless
    {
    TSH::Utility::Error "Assertion failed: can't resolve non-leader pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairOneGibson($psp, $last_prize_rank, $sr0, $rounds_left)

Pair one gibson.  If the bottom prize band (of players who have 
nothing at stake but ratings and pride) is occupied, the victim is
the highest ranked among those who have played the gibson least often.
If not, then it's the lowest ranked overall (including higher ranked
players who may still be concerned with reaching prize money or
qualification status) among those who have played the gibson least often.

=cut

sub PairOneGibson ($$$$) {
  my $psp = shift;
  my $last_prize_rank = shift;
  my $sr0 = shift;
  my $rounds_left = shift;
  Debug 'CP', 'Pairing one gibson from %d players, lpr=%d.', scalar(@$psp), $last_prize_rank;
  my $gibson = $psp->[0];
  my $gid = $gibson->ID();
  
  if (@$psp % 2) {
    Debug 'CP', 'Gibson gets the bye.';
    $gibson->Division()->Pair($gid, 0, $gibson->CountOpponents());
    splice(@$psp, 0, 1);
    return;
    }

  CalculateBPFs $psp;

  my $victim = undef;
  my $minrep = undef;
  my $hopeless_victim = undef;
  for (my $i = $#$psp; $i > 0; $i--) {
    my $poss_victim = $psp->[$i];
    my $rep = $poss_victim->Repeats($gid);
    my $is_hopeless = $poss_victim->MaxRank() > $last_prize_rank;
    if (!defined $minrep) {
      Debug 'CP', '... victim (rep=%d) = %s', $rep, $poss_victim->TaggedName();
      $minrep = $rep;
      $victim = $i;
      $hopeless_victim = $victim if $is_hopeless;
      }
    elsif ($minrep > $rep) {
      Debug 'CP', '... better victim (rep %d<%d) = %s', $minrep, $rep, $poss_victim->TaggedName();
      $minrep = $rep;
      $victim = $i;
      $hopeless_victim = $victim if $is_hopeless;
      }
    elsif ($is_hopeless && $minrep == $rep) {
      Debug 'CP', '... better victim (rep %d=%d) = %s', $minrep, $rep, $poss_victim->TaggedName();
      $hopeless_victim = $i;
      }
    }
   
  $victim = $hopeless_victim if defined $hopeless_victim;
  $gibson->Division()->Pair($gid, $psp->[$victim]->ID(),
    $gibson->CountOpponents());
  splice(@$psp, $victim, 1);
  splice(@$psp, 0, 1);
  return;
  }

=item $command->Run($tournament, @parsed_args);

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();
  my ($sr, $dp) = @_;
  my $sr0 = $this->{'cp_sr0'} = $sr - 1;
  my $setupp = $this->SetupForPairings(
    'division' => $dp,
    'required' => { 'max_rounds' => 1 },
    'nobye' => 1,
    'source0' => $sr0,
    'wanted_div' => { 'prize_bands' => [1] },
    ) or return 0;
  
  my $start = time;

  return unless $this->RunInit($tournament, $setupp);
  my $round0 = $this->{'cp_round0'};
  my $tobepaired = $this->{'cp_tobepaired'};
  Debug 'CP', 'Beginning Chew Pairings for %d players in round %d based on round %d.', $#$tobepaired+1, $round0+1, $sr0+1;
  # while we still have players left to pair
  while (@$tobepaired) { 
    Debug 'CP', 'Number of players still unpaired: %d', scalar(@$tobepaired);
    # perform any necessary gibsonization
    $this->PairAllGibsons($this->{'cp_tobepaired'});
    # assign a bye if necessary
    if (@$tobepaired % 2) { Debug 'CP', 'need a bye';
      $dp->ChooseBye($sr0, $round0, $tobepaired);
      }
    last unless @$tobepaired;
    # pair anyone in contention for the top prize still available
    $this->PairContenders($dp);
    }
  $dp->Dirty(1);
  $dp->Update();
  $tournament->TellUser('idone');
  Debug('CP', '%d second(s) runtime', time - $start);
  0;
  }

=item $ok = $cmd->RunInit($tournament, $dp);

Perform initialisation necessary prior to running the command.

=cut

sub RunInit ($$$) {
  my $this = shift;
  my $tournament = shift;
  my $setupp = shift;
  my $config = $tournament->Config();
  my $dp = $setupp->{'division'};
  my $sr0 = $this->{'cp_sr0'};
# $this->{'cp_rounds_left'} = $config::max_rounds - ($sr0+1); # see *20060430
  $this->{'cp_current_prize_bands'} = $config::prize_bands{$dp->Name()};
# $dp->ComputeRanks($this->{'cp_sr0'});
  $dp->ComputeRanks($config::max_rounds);
  my $round0 = $this->{'cp_round0'} =  $setupp->{'target0'};
  $this->{'cp_rounds_left'} = $config::max_rounds - $round0; # *20060430
  my $tobepaired = $dp->GetUnpairedRound($round0);
  unless (@$tobepaired) { 
    Debug 'CP', 
      "assertion failed at " . __FILE__ . ' line ' . __LINE__ . ": r0=$round0";
    $tournament->TellUser('ealpaird'); 
    return 0; 
    }
  @$tobepaired = TSH::Player::SortByStanding($this->{'cp_sr0'}, @$tobepaired);
  $this->{'cp_tobepaired'} = $tobepaired;

  return 1;
  }

=item $nleaders = $parser->SplitContenders(\@contenders, $repeats);

Decide where to split those players still in contention into two
pairings groups.  The cut goes as high as possible (that is,
keeping together the fewest leaders) subject to not exceeding the
number of repeats required to pair all the contenders as one group.
The return value is the number of leaders who will be split off
from the nonleading contenders.  If the contenders cannot be split
into two groups without increasing repeats, then the number of leaders
is equal to the number of contenders.

=cut

sub SplitContenders ($$$) {
  my $this = shift;
  my $contendersp = shift;
  my $repeats = shift;
  my $split = 0;
  while (($split += 2) < @$contendersp) {
    if (TSH::Player::CanBePaired $repeats, [@$contendersp[0..$split-1]]) {
      Debug 'CP', 'Top %d can be paired with rep=%d.', $split, $repeats;
      }
    else {
      Debug 'CP', 'Top %d cannot be paired with rep=%d.', $split, $repeats;
      next;
      }
    if (TSH::Player::CanBePaired $repeats, [@$contendersp[$split..$#$contendersp]]) {
      Debug 'CP', 'Rest can also be paired with rep=%d.', $repeats;
      last;
      }
    else {
      Debug 'CP', 'Rest cannot be paired with rep=%d.', $repeats;
      next;
      }
    }
  Debug 'CP', 'split is %d.', $split;
  return $split;
  }

=item @pairlist = Swiss($dp, \@ps, $repeats, $sr0, $round0);

Compute and save Swiss pairings for some set of players.

=cut

sub Swiss ($$$$$) {
  my $dp = shift;
  my $psp = shift;
  my $repeats = shift;
  my $sr0 = shift;
  my $round0 = shift;

# Debug 'CP', 'Swiss(%s,%d,%d,%d,%d)', $dp->Name(), $#$psp+1, $repeats,$sr0,$round0;
  my (@pair_list) = TSH::Division::PairSomeSwiss($psp, $repeats, $sr0);
  die "assertion failed" unless @pair_list;
# die "assertion failed: " . join(',', @pair_list) if @pair_list % 2;
  while (@pair_list) {
    my $p1 = shift @pair_list;
    my $p2 = shift @pair_list;
# die "assertion failed for $p1" unless UNIVERSAL::isa($p1, 'TSH::Player');
# die "assertion failed for $p2" unless UNIVERSAL::isa($p2, 'TSH::Player');
    $dp->Pair($p1->ID(), $p2->ID(), $round0);
    }
  }
=back

=cut

=head1 BUGS

Gibsonization should consider results from partially scored rounds.

The current algorithm assumes (e.g.) that if the top eight players
are paired with each other to contend for the top two places, then
the ninth-ranked player cannot finish higher than ninth place, and
is paired with other such players.

Gibsonization algorithm doesn't work well (or at all?) for more than
one final gibsonization position.

=cut

1;
