#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::PairingCommand;

use strict;
use warnings;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::PairingCommand - abstraction of a C<tsh> pairing command

=head1 SYNOPSIS

This class supports features common to pairings commands, and
is also used to test class membership.
See its parent class C<TSH::Command> for usage.
  
=head1 ABSTRACT

$setup = $command->SetupForPairing(%options);
$command->TidyAfterPairing($dp);

=cut

sub SetupForPairings ($%);
sub TidyAfterPairing ($$);

=head1 DESCRIPTION

=over 4

=item $setup = $command->SetupForPairing(%options);

Check and set up variables prior to performing pairings. 
%options is modified as necessary and a reference is returned.
Supported options:

division: (input) reference to a TSH::Division

exagony: (output) true if players from same team should not play each other

filter: (output) reference to an opponent filter to pass to PairGRT

nobye: (input) true if odd player groups should be left as is without
  assigning a bye, say because of a subsequent complex Gibson calculation,
  or because we're getting ready to do multiround pairings

required: (input) names of required configuration options

repeats: (input) number of repeats allowed

source0: (input) 0-based round on which to base standings

target0: (input) 0-based round in which to store pairings

=cut

sub SetupForPairings ($%) {
  my $this = shift;
  my (%setup) = @_;
  my $dp = $setup{'division'};
  my $sr0 = $setup{'source0'};
  my $repeats = $setup{'repeats'};
  my $tournament = $dp->Tournament();

  $dp->CheckRoundHasResults($sr0) or return 0;
  $setup{'target0'} = $dp->FirstUnpairedRound0();
  if ((defined $config::max_rounds) 
    && $config::max_rounds-1 < $setup{'target0'}) {
    $tournament->TellUser('ebigrd', $setup{'target0'}+1, $config::max_rounds);
    return 0;
    }

  # check for required configuration options
  if ($setup{'required'}) {
    for my $key (keys %{$setup{'required'}}) {
      next if defined $config::{$key};
      $tournament->TellUser("eneed_$key");
      }
    }
  # check for wanted configuration options
  if ($setup{'wanted_div'}) {
    my $dname = $dp->Name();
    for my $key (keys %{$setup{'wanted_div'}}) {
      next if defined $config::{$key} && $config::{$key}{$dname};
      # if not found, warn and apply default value
      $tournament->TellUser("wwant_$key");
      $config::{$key}{$dname} = $setup{'wanted_div'}{$key};
      }
    }

  # Remind user what pairings these are
  $tournament->TellUser("i$this->{'names'}[0]ok", $dp->Name(), 
    $setup{'target0'}+1, $sr0+1, $repeats);
  
  # If exagony is set, players from the same team cannot play each other
  if ($setup{'exagony'} = 
    ($config::exagony 
      || ($config::initial_exagony && $dp->LastPairedRound0() == -1))
    ) { $tournament->TellUser('irsem'); }

  # Build player list
  my $psp = $dp->GetUnpairedRound($setup{'target0'});
  unless (@$psp) { $tournament->TellUser('ealpaird'); return 0; }
  $dp->ChooseBye($sr0, $setup{'target0'}, $psp) 
    if @$psp % 2 && !$setup{'nobye'};
  $setup{'players'} = [TSH::Player::SortByStanding $sr0, @$psp];

  # Set up opponent filter
  $setup{'filter'} = sub {
    ($setup{'exagony'} ? 
      (
	$_[0][$_[1]]->Team() ne $_[0][$_[2]]->Team() 
	|| $_[0][$_[1]]->Team() eq ''
      )
      : 1
    ) 
    && $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $repeats
    };

  return \%setup;
  }

=over 4

=item $command->TidyAfterPairing($dp);

=cut

sub TidyAfterPairing ($$) {
  my $this = shift;
  my $dp = shift;

  $dp->Dirty(1);
  $dp->Update();
  $dp->Tournament()->TellUser('idone');
  }

=back

=cut

=head1 BUGS

SetupForPairings would be a good place to do some basic Gibsonization
detection.

=cut

1;
