#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ChewPair;

use strict;
use warnings;

use TSH::Utility qw(Debug DebugOn DebugOff DebugDumpPairings);
use TSH::PairingCommand;
use TSH::Player;

our (@ISA) = qw(TSH::PairingCommand);

DebugOn('CP');

=pod

=head1 NAME

TSH::Command::ChewPair - implement the C<tsh> ChewPair command

=head1 SYNOPSIS

  my $command = new TSH::Command::ChewPair;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ChewPair is a subclass of TSH::Command.

=cut

=head1 SUBROUTINES

=over 4

=cut

sub CountContenders ($$$$$);
sub FlightCapDefault ($);
sub FlightCapNSC ($);
sub GetPrizeBand($$$);
sub GibsonEquivalent($$);
sub initialise ($$$$);
sub new ($);
sub PairContenders ($$);
sub PairLeaders ($$$);
sub PairNonLeaders ($$);
sub Run ($$@);
sub RunInit ($$$);
sub SplitContenders ($$$);
sub Swiss ($$$$$);

=item $n = CountContenders($maxrank, $psp, $sr0, $rounds_left)

Count the number of players in contention for a prize band
based on results up to $sr0 with $rounds_left remaining.
Round up to an even number.
Cap at four times the number of rounds remaining.

=cut

sub CountContenders ($$$$$) {
  my $minrank = shift;
  my $maxrank = shift;
  my $psp = shift;
  my $sr0 = shift;
  my $rounds_left = shift;

  Debug 'CP', 'Contending rnks %d-%d?  Plyrs left: %d.  Rds left: %d as of Rd %d.', $minrank, $maxrank, scalar(@$psp), $rounds_left, $sr0+1;
  Debug 'CP', 'Now=>Fnl Wins Sprd Player (theoretical best finishes)';
  TSH::PairingCommand::CalculateBPFs $psp;
  
  my $max_flight_size = 0;
  for my $i (0..$#$psp) {
    my $r = $psp->[$i]->MaxRank();
#   Debug 'CP', '%s maxrank is %d', $psp->[$i]->TaggedName(), $r;
    if ($r <= $maxrank) {
      $max_flight_size = $i;
#     Debug 'CP', 'CC: %d <= %d, mfs=%d', $r, $maxrank, $i;
      }
    }
  TSH::Utility::Error "Assertion failed: only one contender" 
    if $max_flight_size == 0;
  $max_flight_size++; # now one-based
  Debug 'CP', 'There are %d contenders.', $max_flight_size;
  $max_flight_size++ if $max_flight_size % 2; # round up to even
  # if contenders are more than four times rounds left, cap to that amount
  # see also CalculateBestPossibleFinish
  my $cap;
  {
    no strict 'refs';
    $cap = &$config::flight_cap($rounds_left);
  }
  if ($minrank == 1 && $max_flight_size > $cap && $sr0 >= 0) {
    $max_flight_size = $cap;
    Debug 'CP', 'Capping to %d contenders.', $max_flight_size;
    }
  return $max_flight_size;
  }

=item $cap = FlightCapDefault($rounds_left);

Apply the default algorithm for mapping rounds left to cap on
number of contenders.

=cut

sub FlightCapDefault ($) {
  my $rounds_left = shift;
  return $rounds_left <= 1 ? 2 :
    $rounds_left >= 4 ? 12 :
    ($rounds_left - 1) * 4;
  }

=item $cap = FlightCapNSC($rounds_left);

Apply the NSC algorithm for mapping rounds left to cap on
number of contenders.

=cut

sub FlightCapNSC ($) {
  my $rounds_left = shift;
  return $rounds_left > 3 ? 12 : $rounds_left * 4;
  }

=item ($minrank,$maxrank) = $parserp->GetPrizeBand($prize_bands, $psp, $sr0)

Find the prize band that contains the given rank.

=cut

sub GetPrizeBand ($$$) {
  my $prize_bands = shift;
  my $psp = shift;
  my $sr0 = shift;
  die "Assertion failed" unless @$psp;
  my $rank = $psp->[0]->RoundRank($config::max_rounds);
  my (@splits) = (0, @$prize_bands, $psp->[-1]->RoundRank($config::max_rounds));
# Debug 'CP', "GetPB: $rank, @splits";
  for my $i (1..$#splits) {
    my $minrank = $splits[$i-1]+1;
    my $maxrank = $splits[$i];
    if ($rank >= $minrank && $rank <= $maxrank) {
      return ($minrank, $maxrank);
      }
    }
  die "Assertion failed with rank=$rank, splits=@splits";
  }

=item $parserp->GibsonEquivalent($dname, $rank)

Return the highest rank equivalent for Gibson purposes to $rank.

=cut

sub GibsonEquivalent($$) {
  my $dname = shift;
  my $rank = shift;

  return ($config::gibson_equivalent{$dname}[$rank] || $rank);
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to compute Chew pairings for a division.
EOF
  $this->{'names'} = [qw(cp chewpair)];
  $this->{'argtypes'} = [qw(BasedOnRound Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { 
  return TSH::Utility::new(@_); 
  }

=item $parserp->PairContenders($dp);

Pair all contenders at the top of $this->{'cp_tobepaired'};

=cut

sub PairContenders ($$) {
  my $this = shift;
  my $dp = shift;
  my $prize_bands = $this->{'cp_current_prize_bands'};
  my $round0 = $this->{'cp_round0'};
  my $sr0 = $this->{'cp_sr0'};
  my $rounds_left = $this->{'cp_rounds_left'};
  my $tobepaired = $this->{'cp_tobepaired'};
  # find the highest remaining prize band
  my ($minrank, $maxrank) = GetPrizeBand($prize_bands, $tobepaired, $sr0);
# Debug 'CP', 'prize_band: [%d,%d]', $minrank, $maxrank;
  # count prize band contenders
  my $ncontenders 
    = CountContenders($minrank,$maxrank,$tobepaired,$sr0,$rounds_left); 
  die "assertion failed" if $ncontenders > @$tobepaired;
  if ($ncontenders == @$tobepaired - 2) {
    my $repeats = $tobepaired->[-2]->Repeats($tobepaired->[-1]->ID());
    if ($repeats > 0) {
      $ncontenders += 2;
      Debug 'CP', 'Adding back last two players, their repeats=%d', $repeats;
      }
    }
  # compute the smallest number of repeats 'minrep' needed to pair band
  my $minrep = 0;
  my (@flight) = @$tobepaired[0..$ncontenders-1];
  until (TSH::Player::CanBePaired $minrep, \@flight) { $minrep++; }
  Debug 'CP', 'Flight can be paired with repeats=%d.', $minrep;
  # if everyone is still a contender, Swiss-pair them
  if ($ncontenders == @$tobepaired) { Debug 'CP', 'Everyone is a contender, will pair Swiss.';
    Swiss $dp, \@flight, $minrep, $sr0, $round0;
    }
  # else there are some contenders and some noncontenders
  else {
    # find the highest split into two groups pairable in minrep
    my $split = $this->SplitContenders(\@flight, $minrep);
#     DebugOn('GRT');
    # pair the flight leaders 
    PairLeaders [@flight[0..$split-1]], $minrep, $sr0;
    # pair the rest Swiss
# OLD PairNonLeaders [@flight[$split..$#flight]], $minrep if $split <= $#flight;
    if ($split <= $#flight) {
      Debug 'CP', 'Pairing %d non-leaders.', $#flight-$split+1;
      Swiss $dp, [@flight[$split..$#flight]], $minrep, $sr0, $round0;
      }
#     DebugOff('GRT');
    }
# splice(@$tobepaired, 0, $ncontenders); # splices are not thread-safe
  TSH::Utility::SpliceSafely(@$tobepaired, 0, $ncontenders);
  }

=item PairLeaders($$$)

Pair an even number of flight leaders with decreasing priority given to
with decreasing priority given to

  - matching leaders with opponents who could catch up to them
  - not exceeding minrep (ever)
  - minimizing repeats
  - avoiding consecutive rematches (where possible)
  - pairing highest with lowest

=cut

sub PairLeaders ($$$) {
  my $psp = shift;
  my $minrep = shift;
  my $sr0 = shift;
  Debug 'CP', 'Pairing %d leaders.', scalar(@$psp);
  return unless @$psp;
  my $r0 = $psp->[0]->CountOpponents();
  unless (
    TSH::Player::PairGRT($psp,
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $ketchup = ($o->MaxRank() > $p->RoundRank($config::max_rounds));
#	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

#	Debug 'GRT', 'pref %d-%d rep=%d prev=%d can=%d (svr=%d)', $pid, $oid, $repeats, $sameopp, $ketchup, $pairsvr;
 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d can=%d', $pid, $oid, $repeats, $sameopp, $ketchup;
	pack('NCCNN',
	  $ketchup, # prefer those who can catch up
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
#	  $pairsvr, # pair those due to start vs those due to reply
	  999999-$_[2], # prefer lower-ranked opponents
	  $_[2], # for ID recovery
	  )
        },
      # allowed opponent filter
      sub {
 	Debug 'GRT', 'filter: %s vs %s %d ?<= %d', $_[0][$_[1]]->TaggedName(), $_[0][$_[2]]->TaggedName(), $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()), $minrep;
	# do not exceed minrep
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $minrep
        },
      [],
      $r0,
      )
    ) # unless
    {
    TSH::Utility::Error "Assertion failed: can't resolve leader pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairNonLeaders($$)

This routine is not currently used, as nonleaders are now paired Swiss.

Pair an even number of flight non-leaders with decreasing priority given to
with decreasing priority given to

  - not exceeding minrep (ever)
  - minimizing rematches
  - avoiding consecutive rematches
  - matching players due to go first with those due to go second
  - keeping rank differences close to half the group size

=cut

sub PairNonLeaders ($$) {
  my $psp = shift;
  my $minrep = shift;
  Debug 'CP', 'Pairing %d non-leaders.', scalar(@$psp);
  return unless @$psp;
  my $r0 = $psp->[0]->CountOpponents();
  unless (
    TSH::Player::PairGRT($psp,
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs(@{$_[0]}-abs(2*($_[1]-$_[2])));
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d dist=%d svr=%d', $pid, $oid, $repeats, $sameopp, $distance, $pairsvr;
	pack('NCCNN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance,# prefer opponents close to half the group size away in rank
	  # provide index for GRT to extract, effectively prefer higher opps
	  $_[2],
	  )
        },
      # allowed opponent filter
      sub {
#	Debug 'GRT', 'filter %d %d', $_[1], $_[2];
  	Debug 'GRT', 'filter: %s vs %s %d ?<= %d', $_[0][$_[1]]->TaggedName(), $_[0][$_[2]]->TaggedName(), $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()), $minrep;
	# do not exceed minrep
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $minrep
        },
      [],
      $r0,
      )
    ) # unless
    {
    TSH::Utility::Error "Assertion failed: can't resolve non-leader pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item $command->Run($tournament, @parsed_args);

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();
  my ($sr, $dp) = @_;
  my $sr0 = $this->{'cp_sr0'} = $sr - 1;
  my @default_bands : shared = (int($dp->CountPlayers/4)||1);
  my $setupp = $this->SetupForPairings(
    'division' => $dp,
    'required' => { 'max_rounds' => 1 },
    'nobye' => 1,
    'source0' => $sr0,
    'wanted_div' => { 'prize_bands' => \@default_bands },
    ) or return 0;
  
  my $start = time;

  return unless $this->RunInit($tournament, $setupp);
  my $round0 = $this->{'cp_round0'};
  my $tobepaired = $this->{'cp_tobepaired'};
  Debug 'CP', 'Beginning Chew Pairings for %d players in round %d based on round %d.', $#$tobepaired+1, $round0+1, $sr0+1;
  # while we still have players left to pair
  while (@$tobepaired) { 
    Debug 'CP', 'Number of players still unpaired: %d', scalar(@$tobepaired);
    # perform any necessary gibsonization
    $this->PairAllGibsons($this->{'cp_tobepaired'}, $this->{'cp_sr0'},
      $this->{'cp_rounds_left'}, $this->{'cp_current_prize_bands'}[-1]);
    # assign a bye if necessary
    if (@$tobepaired % 2) { Debug 'CP', 'need a bye';
      $dp->ChooseBye($sr0, $round0, $tobepaired);
      }
    last unless @$tobepaired;
    # pair anyone in contention for the top prize still available
    $this->PairContenders($dp);
    }
  $dp->Dirty(1);
  $dp->Update();
  $tournament->TellUser('idone');
  Debug('CP', '%d second(s) runtime', time - $start);
  0;
  }

=item $ok = $cmd->RunInit($tournament, $dp);

Perform initialisation necessary prior to running the command.

=cut

sub RunInit ($$$) {
  my $this = shift;
  my $tournament = shift;
  my $setupp = shift;
  my $config = $tournament->Config();
  my $dp = $setupp->{'division'};
  my $sr0 = $this->{'cp_sr0'};
# $this->{'cp_rounds_left'} = $config::max_rounds - ($sr0+1); # see *20060430
  $this->{'cp_current_prize_bands'} = $config::prize_bands{$dp->Name()};
# $dp->ComputeRanks($this->{'cp_sr0'});
  $dp->ComputeRanks($config::max_rounds);
  my $round0 = $this->{'cp_round0'} =  $setupp->{'target0'};
  $this->{'cp_rounds_left'} = $config::max_rounds - $round0; # *20060430
  my $tobepaired = $dp->GetUnpairedRound($round0);
  unless (@$tobepaired) { 
    Debug 'CP', 
      "assertion failed at " . __FILE__ . ' line ' . __LINE__ . ": r0=$round0";
    $tournament->TellUser('ealpaird'); 
    return 0; 
    }
  @$tobepaired = TSH::Player::SortByStanding($this->{'cp_sr0'}, @$tobepaired);
  $this->{'cp_tobepaired'} = $tobepaired;

  return 1;
  }

=item $nleaders = $parser->SplitContenders(\@contenders, $repeats);

Decide where to split those players still in contention into two
pairings groups.  The cut goes as high as possible (that is,
keeping together the fewest leaders) subject to not exceeding the
number of repeats required to pair all the contenders as one group.
The return value is the number of leaders who will be split off
from the nonleading contenders.  If the contenders cannot be split
into two groups without increasing repeats, then the number of leaders
is equal to the number of contenders.

=cut

sub SplitContenders ($$$) {
  my $this = shift;
  my $contendersp = shift;
  my $repeats = shift;
  my $split = 0;
  while (($split += 2) < @$contendersp) {
    if (TSH::Player::CanBePaired $repeats, [@$contendersp[0..$split-1]]) {
      Debug 'CP', 'Top %d can be paired with rep=%d.', $split, $repeats;
      }
    else {
      Debug 'CP', 'Top %d cannot be paired with rep=%d.', $split, $repeats;
      next;
      }
    if (TSH::Player::CanBePaired $repeats, [@$contendersp[$split..$#$contendersp]]) {
      Debug 'CP', 'Rest can also be paired with rep=%d.', $repeats;
      last;
      }
    else {
      Debug 'CP', 'Rest cannot be paired with rep=%d.', $repeats;
      next;
      }
    }
  Debug 'CP', 'split is %d.', $split;
  return $split;
  }

=item @pairlist = Swiss($dp, \@ps, $repeats, $sr0, $round0);

Compute and save Swiss pairings for some set of players.

=cut

sub Swiss ($$$$$) {
  my $dp = shift;
  my $psp = shift;
  my $repeats = shift;
  my $sr0 = shift;
  my $round0 = shift;

# Debug 'CP', 'Swiss(%s,%d,%d,%d,%d)', $dp->Name(), $#$psp+1, $repeats,$sr0,$round0;
  my (@pair_list) = TSH::Division::PairSomeSwiss($psp, $repeats, $sr0);
  die "assertion failed" unless @pair_list;
# die "assertion failed: " . join(',', @pair_list) if @pair_list % 2;
  while (@pair_list) {
    my $p1 = shift @pair_list;
    my $p2 = shift @pair_list;
# die "assertion failed for $p1" unless UNIVERSAL::isa($p1, 'TSH::Player');
# die "assertion failed for $p2" unless UNIVERSAL::isa($p2, 'TSH::Player');
    $dp->Pair($p1->ID(), $p2->ID(), $round0);
    }
  }
=back

=cut

=head1 BUGS

None known.

=cut

1;
