#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundRobin;

use strict;
use warnings;

use TSH::PairingCommand;
use TSH::Utility;
use TSH::Division::Pairing::Clark;

our (@ISA) = qw(TSH::PairingCommand);
our @Pairings;
our @Starts;

=pod

=head1 NAME

TSH::Command::RoundRobin - implement the C<tsh> RoundRobin command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundRobin;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundRobin is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the RoundRobin command to manually add a set of full round robin pairings
to a division.  When you use this command, the last paired round must
be fully, not partially paired.  If you specify an integer before the
division name, each pairing will be repeated that number of times.
EOF
  $this->{'names'} = [qw(rr roundrobin)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $count = @_ ? pop @_ : 1;

  my $datap = $dp->{'data'};
# my $nplayers = $#$datap;

  # check that division is not partially paired
  my $round0 = $dp->FirstUnpairedRound0();
  if ($round0 != $dp->LastPairedRound0()+1) {
    $tournament->TellUser('errpartp');
    return;
    }
# TSH::Player::SpliceInactive @$unpaired, $nplayers-1;

  my $psp = $dp->GetUnpairedRound($round0);
  my $need_bye = @$psp % 2;
  my $nplayers = scalar(@$psp);
  my $schedule_size = $nplayers + $need_bye;
  if (my $ptablep = $Pairings[$schedule_size]) {
    $tournament->TellUser('irrtok', $dp->{'name'}, $count);
    my (@ps) = TSH::Player::SortByCurrentStanding @$psp;
#   print join("\n", map { $_->TaggedName() } @ps), "\n";
    my $stablep;
    if ($config::track_firsts && $config::assign_firsts) 
      { $stablep = $Starts[$schedule_size]; }
    for my $r (1..$schedule_size-1) {
      my $round_pairings_p = $ptablep->[$r];
      my $round_starts_p;
      if ($stablep) { $round_starts_p = $stablep->[$r]; }
      my $phase = 0;
      for my $j (1..$count) {
	for my $i (1..$nplayers) {
	  my $p1p = $ps[$i-1];
	  my $p1 = $p1p->ID();
	  my $p2i = $round_pairings_p->[$i];
	  my $p2 = $p2i > $nplayers ? 0 : $ps[$p2i-1]->ID();
	  if ($round_starts_p) { 
	    my $p12 = $p2 ? $round_starts_p->[$i] : 0;
	    if ($phase && ($p12 == 1 || $p12 == 2)) {
	      $p12 = 3 - $p12;
	      }
  #	  printf "Setting p12 for %d to %d in round %d\n", $p1, $p12, $round0+1;
	    $p1p->First($round0, $p12);
	    }
	  $dp->Pair($p1, $p2, $round0) if $p1 > $p2;
	  }
	$round0++;
	$phase = 1 - $phase;
        }
      }
    }
  else {
    $tournament->TellUser('irrcok', $dp->{'name'});
    # add pairings information one round at a time
    for (my $oppi = $schedule_size; $oppi > 1; $oppi--) {
      for my $j (1..$count) {
	TSH::Division::Pairing::Clark::PairGroup 
	  $dp->GetUnpairedRound($round0++), $oppi; 
	}
      }
    }
  $tournament->TellUser('idone');

  $dp->Dirty(1);
  $dp->Update();
  }

=item @Pairings, @Starts

Balanced round-robin schedules are precomputed by util/rr.pl
for up to 26 players.  If you need to run a larger round-robin
tournament please contact John Chew and he'll add to this list.

=cut

$Pairings[2][1] = [undef,2,1];
$Starts[2][1] = [undef,4,4];

$Pairings[4][1] = [undef,4,3,2,1];
$Pairings[4][2] = [undef,3,4,1,2];
$Pairings[4][3] = [undef,2,1,4,3];
$Starts[4][1] = [undef,1,1,2,2];
$Starts[4][2] = [undef,2,2,1,1];
$Starts[4][3] = [undef,4,4,4,4];

$Pairings[6][1] = [undef,6,5,4,3,2,1];
$Pairings[6][2] = [undef,5,3,2,6,1,4];
$Pairings[6][3] = [undef,4,6,5,1,3,2];
$Pairings[6][4] = [undef,3,4,1,2,6,5];
$Pairings[6][5] = [undef,2,1,6,5,4,3];
$Starts[6][1] = [undef,2,2,1,2,1,1];
$Starts[6][2] = [undef,1,1,2,1,2,2];
$Starts[6][3] = [undef,1,1,2,2,1,2];
$Starts[6][4] = [undef,2,2,1,1,2,1];
$Starts[6][5] = [undef,4,4,4,4,4,4];

$Pairings[8][1] = [undef,8,3,2,6,7,4,5,1];
$Pairings[8][2] = [undef,7,4,5,2,3,8,1,6];
$Pairings[8][3] = [undef,6,5,7,8,2,1,3,4];
$Pairings[8][4] = [undef,5,6,4,3,1,2,8,7];
$Pairings[8][5] = [undef,4,7,6,1,8,3,2,5];
$Pairings[8][6] = [undef,3,8,1,5,4,7,6,2];
$Pairings[8][7] = [undef,2,1,8,7,6,5,4,3];
$Starts[8][1] = [undef,1,1,2,2,1,1,2,2];
$Starts[8][2] = [undef,2,2,1,1,2,2,1,1];
$Starts[8][3] = [undef,1,1,1,2,2,2,2,1];
$Starts[8][4] = [undef,2,2,2,1,1,1,1,2];
$Starts[8][5] = [undef,1,1,2,2,1,1,2,2];
$Starts[8][6] = [undef,2,2,1,1,2,2,1,1];
$Starts[8][7] = [undef,4,4,4,4,4,4,4,4];

$Pairings[10][1] = [undef,10,9,5,7,3,8,4,6,2,1];
$Pairings[10][2] = [undef,9,5,7,6,2,4,3,10,1,8];
$Pairings[10][3] = [undef,8,6,4,3,7,2,5,1,10,9];
$Pairings[10][4] = [undef,7,3,2,8,9,10,1,4,5,6];
$Pairings[10][5] = [undef,6,10,9,5,4,1,8,7,3,2];
$Pairings[10][6] = [undef,5,7,6,10,1,3,2,9,8,4];
$Pairings[10][7] = [undef,4,8,10,1,6,5,9,2,7,3];
$Pairings[10][8] = [undef,3,4,1,2,8,9,10,5,6,7];
$Pairings[10][9] = [undef,2,1,8,9,10,7,6,3,4,5];
$Starts[10][1] = [undef,1,1,1,1,2,2,2,1,2,2];
$Starts[10][2] = [undef,2,2,2,2,1,1,1,2,1,1];
$Starts[10][3] = [undef,1,1,2,1,1,2,2,2,2,1];
$Starts[10][4] = [undef,2,2,1,2,2,1,1,1,1,2];
$Starts[10][5] = [undef,2,2,2,2,1,1,1,2,1,1];
$Starts[10][6] = [undef,1,1,1,1,2,2,2,1,2,2];
$Starts[10][7] = [undef,2,2,1,1,2,1,1,1,2,2];
$Starts[10][8] = [undef,1,1,2,2,1,2,2,2,1,1];
$Starts[10][9] = [undef,4,4,4,4,4,4,4,4,4,4];

$Pairings[12][1] = [undef,12,7,5,6,3,4,2,10,11,8,9,1];
$Pairings[12][2] = [undef,11,3,2,5,4,8,9,6,7,12,1,10];
$Pairings[12][3] = [undef,10,9,4,3,6,5,11,12,2,1,7,8];
$Pairings[12][4] = [undef,9,5,7,8,2,10,3,4,1,6,12,11];
$Pairings[12][5] = [undef,8,11,9,10,7,12,5,1,3,4,2,6];
$Pairings[12][6] = [undef,7,4,6,2,8,3,1,5,12,11,10,9];
$Pairings[12][7] = [undef,6,10,8,11,9,1,12,3,5,2,4,7];
$Pairings[12][8] = [undef,5,6,12,7,1,2,4,11,10,9,8,3];
$Pairings[12][9] = [undef,4,12,10,1,11,7,6,9,8,3,5,2];
$Pairings[12][10] = [undef,3,8,1,9,12,11,10,2,4,7,6,5];
$Pairings[12][11] = [undef,2,1,11,12,10,9,8,7,6,5,3,4];
$Starts[12][1] = [undef,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[12][2] = [undef,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[12][3] = [undef,2,2,1,2,1,2,2,1,1,1,1,2];
$Starts[12][4] = [undef,1,1,2,1,2,1,1,2,2,2,2,1];
$Starts[12][5] = [undef,1,1,1,2,1,2,2,2,2,1,2,1];
$Starts[12][6] = [undef,2,2,2,1,2,1,1,1,1,2,1,2];
$Starts[12][7] = [undef,1,1,2,1,2,2,2,1,1,2,2,1];
$Starts[12][8] = [undef,2,2,1,2,1,1,1,2,2,1,1,2];
$Starts[12][9] = [undef,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[12][10] = [undef,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[12][11] = [undef,4,4,4,4,4,4,4,4,4,4,4,4];

$Pairings[14][1] = [undef,14,13,5,11,3,7,6,9,8,12,4,10,2,1];
$Pairings[14][2] = [undef,13,9,11,6,8,4,10,5,2,7,3,14,1,12];
$Pairings[14][3] = [undef,12,6,8,7,11,2,4,3,13,14,5,1,9,10];
$Pairings[14][4] = [undef,11,8,4,3,9,10,12,2,5,6,1,7,14,13];
$Pairings[14][5] = [undef,10,14,9,5,4,12,11,13,3,1,7,6,8,2];
$Pairings[14][6] = [undef,9,5,6,10,2,3,14,11,1,4,8,13,12,7];
$Pairings[14][7] = [undef,8,3,2,12,13,14,9,1,7,11,10,4,5,6];
$Pairings[14][8] = [undef,7,12,14,8,6,5,1,4,11,13,9,2,10,3];
$Pairings[14][9] = [undef,6,10,13,9,7,1,5,14,4,2,12,11,3,8];
$Pairings[14][10] = [undef,5,11,10,14,1,8,13,6,12,3,2,9,7,4];
$Pairings[14][11] = [undef,4,7,12,1,14,9,2,10,6,8,13,3,11,5];
$Pairings[14][12] = [undef,3,4,1,2,12,13,8,7,10,9,14,5,6,11];
$Pairings[14][13] = [undef,2,1,7,13,10,11,3,12,14,5,6,8,4,9];
$Starts[14][1] = [undef,2,2,2,2,1,1,2,2,1,1,1,2,1,1];
$Starts[14][2] = [undef,1,1,1,1,2,2,1,1,2,2,2,1,2,2];
$Starts[14][3] = [undef,1,1,1,2,1,2,1,2,2,1,2,2,1,2];
$Starts[14][4] = [undef,2,2,2,1,2,1,2,1,1,2,1,1,2,1];
$Starts[14][5] = [undef,1,1,1,1,2,2,1,1,2,2,2,1,2,2];
$Starts[14][6] = [undef,2,2,2,2,1,1,2,2,1,1,1,2,1,1];
$Starts[14][7] = [undef,1,1,2,1,1,2,2,2,1,1,2,2,2,1];
$Starts[14][8] = [undef,2,2,1,2,2,1,1,1,2,2,1,1,1,2];
$Starts[14][9] = [undef,2,2,2,2,1,1,2,2,1,1,1,2,1,1];
$Starts[14][10] = [undef,1,1,1,1,2,2,1,1,2,2,2,1,2,2];
$Starts[14][11] = [undef,1,1,2,2,2,2,2,1,1,2,2,1,1,1];
$Starts[14][12] = [undef,2,2,1,1,1,1,1,2,2,1,1,2,2,2];
$Starts[14][13] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4];

$Pairings[16][1] = [undef,16,3,2,6,7,4,5,10,11,8,9,14,15,12,13,1];
$Pairings[16][2] = [undef,15,4,5,2,3,8,9,6,7,12,13,10,11,16,1,14];
$Pairings[16][3] = [undef,14,5,7,8,2,10,3,4,13,6,15,16,9,1,11,12];
$Pairings[16][4] = [undef,13,6,4,3,9,2,11,12,5,14,7,8,1,10,16,15];
$Pairings[16][5] = [undef,12,7,6,10,11,3,2,14,15,4,5,1,16,8,9,13];
$Pairings[16][6] = [undef,11,8,9,5,4,12,13,2,3,16,1,6,7,15,14,10];
$Pairings[16][7] = [undef,10,9,11,12,13,14,15,16,2,1,3,4,5,6,7,8];
$Pairings[16][8] = [undef,9,10,8,7,6,5,4,3,1,2,16,15,14,13,12,11];
$Pairings[16][9] = [undef,8,11,10,14,15,7,6,1,16,3,2,13,12,4,5,9];
$Pairings[16][10] = [undef,7,12,13,9,8,16,1,5,4,15,14,2,3,11,10,6];
$Pairings[16][11] = [undef,6,13,15,16,10,1,8,7,14,5,12,11,2,9,3,4];
$Pairings[16][12] = [undef,5,14,12,11,1,9,16,15,6,13,4,3,10,2,8,7];
$Pairings[16][13] = [undef,4,15,14,1,16,11,10,13,12,7,6,9,8,3,2,5];
$Pairings[16][14] = [undef,3,16,1,13,12,15,14,9,8,11,10,5,4,7,6,2];
$Pairings[16][15] = [undef,2,1,16,15,14,13,12,11,10,9,8,7,6,5,4,3];
$Starts[16][1] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[16][2] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[16][3] = [undef,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[16][4] = [undef,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[16][5] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[16][6] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[16][7] = [undef,1,1,1,2,1,2,1,2,2,2,2,1,2,1,2,1];
$Starts[16][8] = [undef,2,2,2,1,2,1,2,1,1,1,1,2,1,2,1,2];
$Starts[16][9] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[16][10] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[16][11] = [undef,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[16][12] = [undef,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[16][13] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[16][14] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[16][15] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];

$Pairings[18][1] = [undef,18,4,5,2,3,8,9,6,7,11,10,13,12,16,17,14,15,1];
$Pairings[18][2] = [undef,17,6,4,3,7,2,5,10,12,8,14,9,15,11,13,18,1,16];
$Pairings[18][3] = [undef,16,3,2,5,4,10,14,11,13,6,8,15,9,7,12,1,18,17];
$Pairings[18][4] = [undef,15,8,7,6,9,4,3,2,5,12,16,10,17,18,1,11,13,14];
$Pairings[18][5] = [undef,14,5,6,8,2,3,13,4,15,16,18,17,7,1,9,10,12,11];
$Pairings[18][6] = [undef,13,10,9,7,12,11,4,14,3,2,6,5,1,8,18,17,16,15];
$Pairings[18][7] = [undef,12,11,8,10,16,14,15,3,17,4,2,1,18,6,7,5,9,13];
$Pairings[18][8] = [undef,11,7,12,9,6,5,2,13,4,18,1,3,8,17,16,15,14,10];
$Pairings[18][9] = [undef,10,9,13,11,8,16,17,5,2,1,4,18,3,15,14,6,7,12];
$Pairings[18][10] = [undef,9,14,10,12,15,7,6,18,1,3,17,4,16,2,5,13,11,8];
$Pairings[18][11] = [undef,8,12,18,14,17,9,16,1,6,13,15,2,10,4,11,7,5,3];
$Pairings[18][12] = [undef,7,16,11,13,14,15,1,12,18,17,3,8,4,5,6,2,10,9];
$Pairings[18][13] = [undef,6,13,17,16,11,1,18,9,8,15,5,14,2,12,10,4,3,7];
$Pairings[18][14] = [undef,5,18,14,15,1,12,10,17,16,7,13,6,11,3,4,9,8,2];
$Pairings[18][15] = [undef,4,15,16,1,18,17,8,7,10,9,12,11,14,13,2,3,6,5];
$Pairings[18][16] = [undef,3,17,1,18,10,13,11,15,14,5,7,16,6,9,8,12,2,4];
$Pairings[18][17] = [undef,2,1,15,17,13,18,12,16,11,14,9,7,5,10,3,8,4,6];
$Starts[18][1] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[18][2] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[18][3] = [undef,2,2,1,1,2,2,2,1,1,1,2,2,2,1,1,1,1,2];
$Starts[18][4] = [undef,1,1,2,2,1,1,1,2,2,2,1,1,1,2,2,2,2,1];
$Starts[18][5] = [undef,1,1,2,2,2,1,1,1,1,2,2,1,2,2,2,1,2,1];
$Starts[18][6] = [undef,2,2,1,1,1,2,2,2,2,1,1,2,1,1,1,2,1,2];
$Starts[18][7] = [undef,2,2,2,1,2,1,1,1,2,2,1,1,2,2,2,1,1,1];
$Starts[18][8] = [undef,1,1,1,2,1,2,2,2,1,1,2,2,1,1,1,2,2,2];
$Starts[18][9] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[18][10] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[18][11] = [undef,2,1,2,2,2,1,1,1,2,2,1,2,1,1,2,2,1,1];
$Starts[18][12] = [undef,1,2,1,1,1,2,2,2,1,1,2,1,2,2,1,1,2,2];
$Starts[18][13] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[18][14] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[18][15] = [undef,2,2,1,1,1,2,1,2,1,2,2,1,1,2,1,2,1,2];
$Starts[18][16] = [undef,1,1,2,2,2,1,2,1,2,1,1,2,2,1,2,1,2,1];
$Starts[18][17] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];

$Pairings[20][1] = [undef,20,15,7,6,13,4,3,10,11,8,9,14,5,12,2,18,19,16,17,1];
$Pairings[20][2] = [undef,19,11,5,13,3,8,9,6,7,12,2,10,4,16,17,14,15,20,1,18];
$Pairings[20][3] = [undef,18,17,9,5,4,13,11,12,3,14,7,8,6,10,19,20,2,1,15,16];
$Pairings[20][4] = [undef,17,9,13,8,7,10,5,4,2,6,15,16,3,18,11,12,1,14,20,19];
$Pairings[20][5] = [undef,16,19,11,10,6,5,13,14,15,4,3,18,7,8,9,1,20,12,2,17];
$Pairings[20][6] = [undef,15,7,4,3,9,12,2,13,5,16,17,6,8,20,1,10,11,19,18,14];
$Pairings[20][7] = [undef,14,10,6,7,8,3,4,5,13,2,12,11,9,1,20,19,18,17,16,15];
$Pairings[20][8] = [undef,13,3,2,12,20,14,15,16,17,18,19,4,1,6,7,8,9,10,11,5];
$Pairings[20][9] = [undef,12,5,15,14,2,16,17,18,19,20,13,1,11,4,3,6,7,8,9,10];
$Pairings[20][10] = [undef,11,20,8,9,10,7,6,3,4,5,1,13,12,19,18,17,16,15,14,2];
$Pairings[20][11] = [undef,10,18,17,11,12,9,19,20,6,1,4,5,14,13,16,15,3,2,7,8];
$Pairings[20][12] = [undef,9,13,10,16,15,18,8,7,1,3,20,19,2,17,5,4,14,6,12,11];
$Pairings[20][13] = [undef,8,16,19,18,14,11,10,1,20,7,6,17,15,5,13,2,12,4,3,9];
$Pairings[20][14] = [undef,7,4,12,2,17,20,1,9,8,19,18,3,16,15,14,13,5,11,10,6];
$Pairings[20][15] = [undef,6,14,18,20,19,1,16,11,10,9,8,15,17,2,12,7,13,3,5,4];
$Pairings[20][16] = [undef,5,6,14,15,1,2,12,19,18,17,16,7,20,3,4,11,10,9,8,13];
$Pairings[20][17] = [undef,4,12,20,1,18,15,14,17,16,11,10,2,19,7,6,9,8,5,13,3];
$Pairings[20][18] = [undef,3,8,1,17,16,19,20,2,12,15,14,9,18,11,10,5,4,13,6,7];
$Pairings[20][19] = [undef,2,1,16,19,11,17,18,15,14,13,5,20,10,9,8,3,6,7,4,12];
$Starts[20][1] = [undef,2,2,2,1,1,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[20][2] = [undef,1,1,1,2,2,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[20][3] = [undef,1,1,1,1,2,1,1,2,2,2,2,1,2,1,1,2,2,2,2,1];
$Starts[20][4] = [undef,2,2,2,2,1,2,2,1,1,1,1,2,1,2,2,1,1,1,1,2];
$Starts[20][5] = [undef,1,1,1,2,2,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[20][6] = [undef,2,2,2,1,1,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[20][7] = [undef,1,1,2,2,2,1,1,1,1,2,2,1,2,2,2,2,2,1,1,1];
$Starts[20][8] = [undef,2,2,1,1,1,2,2,2,2,1,1,2,1,1,1,1,1,2,2,2];
$Starts[20][9] = [undef,1,2,1,2,1,2,1,2,1,2,2,2,1,1,2,1,2,1,2,1];
$Starts[20][10] = [undef,2,1,2,1,2,1,2,1,2,1,1,1,2,2,1,2,1,2,1,2];
$Starts[20][11] = [undef,2,2,2,2,1,2,2,1,1,1,1,2,1,2,2,1,1,1,1,2];
$Starts[20][12] = [undef,1,1,1,1,2,1,1,2,2,2,2,1,2,1,1,2,2,2,2,1];
$Starts[20][13] = [undef,1,1,1,2,2,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[20][14] = [undef,2,2,2,1,1,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[20][15] = [undef,1,1,1,2,2,2,1,2,1,2,1,2,2,2,1,2,1,2,1,1];
$Starts[20][16] = [undef,2,2,2,1,1,1,2,1,2,1,2,1,1,1,2,1,2,1,2,2];
$Starts[20][17] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,1,1,2];
$Starts[20][18] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,2,2,1];
$Starts[20][19] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];

$Pairings[22][1] = [undef,22,4,5,2,3,7,6,9,8,12,13,10,11,15,14,17,16,20,21,18,19,1];
$Pairings[22][2] = [undef,21,6,4,3,8,2,10,5,11,7,9,14,16,12,18,13,19,15,17,22,1,20];
$Pairings[22][3] = [undef,20,3,2,5,4,10,12,11,15,6,8,7,17,18,9,19,13,14,16,1,22,21];
$Pairings[22][4] = [undef,19,7,8,6,9,4,2,3,5,14,16,13,12,10,20,11,21,22,1,15,17,18];
$Pairings[22][5] = [undef,18,5,9,7,2,12,4,13,3,15,17,6,8,20,10,21,11,1,22,14,16,19];
$Pairings[22][6] = [undef,17,10,6,8,11,3,14,4,16,2,5,18,19,7,22,9,1,12,13,21,20,15];
$Pairings[22][7] = [undef,16,12,11,10,6,5,15,18,17,4,3,2,21,22,7,1,9,8,20,19,13,14];
$Pairings[22][8] = [undef,15,8,7,9,13,14,3,2,4,16,19,20,5,6,1,10,22,21,11,12,18,17];
$Pairings[22][9] = [undef,14,9,13,12,7,15,5,17,2,20,21,4,3,1,6,22,8,19,18,10,11,16];
$Pairings[22][10] = [undef,13,14,10,11,16,8,18,6,19,3,4,22,1,2,21,5,20,7,9,17,15,12];
$Pairings[22][11] = [undef,12,11,16,13,20,9,8,7,6,22,2,1,4,21,19,3,18,17,15,5,14,10];
$Pairings[22][12] = [undef,11,15,12,14,10,18,17,19,21,5,1,3,22,4,2,20,7,6,8,16,9,13];
$Pairings[22][13] = [undef,10,13,17,15,12,20,9,21,7,1,22,5,2,19,4,18,3,16,14,6,8,11];
$Pairings[22][14] = [undef,9,18,14,16,19,11,22,10,1,8,6,21,20,3,17,4,15,2,5,13,12,7];
$Pairings[22][15] = [undef,8,20,22,18,21,13,11,1,10,9,7,19,6,17,16,15,14,4,12,2,5,3];
$Pairings[22][16] = [undef,7,16,15,17,14,19,1,12,22,21,20,8,18,5,3,2,4,13,6,11,10,9];
$Pairings[22][17] = [undef,6,17,21,20,15,1,13,22,12,19,18,9,7,16,5,14,2,11,10,4,3,8];
$Pairings[22][18] = [undef,5,22,18,19,1,16,21,14,20,11,10,17,15,8,13,6,12,3,4,9,7,2];
$Pairings[22][19] = [undef,4,19,20,1,22,21,16,15,18,17,12,11,14,13,8,7,10,9,2,3,6,5];
$Pairings[22][20] = [undef,3,21,1,22,18,17,19,20,14,13,15,16,10,9,11,12,6,5,7,8,2,4];
$Pairings[22][21] = [undef,2,1,19,21,17,22,20,16,13,18,14,15,9,11,12,8,5,10,3,7,4,6];
$Starts[22][1] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[22][2] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[22][3] = [undef,2,2,1,1,2,2,1,2,1,1,1,2,1,2,2,2,2,1,1,1,1,2];
$Starts[22][4] = [undef,1,1,2,2,1,1,2,1,2,2,2,1,2,1,1,1,1,2,2,2,2,1];
$Starts[22][5] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[22][6] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[22][7] = [undef,2,2,1,1,1,2,2,1,2,2,2,1,2,1,1,1,1,2,1,2,1,2];
$Starts[22][8] = [undef,1,1,2,2,2,1,1,2,1,1,1,2,1,2,2,2,2,1,2,1,2,1];
$Starts[22][9] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[22][10] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[22][11] = [undef,1,1,1,2,2,2,1,2,1,1,2,2,1,1,2,2,2,1,1,1,2,2];
$Starts[22][12] = [undef,2,2,2,1,1,1,2,1,2,2,1,1,2,2,1,1,1,2,2,2,1,1];
$Starts[22][13] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[22][14] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[22][15] = [undef,1,1,2,2,1,1,2,2,2,1,1,1,2,2,1,2,1,1,2,2,2,1];
$Starts[22][16] = [undef,2,2,1,1,2,2,1,1,1,2,2,2,1,1,2,1,2,2,1,1,1,2];
$Starts[22][17] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[22][18] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[22][19] = [undef,2,2,1,1,1,2,2,1,1,2,1,2,1,2,2,1,1,2,1,2,1,2];
$Starts[22][20] = [undef,1,1,2,2,2,1,1,2,2,1,2,1,2,1,1,2,2,1,2,1,2,1];
$Starts[22][21] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];
$Pairings[24][1] = [undef,24,11,9,6,7,4,5,10,3,8,2,14,15,12,13,18,19,16,17,22,23,20,21,1];
$Pairings[24][2] = [undef,23,7,5,9,3,8,2,6,4,12,13,10,11,16,17,14,15,20,21,18,19,24,1,22];
$Pairings[24][3] = [undef,22,13,7,8,9,10,3,4,5,6,15,16,2,18,11,12,21,14,23,24,17,1,19,20];
$Pairings[24][4] = [undef,21,5,4,3,2,9,11,12,6,14,7,8,17,10,19,20,13,22,15,16,1,18,24,23];
$Pairings[24][5] = [undef,20,3,2,10,11,12,9,14,7,4,5,6,19,8,21,22,23,24,13,1,15,16,17,18];
$Pairings[24][6] = [undef,19,15,6,5,4,3,13,9,8,16,17,18,7,20,2,10,11,12,1,14,24,23,22,21];
$Pairings[24][7] = [undef,18,17,8,7,6,5,4,3,10,9,19,20,21,22,23,24,2,1,11,12,13,14,15,16];
$Pairings[24][8] = [undef,17,9,11,12,13,14,15,16,2,18,3,4,5,6,7,8,1,10,24,23,22,21,20,19];
$Pairings[24][9] = [undef,16,19,10,14,15,7,6,18,11,3,9,22,23,4,5,1,24,8,2,21,20,12,13,17];
$Pairings[24][10] = [undef,15,4,13,2,8,16,17,5,12,20,21,9,3,24,1,6,7,23,22,10,11,19,18,14];
$Pairings[24][11] = [undef,14,21,15,16,17,18,19,20,13,22,23,24,9,1,3,4,5,6,7,8,2,10,11,12];
$Pairings[24][12] = [undef,13,6,12,11,10,2,8,7,14,5,4,3,1,9,24,23,22,21,20,19,18,17,16,15];
$Pairings[24][13] = [undef,12,23,14,18,19,11,10,22,15,7,6,1,24,3,9,21,20,4,5,17,16,8,2,13];
$Pairings[24][14] = [undef,11,8,17,13,12,20,21,2,16,24,1,5,4,23,22,9,3,19,18,6,7,15,14,10];
$Pairings[24][15] = [undef,10,18,16,23,21,22,12,19,17,1,24,7,14,13,20,3,9,2,8,15,5,6,4,11];
$Pairings[24][16] = [undef,9,10,24,15,22,13,20,11,1,2,8,23,6,21,4,19,18,17,16,7,14,5,12,3];
$Pairings[24][17] = [undef,8,12,21,22,23,24,14,1,19,11,10,2,20,7,18,17,16,15,9,13,3,4,5,6];
$Pairings[24][18] = [undef,7,24,18,17,16,15,1,13,20,23,22,21,8,19,6,5,4,3,14,9,12,11,10,2];
$Pairings[24][19] = [undef,6,22,23,24,18,1,16,15,21,13,20,19,10,17,8,7,14,5,12,11,9,2,3,4];
$Pairings[24][20] = [undef,5,14,20,19,1,17,24,23,22,21,12,11,18,2,16,15,6,13,4,3,10,9,8,7];
$Pairings[24][21] = [undef,4,20,22,1,24,19,18,21,23,15,14,17,16,11,10,13,12,7,6,2,8,3,9,5];
$Pairings[24][22] = [undef,3,16,1,21,20,23,22,17,24,19,18,13,12,15,14,2,8,11,10,5,4,7,6,9];
$Pairings[24][23] = [undef,2,1,19,20,14,21,23,24,18,17,16,15,22,5,12,11,10,9,3,4,6,13,7,8];
$Starts[24][1] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][2] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][3] = [undef,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[24][4] = [undef,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[24][5] = [undef,2,1,2,1,2,1,1,1,2,2,1,2,2,2,2,1,2,1,1,1,1,2,1,2];
$Starts[24][6] = [undef,1,2,1,2,1,2,2,2,1,1,2,1,1,1,1,2,1,2,2,2,2,1,2,1];
$Starts[24][7] = [undef,2,2,1,2,1,2,1,2,1,2,2,1,2,1,2,1,1,1,1,2,1,2,1,2];
$Starts[24][8] = [undef,1,1,2,1,2,1,2,1,2,1,1,2,1,2,1,2,2,2,2,1,2,1,2,1];
$Starts[24][9] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][10] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][11] = [undef,1,1,1,2,1,2,1,2,1,2,1,2,2,2,2,1,2,1,2,1,2,1,2,1];
$Starts[24][12] = [undef,2,2,2,1,2,1,2,1,2,1,2,1,1,1,1,2,1,2,1,2,1,2,1,2];
$Starts[24][13] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][14] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][15] = [undef,1,1,2,1,2,2,2,1,2,2,2,1,1,2,2,1,1,2,2,1,1,1,2,1];
$Starts[24][16] = [undef,2,2,1,2,1,1,1,2,1,1,1,2,2,1,1,2,2,1,1,2,2,2,1,2];
$Starts[24][17] = [undef,2,1,2,1,2,1,1,1,2,2,1,2,2,2,2,1,2,1,1,1,1,2,1,2];
$Starts[24][18] = [undef,1,2,1,2,1,2,2,2,1,1,2,1,1,1,1,2,1,2,2,2,2,1,2,1];
$Starts[24][19] = [undef,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[24][20] = [undef,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[24][21] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][22] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][23] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];
$Pairings[24][1] = [undef,24,11,9,6,7,4,5,10,3,8,2,14,15,12,13,18,19,16,17,22,23,20,21,1];
$Pairings[24][2] = [undef,23,7,5,9,3,8,2,6,4,12,13,10,11,16,17,14,15,20,21,18,19,24,1,22];
$Pairings[24][3] = [undef,22,13,7,8,9,10,3,4,5,6,15,16,2,18,11,12,21,14,23,24,17,1,19,20];
$Pairings[24][4] = [undef,21,5,4,3,2,9,11,12,6,14,7,8,17,10,19,20,13,22,15,16,1,18,24,23];
$Pairings[24][5] = [undef,20,3,2,10,11,12,9,14,7,4,5,6,19,8,21,22,23,24,13,1,15,16,17,18];
$Pairings[24][6] = [undef,19,15,6,5,4,3,13,9,8,16,17,18,7,20,2,10,11,12,1,14,24,23,22,21];
$Pairings[24][7] = [undef,18,17,8,7,6,5,4,3,10,9,19,20,21,22,23,24,2,1,11,12,13,14,15,16];
$Pairings[24][8] = [undef,17,9,11,12,13,14,15,16,2,18,3,4,5,6,7,8,1,10,24,23,22,21,20,19];
$Pairings[24][9] = [undef,16,19,10,14,15,7,6,18,11,3,9,22,23,4,5,1,24,8,2,21,20,12,13,17];
$Pairings[24][10] = [undef,15,4,13,2,8,16,17,5,12,20,21,9,3,24,1,6,7,23,22,10,11,19,18,14];
$Pairings[24][11] = [undef,14,21,15,16,17,18,19,20,13,22,23,24,9,1,3,4,5,6,7,8,2,10,11,12];
$Pairings[24][12] = [undef,13,6,12,11,10,2,8,7,14,5,4,3,1,9,24,23,22,21,20,19,18,17,16,15];
$Pairings[24][13] = [undef,12,23,14,18,19,11,10,22,15,7,6,1,24,3,9,21,20,4,5,17,16,8,2,13];
$Pairings[24][14] = [undef,11,8,17,13,12,20,21,2,16,24,1,5,4,23,22,9,3,19,18,6,7,15,14,10];
$Pairings[24][15] = [undef,10,18,16,23,21,22,12,19,17,1,24,7,14,13,20,3,9,2,8,15,5,6,4,11];
$Pairings[24][16] = [undef,9,10,24,15,22,13,20,11,1,2,8,23,6,21,4,19,18,17,16,7,14,5,12,3];
$Pairings[24][17] = [undef,8,12,21,22,23,24,14,1,19,11,10,2,20,7,18,17,16,15,9,13,3,4,5,6];
$Pairings[24][18] = [undef,7,24,18,17,16,15,1,13,20,23,22,21,8,19,6,5,4,3,14,9,12,11,10,2];
$Pairings[24][19] = [undef,6,22,23,24,18,1,16,15,21,13,20,19,10,17,8,7,14,5,12,11,9,2,3,4];
$Pairings[24][20] = [undef,5,14,20,19,1,17,24,23,22,21,12,11,18,2,16,15,6,13,4,3,10,9,8,7];
$Pairings[24][21] = [undef,4,20,22,1,24,19,18,21,23,15,14,17,16,11,10,13,12,7,6,2,8,3,9,5];
$Pairings[24][22] = [undef,3,16,1,21,20,23,22,17,24,19,18,13,12,15,14,2,8,11,10,5,4,7,6,9];
$Pairings[24][23] = [undef,2,1,19,20,14,21,23,24,18,17,16,15,22,5,12,11,10,9,3,4,6,13,7,8];
$Starts[24][1] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][2] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][3] = [undef,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[24][4] = [undef,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[24][5] = [undef,2,1,2,1,2,1,1,1,2,2,1,2,2,2,2,1,2,1,1,1,1,2,1,2];
$Starts[24][6] = [undef,1,2,1,2,1,2,2,2,1,1,2,1,1,1,1,2,1,2,2,2,2,1,2,1];
$Starts[24][7] = [undef,2,2,1,2,1,2,1,2,1,2,2,1,2,1,2,1,1,1,1,2,1,2,1,2];
$Starts[24][8] = [undef,1,1,2,1,2,1,2,1,2,1,1,2,1,2,1,2,2,2,2,1,2,1,2,1];
$Starts[24][9] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][10] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][11] = [undef,1,1,1,2,1,2,1,2,1,2,1,2,2,2,2,1,2,1,2,1,2,1,2,1];
$Starts[24][12] = [undef,2,2,2,1,2,1,2,1,2,1,2,1,1,1,1,2,1,2,1,2,1,2,1,2];
$Starts[24][13] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][14] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][15] = [undef,1,1,2,1,2,2,2,1,2,2,2,1,1,2,2,1,1,2,2,1,1,1,2,1];
$Starts[24][16] = [undef,2,2,1,2,1,1,1,2,1,1,1,2,2,1,1,2,2,1,1,2,2,2,1,2];
$Starts[24][17] = [undef,2,1,2,1,2,1,1,1,2,2,1,2,2,2,2,1,2,1,1,1,1,2,1,2];
$Starts[24][18] = [undef,1,2,1,2,1,2,2,2,1,1,2,1,1,1,1,2,1,2,2,2,2,1,2,1];
$Starts[24][19] = [undef,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[24][20] = [undef,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[24][21] = [undef,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[24][22] = [undef,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[24][23] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];

$Pairings[26][1] = [undef,26,4,5,2,3,8,9,6,7,12,13,10,11,15,14,17,16,19,18,21,20,23,22,25,24,1];
$Pairings[26][2] = [undef,25,6,4,3,7,2,5,10,11,8,9,14,16,12,18,13,20,15,22,17,24,19,26,21,1,23];
$Pairings[26][3] = [undef,24,8,7,6,9,4,3,2,5,14,18,15,17,10,12,20,13,11,23,16,25,26,19,1,21,22];
$Pairings[26][4] = [undef,23,3,2,5,4,10,11,12,13,6,7,8,9,16,19,14,21,22,15,24,17,18,1,20,26,25];
$Pairings[26][5] = [undef,22,5,9,8,2,12,13,4,3,15,17,6,7,19,10,21,11,23,14,25,16,1,18,26,20,24];
$Pairings[26][6] = [undef,21,10,6,7,11,3,4,14,16,2,5,18,20,8,22,9,24,12,26,13,1,15,25,17,23,19];
$Pairings[26][7] = [undef,20,12,11,10,6,5,16,15,19,4,3,2,21,22,8,7,25,26,9,1,13,14,24,23,17,18];
$Pairings[26][8] = [undef,19,7,8,9,13,14,2,3,4,18,20,17,5,6,23,24,12,10,1,11,26,25,15,16,22,21];
$Pairings[26][9] = [undef,18,9,13,12,8,15,17,5,2,19,21,4,3,23,6,25,7,1,10,26,11,24,14,22,16,20];
$Pairings[26][10] = [undef,17,14,10,11,16,7,6,18,20,3,4,22,24,2,26,5,1,8,25,9,23,12,21,13,19,15];
$Pairings[26][11] = [undef,16,15,12,14,10,18,22,19,21,5,24,3,25,4,2,1,26,6,8,23,9,7,20,11,13,17];
$Pairings[26][12] = [undef,15,11,16,13,17,9,8,7,6,20,2,23,4,26,1,3,5,25,24,10,22,21,12,19,18,14];
$Pairings[26][13] = [undef,14,13,17,15,12,19,21,9,8,23,25,5,2,1,4,26,3,24,6,22,7,20,10,18,11,16];
$Pairings[26][14] = [undef,13,18,14,16,20,11,10,22,24,7,6,26,1,3,25,4,23,2,21,5,19,8,17,9,15,12];
$Pairings[26][15] = [undef,12,16,15,17,23,22,24,11,10,9,8,1,26,25,3,2,4,21,20,19,18,6,5,7,14,13];
$Pairings[26][16] = [undef,11,19,20,18,14,13,12,21,25,26,1,7,6,5,24,23,22,4,2,3,8,17,16,15,9,10];
$Pairings[26][17] = [undef,10,17,21,19,15,23,25,13,12,1,26,9,8,24,5,22,2,20,4,18,3,16,6,14,7,11];
$Pairings[26][18] = [undef,9,22,18,20,24,16,14,26,1,11,10,25,23,7,21,6,19,3,17,4,15,2,13,5,12,8];
$Pairings[26][19] = [undef,8,20,26,21,18,17,15,1,14,25,23,24,22,9,7,19,6,5,16,2,4,13,11,12,10,3];
$Pairings[26][20] = [undef,7,23,19,22,25,24,1,16,26,13,12,11,10,21,20,8,18,17,3,15,14,4,2,6,5,9];
$Pairings[26][21] = [undef,6,21,25,23,19,1,26,17,15,24,22,13,12,20,9,18,8,16,5,14,2,11,4,10,3,7];
$Pairings[26][22] = [undef,5,26,22,24,1,20,18,25,23,16,14,21,19,11,17,10,15,7,13,6,12,3,9,4,8,2];
$Pairings[26][23] = [undef,4,24,23,1,26,25,19,20,22,21,15,16,18,17,11,12,14,13,7,8,10,9,3,2,6,5];
$Pairings[26][24] = [undef,3,25,1,26,22,21,23,24,18,17,19,20,14,13,16,15,10,9,11,12,6,5,7,8,2,4];
$Pairings[26][25] = [undef,2,1,24,25,21,26,20,23,17,22,16,19,15,18,13,11,9,14,12,7,5,10,8,3,4,6];
$Starts[26][1] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[26][2] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[26][3] = [undef,1,1,2,2,1,1,1,2,2,2,2,1,1,1,2,2,2,1,1,1,1,2,2,2,2,1];
$Starts[26][4] = [undef,2,2,1,1,2,2,2,1,1,1,1,2,2,2,1,1,1,2,2,2,2,1,1,1,1,2];
$Starts[26][5] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[26][6] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[26][7] = [undef,2,2,1,1,1,2,1,2,2,2,2,1,2,1,1,2,2,1,1,1,1,2,2,1,1,2];
$Starts[26][8] = [undef,1,1,2,2,2,1,2,1,1,1,1,2,1,2,2,1,1,2,2,2,2,1,1,2,2,1];
$Starts[26][9] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[26][10] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[26][11] = [undef,1,1,1,2,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,2,1,1,1,2,2];
$Starts[26][12] = [undef,2,2,2,1,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,1,2,2,2,1,1];
$Starts[26][13] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[26][14] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[26][15] = [undef,1,1,2,2,2,1,1,1,1,2,2,2,2,1,1,2,1,1,2,1,2,2,1,2,2,1];
$Starts[26][16] = [undef,2,2,1,1,1,2,2,2,2,1,1,1,1,2,2,1,2,2,1,2,1,1,2,1,1,2];
$Starts[26][17] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[26][18] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[26][19] = [undef,1,1,1,2,1,2,2,2,1,1,1,2,2,2,1,1,1,2,2,2,1,1,2,1,2,2];
$Starts[26][20] = [undef,2,2,2,1,2,1,1,1,2,2,2,1,1,1,2,2,2,1,1,1,2,2,1,2,1,1];
$Starts[26][21] = [undef,1,1,1,2,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2];
$Starts[26][22] = [undef,2,2,2,1,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1,2,2,1,1];
$Starts[26][23] = [undef,2,2,1,1,1,2,1,2,1,2,1,2,1,2,2,1,1,2,2,1,1,2,2,1,1,2];
$Starts[26][24] = [undef,1,1,2,2,2,1,2,1,2,1,2,1,2,1,1,2,2,1,1,2,2,1,1,2,2,1];
$Starts[26][25] = [undef,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4];
=back

=cut

=head1 BUGS

None known.

=cut

1;
