#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ShowPairings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ShowPairings - implement the C<tsh> ShowPairings command

=head1 SYNOPSIS

  my $command = new TSH::Command::ShowPairings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ShowPairings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub ShowAlphaPairings ($$$);
sub ShowBye ($$$);
sub ShowHeader ($$$);
sub ShowUnpaired ($$$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;

  $this->{'help'} = <<'EOF';
Use this command to display the pairings for the specified round
and division.
EOF
  $this->{'names'} = [qw(sp showpairings)];
  $this->{'argtypes'} = [qw(Round Division)];

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round, $dp) = @_;
# my $opt_p = 0;
# if (@$argvp && $argvp->[0] eq '-p') {
#   shift @$argvp;
#   $opt_p = 1;
#   }
  my $round0 = $round-1;

  $dp->CheckAutoPair($this->Processor(), $round);
  if ($round0 > $dp->LastPairedRound0()) {
    $tournament->TellUser('enopryet', $dp->Name(), $round);
    return;
    }
# # a kludge formerly used with make-rr.pl
# if ($opt_p) {
#   print '[';
#   print join(',', map { $_->{'pairings'}[$round0]-1 } @$datap[1..$#$datap]);
#   print "]\n";
#   return 0;
#   }
  my $logp = new TSH::Log($tournament, $dp, 'pairings', $round);
  # sample config line: perl $config'tables{'A'} = [1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10]; # (for 20 2-board tables)
  my $tables = $config::tables{$dp->Name()};
  ShowHeader $tournament, $logp, $tables;
  {
    my @boards;
    my %done;
    my @sorted;
    {
      my $sr0 = $round - 2;
      $sr0 = 0 if $sr0 < 0;
      # shouldn't this be MostScores() - 1?  does it matter? see SMP.pm too.
      $sr0 = $dp->MostScores() if $sr0 > $dp->MostScores()-1;
      $dp->ComputeBoards($sr0, $round0);
      @sorted = TSH::Player::SortByStanding $sr0, $dp->Players();
    }
    # scan the list of players, display byes right away, queue unpaired
    # players for display after this loop, and set up the @boards list
    # to show who's playing at which board
    my @unpaired;
    for my $p (@sorted) {
      next unless $p->Active();
      my $oppid = $p->OpponentID($round0);
      my $pid = $p->ID();
      if (!defined $oppid) {
	push(@unpaired, $p);
        }
      elsif ($oppid == 0) {
	ShowBye $logp, $tables, $p;
        }
      elsif (!$done{$pid}++) {
	my $opp = $dp->Player($oppid);
	if ($pid != $opp->OpponentID($round0)) {
	  my $ooid = $opp->OpponentID($round0);
	  my $oppoppname = $ooid ? $opp->Opponent($round0)->Name() : 'bye';
	  $tournament->TellUser('ebadpair', 
	    $pid, $p->Name(),
	    $oppid, $opp->Name(),
	    $ooid, $oppoppname);
	  }
	elsif ($pid == $oppid) {
	  $tournament->TellUser('eselpair', $pid, $p->Name());
	  }
	else {
	  $done{$oppid}++;
	  my $ob = $opp->Board($round0) || 0;
	  my $pb = $p->Board($round0) || 0;
	  if ($ob != $pb) {
	    $tournament->TellUser('eboarddiff', $p->Name(), $opp->Name());
	    }
	  $boards[$pb] = [$p, $opp];
	  }
        }
      } # for $p
    # display the queued up unpaired players
    for my $p (@unpaired) {
      ShowUnpaired $logp, $tables, $p;
      }
    # display who's at each board
    {
      # $tables is zero-indexed, but players prefer one-indexed tables
      for my $board1 (1..$#boards) {
	my $board = $board1 - 1;
	my $boardp = $boards[$board1];
	# a board might be empty if a pairing was deactivated after
	# boards were assigned, or if all seating is reserved
	next unless $boardp; 
	$logp->Write('', '<tr>');
        $logp->Write(sprintf(" $config'table_format  ", $tables->[$board]),
	  "<td class=table>$tables->[$board]</td>")
	  if defined $tables;
	my ($p1, $p2) = @$boardp;
	if ($p1->Board($round0)) {
	  $p2->Board($round0, $p1->Board($round0)); # just in case
	  }
	else {
	  $p1->Board($round0, $board1);
	  $p2->Board($round0, $board1);
	  }
	my $vs = $dp->FormatPairing($round0, $p1->ID());
	unless ($vs) {
	  my $pname = $p1->Name();
	  $logp->Write("Lost track of $pname", "Lost track of $pname");
	  next;
	  }
	my $vshtml = $vs;
	$vshtml =~ s/\*(?:starts|draws)\*/<span class=starts>$&<\/span>/;
        $logp->Write(sprintf(" %3d  %s.\n", $board1, $vs), '<td class=board>' . ($board1) . "</td><td class=name>$vshtml</td></tr>\n"
	  );
	}
    }
  $logp->Close();
  $this->ShowAlphaPairings($dp, $round);
  }
  $dp->Update(); # in case table numbers were changed
  }

=item ShowAlphaPairings;

Create an HTML file listing pairings in alphabetical order by player name.

=cut

sub ShowAlphaPairings ($$$) {
  my $this = shift;
  my $dp = shift;
  my $round = shift;
  my $round0 = $round - 1;
  my $tables = $config::tables{$dp->Name()};
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $hasphotos = $config->Value('player_photos');
  my $page_break = $config->Value('alpha_pair_page_break');

  my @entries;
  my @headless;
  for my $p (sort { $a->Name() cmp $b->Name(); } $dp->Players()) {
    next unless $p->Active();
    my $oppid = $p->OpponentID($round0, 'undef for unpaired');
    my $entry = '';
    my $board = $p->Board($round0);
#   die "No board for player $p->{'name'} in round $round0+1\n";
    if (defined $tables) {
      if ($oppid) {
	$entry .= "<td class=table>$tables->[$board-1]</td>";
        }
      else {
	$entry .= "<td class=table>&nbsp;</td>";
        }
      }
    if ($oppid) { $entry .= "<td class=board>$board</td>"; }
    else { $entry .= "<td></td>"; }
    $entry .= "<td>" . (TSH::Utility::TaggedName $p) . "</td>";
    if (!defined $oppid) {
      $entry .= "<td></td><td>Unpaired</td>";
      }
    elsif (!$oppid) {
      $entry .= "<td></td><td>Bye</td>";
      }
    else {
      my $opp = $dp->Player($oppid);
      my $vs = $dp->FormatPairing($round0, $p->ID(), 'brief');
      $vs =~ s/\..*/./;
      my $pic = '';
      if ($hasphotos) {
#	warn join("\n", keys %{$tournament->{'bios'}});
#	die $opp->Name();
	if (my $url = $opp->PhotoURL()) {
	  $pic = qq(<img align=left height=36 width=36 src="$url">);
	  }
	else {
	  push(@headless, $opp);
	  }
        }
      $entry .= "<td><span class=starts>$vs</span></td><td>$pic" . (TSH::Utility::TaggedName $opp) . "</td>";
      }
    push(@entries, $entry);
    }
  if ($hasphotos && @headless) {
    my $fn = $config->MakeRootPath($dp->Name() . '-headless.txt');
    if (open my $fh, ">$fn") {
      print $fh map { $_->Board($round0) . ' ' . $_->Name() . "\n" } 
        sort { $a->Board($round0) <=> $b->Board($round0) } @headless;
      close $fh;
      }
    }
  my $logp = new TSH::Log($tournament, $dp, 'alpha-pairings', $round, {
    'noconsole' => 1,
    'title' => 'Division '.$dp->Name()." Round $round Alphabetic Pairings"});
  my $html = '';
  my $headings = '';
  $headings .= "<th class=table>$config::table_title</th>" if defined $tables;
  $headings .= '<th class=board>Board</th><th class=name>Player</th><th></th><th class=name>Opponent</th>';
# if ($hasphotos < @entries / 5) { # too few photos looks silly
#   for my $entry (@entries) { $entry =~ s/<img.*?>//; }
#   $hasphotos = 0;
#   }
  if ($hasphotos || ($page_break && @entries > $page_break)) {
    $html .= "<tr>$headings$headings</tr>";
    push(@entries, '') if @entries % 2;
    my $half = @entries / 2;
    for my $i (0..$half-1) {
      my $break = '';
      if ($page_break && ($i+1) % $page_break == 0) {
	$break = qq(</table><table class=alpha_pairings align=center cellspacing=0 style="page-break-before:always"><tr>$headings$headings</tr>);
        }
      $entries[$i] =~ s/(.*)<td>/$1<td style="border-right:1px solid black">/;
      $html .= qq($break<tr class=double style="page-break-inside:avoid">$entries[$i]$entries[$i+$half]</tr>);
      }
    }
  else {
    $html .= "<tr>$headings</tr>";
    $html .= join('', map { qq(<tr style="page-break-inside:avoid">$_</tr>); } @entries);
    }
  $logp->Write('', $html);
  $logp->Close();
  }

=item ShowBye $log, $tables, $p

Used internally to show a bye player

=cut

sub ShowBye ($$$) {
  my $logp = shift;
  my $tables = shift;
  my $p = shift;

  $logp->Write('', '<tr>');
  $logp->Write(sprintf("  $config'table_format ", ''), 
    '<td class=notable>&nbsp;</td>') if defined $tables;
  $logp->Write(sprintf("      %s: BYE.\n", ($p->TaggedName())),
    '<td class=bye>BYE</td><td class=name>' . 
    ($p->TaggedName()) . '</td>');
  $logp->Write('', '</tr>');
  }

=item ShowHeader $tournament, $log, $tables

Used internally to show the header of the pairings listings

=cut

sub ShowHeader ($$$) {
  my $tournament = shift;
  my $logp = shift;
  my $tables = shift;

  $logp->Write('', '<tr class=top1>');
  if (defined $tables) {
    $logp->Write("$config::table_title ", "<th class=table>$config::table_title");
    my $shortage = length(sprintf($config::table_format, ''))-3;
    if ($shortage < 0) {
      $tournament->TellUser('wsmtfmt');
      $config::table_format = '%3s';
      $shortage = 0;
      }
    $logp->Write((' ' x $shortage), '</th>');
    }
  $logp->Write("Board Players\n", <<'EOF');
<th class=board>Board</th>
<th class=name>Who Plays Whom</th>
</tr>
EOF
  }

=item ShowUnpaired $log, $tables, $p

Used internally to show an unpaired player

=cut

sub ShowUnpaired ($$$) {
  my $logp = shift;
  my $tables = shift;
  my $p = shift;

  $logp->Write('', '<tr>');
  $logp->Write(sprintf("  $config'table_format ", ''), 
    '<td class=notable>&nbsp;</td>') if defined $tables;
  $logp->Write(sprintf("      %s: UNPAIRED.\n", ($p->TaggedName())),
    '<td class=unpaired>UNPAIRED</td><td class=name>' . 
    ($p->TaggedName()) . '</td>');
  $logp->Write('', '</tr>');
  }

=back

=cut

=head1 BUGS

A reasonable guess should be made when there are not enough tables
configured to accommodate the boards needed.

enopryet should be reported even when only inactive players are paired

Should use new TSH::Log table code.

=cut

1;
