#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Utility;

use strict;
use warnings;

use threads::shared;

use File::Spec;
use File::Temp qw(tempfile);

our(@ISA) = 'Exporter';
our(@EXPORT_OK) = qw(Debug DebugOn DebugOff DebugDumpPairings
  Error OpenFile);

=pod

=head1 NAME

TSH::Utility- miscellaneous Perl utilities

=head1 SYNOPSIS

  sub new { Util::new(@_); }
  PrintColour $colour_name, $text;
  Error "You goofed.\n";
  print TaggedName($player);
  print Wrap($indent, @text);
  $old_value = GetOrSet($object, $field, $new_value);
  $current_value = GetOrSet($object, $field);
  DebugOn($code);
  DebugOff($code);
  Debug($code, $format, @args);
  DebugDumpPairings($code, $round0, $psp);
  Prompt('subtsh>');
  $fh = OpenFile(">", $dir1, $dir2, ..., $file);
  ReplaceFile($filename, $data);

  DoRanked($list $comparator, $selector, $actor);
  TruncateFileHandle($fh, $length);
  ShareSafely($ref);
  SpliceSafely(@array, $offset, $length, $list);

=head1 ABSTRACT

This library contains miscellaneous bits of code used by in more than 
one C<tsh> source file.

=cut

sub Colour ($$);
sub Debug ($$@);
sub DebugDumpPairings($$$);
sub DebugOff ($);
sub DebugOn ($);
sub Error ($);
sub GetColourData ($$);
sub GetOrSet ($@);
sub new (@);
sub OpenFile ($@);
sub PrintColour ($$);
sub Prompt ($);
sub TaggedName ($);
sub TruncateFileHandle ($$);
sub Wrap ($@);

our ($BG_BLUE, $BG_WHITE, $FG_BLUE, $FG_GRAY, $FG_GREEN, $FG_RED, $FG_YELLOW, $FG_WHITE);
my %Colours = (
  'red' => { 'ansi' => "\e[31m", 'win' => $FG_RED },
  'green' => { 'ansi' => "\e[32m", 'win' => $FG_GREEN },
  'blue' => { 'ansi' => "\e[34m", 'win' => ($FG_BLUE||0)|($BG_WHITE||0) },
  'yellow on blue' => { 'ansi' => "\e[44;33m", 'win' => (($FG_YELLOW||0) | ($BG_BLUE||0)) },
  'plain' => { 'ansi' => "\e[30;47;0m", 'win' => $FG_WHITE },
  );
my %gComplained;
my %gDebug;
my $gDebugFH;
my $WinConsole; 
BEGIN { 
  if ($^O eq 'MSWin32') { 
    require Win32::Console; import Win32::Console;
    $WinConsole = new Win32::Console &STD_OUTPUT_HANDLE;
    $WinConsole->Title("tsh");
    };
  }

=head1 DESCRIPTION

=over 4

=cut
 
=item Colour($colour, $text)

Add escape sequences to add colour to text.
Deprecated in favour of PrintColour because Windows console
doesn't use escape sequences.

=cut

sub Colour ($$) {
  die "deprecated function TSH::Utility::Colour";
  my $colour = shift;
  my $text = shift;
  return $text if (defined $config'colour) && $config'colour =~ /^(?:no|0)$/i;
  return $text if $^O eq 'MSWin32' &&
    ((!defined $config'colour) || $config'colour !~ /^yes$/i);
  my $escape = $Colours{$colour}{'ansi'};
  die "Unknown colour: $colour\n" unless defined $escape;
  return "$escape$text$Colours{'plain'}{'ansi'}";
  }

=item Debug($code, $format, @args)

Display and log debug text.

=cut

sub Debug ($$@) {
  my $code = shift;
  return unless $gDebug{$code};
  my $format = shift;
  my $s = "[$code] " .  sprintf($format, @_);
  $s .= "\n" unless $s =~ /\n$/;
  print "Debug: " . $s;
  unless ($gDebugFH) {
    $gDebugFH = OpenFile ">", "debug.txt";
    }
  if ($gDebugFH) {
    print $gDebugFH $s;
    }
  }

=item DebugDumpPairings($code, $round0, $psp)

Dump the $round0 pairings for the players in $psp if DebugOn($code);

=cut

sub DebugDumpPairings($$$) {
  my $code = shift;
  my $round0 = shift;
  my $psp = shift;
  return unless $gDebug{$code};
  Debug $code, 'Pairings:';
  my %done;
  for my $i (0..$#$psp) {
    my $p = $psp->[$i];
    my $opp = $p->Opponent(-1);
    next if $done{$p->ID()};
    $done{$opp->ID()}++;
    Debug $code, '... %s vs %s.', $p->TaggedName(), $opp->TaggedName();
    }
  }

=item DebugOff($code)

Turn off debugging of type $code.

=cut

sub DebugOff ($) {
  my $code = shift;
  $gDebug{$code} = 0;
  }

=item DebugOn($code)

Turn on debugging of type $code.

=cut

sub DebugOn ($) {
  my $code = shift;
  $gDebug{$code} = 1;
  }

=item DoRanked($list, $comparator, $selector, $actor);

Sort @$list privately by &$comparator, then iterate calling 
&$actor($list->[$i], $rank), where $rank starts at 0 and is set to
$i whenever the value of &$selector($list->[$i]), a list of scalars, changes.


=cut

sub DoRanked ($$$$) {
  my $listp = shift;
  my $comparatorp = shift;
  my $selectorp = shift;
  my $actorp = shift;

  my (@sorted) = sort $comparatorp @$listp;
  my @lastvalue;
  my $rank = 0;
  for my $i (0..$#sorted) {
    my $item = $sorted[$i];
    my (@newvalue) = &$selectorp($item);
    my $changed = 0;
    if (@lastvalue != @newvalue) { $changed = 1; }
    else {
      for my $j (0..$#lastvalue) {
	if ($newvalue[$j] != $lastvalue[$j]) {
	  $changed = 1;
	  last;
	  }
        }
      }
    if ($changed) {
      $rank = $i;
      @lastvalue = @newvalue;
      }
    &$actorp($sorted[$i], $rank);
    }
  }

=item Error($text)

Display an error message.

=cut

sub Error ($) {
  my $message = shift;
  $message .= "\n" unless $message =~ /\n$/;
  PrintColour 'red', $message;
  }

=item $s = GetColourData $os, $colour;

Return OS-dependent string corresponding to colour name.

=cut

sub GetColourData ($$) {
  my $os = shift;
  my $colour = shift;
  my $s = $Colours{$colour}{$os};
  unless (defined $colour) {
    warn "Unexpected error, contact John Chew. os=$os, colour=$colour.\n"
      unless $gComplained{'badcolour'}++;
    }
  return $s;
  }

=item GetOrSet()

Boilerplate code for get/set methods.

=cut

sub GetOrSet ($@) {
  my $field = shift;
  my $object = shift;
  my $value = shift;
  my $old = $object->{$field};
  if (defined $value) {
    $object->{$field} = $value;
    }
  return $old;
  }

=item new()

Boilerplate code for creating Perl objects.

=cut

sub new (@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $this = { };
  bless($this, $class);
  $this->initialise(@_);
  return $this;
  }

=item newshared()

Boilerplate code for creating Perl objects.

=cut

sub newshared (@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  # combining the following two lines fails in 5.8.0, assigning $this = 1
  my $this = {};
  &share($this);
  bless($this, $class);
  $this->initialise(@_);
  return $this;
  }

=item OpenFile($mode, $dir1,...,$dirn, $file)

Open a file as indicated.  Use this for consistency rather
than the various "standard" ways of calling open().

=cut

sub OpenFile ($@) {
  my $fh;
  my $mode = shift;
  my $fn = File::Spec->join(@_);
  return undef unless open($fh, $mode, $fn);
  return $fh;
  }

=item OpenFileThreadShared($mode, $dir1,...,$dirn, $file)

Open a file as indicated.  Use this version for any file whose
handle needs to be shared: typeglobs are not yet shareable, so
we resort to symbolic references.  TODO: Should be replaced by a call
to OpenFile when threads::shared is fully developed.

=cut

our $HANDLE_NUMBER = 0;

sub OpenFileThreadShared ($@) {
  my $fh = "TSH::Utility::HANDLE$HANDLE_NUMBER";
  my $mode = shift;
  my $fn = File::Spec->join(@_);
  {
    no strict "refs";
    return undef unless open($fh, $mode, $fn);
  }
  return $fh;
  }

=item PrintColour($colour, $text)

Print text in specified colour.

=cut

sub PrintColour ($$) {
  my $colour = shift;
  my $text = shift;
  if ((defined $config'colour) && $config'colour =~ /^(?:no|0)$/i) {
    print $text;
    }
  elsif ($^O eq 'MSWin32') { 
#   if ((!defined $config'colour) || $config'colour !~ /^(?:yes|1)$/i) {
#     print $text;
#     }
#   else 
      {
      my $attr = GetColourData 'win', $colour;
      $WinConsole->Attr($attr) if defined $attr;
      print $text;
      $attr = GetColourData 'win', 'plain';
      $WinConsole->Attr($attr) if defined $attr;
      }
    }
  else {
    my $escape = GetColourData 'ansi', $colour;
    print $escape if defined $escape;
    print $text;
    $escape = GetColourData 'ansi', 'plain';
    print $escape if defined $escape;
    }
  }

=item Prompt($text)

Displays the given prompt in blue, followed by a plain space.

=cut

sub Prompt ($) {
  my $text = shift;
  PrintColour 'blue', $text;
  print ' ';
  }

=item $success = ReplaceFile($filename, $data)

Carefully replace the contents of the file called $filename with
$data, by creating a temporary file in the same directory, then
renaming it, so as to prevent another process from reading garbled
data in the middle of an update.  Returns boolean indicating success.

=cut

sub ReplaceFile ($$) {
  my $filename = shift;
  my $data = shift;
  my (@stat) = stat $filename;

  my ($volume, $dirs, $file) = File::Spec->splitpath($filename);
  my $dir = File::Spec->catpath($volume, $dirs,'');
  my ($newfh, $newfn) = tempfile(DIR => $dir, UNLINK => 0);
  print $newfh $data or return 0;
  close $newfh or return 0;
  chmod $stat[2], $newfn if @stat;
  rename $newfn, $filename or return 0;
  return 1;
  }

=item $newref = ShareSafely($oldref)

Returns a reference to a thread-shared version of a reference.
At present, using threads::shared::share() on a reference clears
its contents.

=cut

sub ShareSafely($) {
  my $ref = shift;
  my $type = ref($ref);
  if ($type eq '') {
    my $shared : shared = $ref;
    return $shared;
    }
  elsif ($type eq 'ARRAY') {
    my @shared : shared = map { ShareSafely($_) } @$ref;
    return \@shared;
    }
  elsif ($type eq 'HASH') {
    my %shared : shared = map { ($_, ShareSafely($ref->{$_})) } keys %$ref;
    return \%shared;
    }
  else {
    die "ShareSafely: don't yet support reference type $type";
    }
  }

=item @removed = SpliceSafely(@array, $offset, $length, @list);

Does what the built-in splice does, but works even when applied
to a shared object in a threaded environment.  Not particularly
efficient, and should be replaced as soon as splicing works properly
with threads.  Note that unlike in the calling sequence for splice(),
$offset and $length are not optional.

=cut

sub SpliceSafely(\@$$@) {
  my $arrayp = shift;
  my $offset = shift;
  my $length = shift;
  my @list = @_;
  
  my @copy = @$arrayp;
  my @removed = splice(@copy, $offset, $length, @list);
  while (@$arrayp > @copy) { pop @$arrayp; }
  for my $i ($offset..$#copy) {
    $arrayp->[$i] = $copy[$i];
    }
  while (@$arrayp < @copy) {
    push(@$arrayp, $copy[@$arrayp]);
    }
  return @removed;
  }

=item TaggedName($player)

Safe wrapper for TSH::Player::TaggedName

=cut

sub TaggedName ($) {
  my $p = shift;
  if (UNIVERSAL::isa($p,'TSH::Player')) {
    return $p->TaggedName();
    }
  else {
    return 'nobody';
    }
  }

=item TruncateFileHandle($filehandle, $length);

Works like the built-in truncate(), but assumes its first argument
is a symbolic reference rather than a filename, if it's not a
file handle.  Part of a collection of routines to work around
the inability of the threads::shared module to share fileglobs.

=cut

sub TruncateFileHandle ($$) {
  my $fh = shift;
  my $length = shift;
  unless (ref($fh)) {
    no strict 'refs';
    $fh = *{$fh};
    }
  return truncate $fh, $length;
  }

=item Wrap($indent, @text)

Justify text right-ragged.

=cut

sub Wrap ($@) {
  my $indent = shift;
  my $indent_space = ' ' x $indent;
  my $width = 78;
  my $s = $indent_space;
  my $hpos = $indent;
  my $atsol = 1;
  while (@_) {
    my (@words) = split(/\s+/, shift);
    for my $word (@words) {
      $word =~ s/\.$/. / unless $word =~ /\../;
      my $l = length($word);
      if ($hpos + $l > $width) {
	$s .= "\n$indent_space$word";
	$hpos = $indent + $l;
        }
      else {
	unless ($atsol) {
	  $word = " $word";
	  $l++;
	  }
	$s .= $word;
	$hpos += $l;
	$atsol = 0;
        }
      }
    }
  $s .= "\n";
  return $s;
  }

=back

=cut

=head1 BUGS

C<Colour>,
C<PrintColour>
and
C<Wrap>
should be in something called 
C<TSH::TTY>.

C<Wrap> should dynamically determine the width of the
current console.

=cut

1;
