#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package ABSP;

use strict;
use warnings;

=pod

=head1 NAME

ABSP - Association of British Scrabble Players support class

=head1 SYNOPSIS

  use ABSP;
  my @players = (undef, 
    { 'rating' => 150, 'scores' => [400, 450,  50], 'pairings' => [2, 3, 0] },
    { 'rating' => 140, 'scores' => [350,  50, 350], 'pairings' => [1, 0, 3] },
    { 'rating' => 130, 'scores' => [ 50, 300, 300], 'pairings' => [0, 1, 2] },
    );
  ABSP::CalculateRatings(@players);
  print "Player #1's new rating: $players[1]{'newr'}\n";

  
=head1 ABSTRACT

This Perl library provides support for ABSP-related calculations.

=head1 DESCRIPTION

=over 4

=cut

sub CalculateRating(\@$$);
sub CalculateRatings(\@);

=item ABSP::CalculateRating(@players, $player_number, $phase)

Updates one player's rating, called by CalculateRatings.
In Phase 1, unrated player ratings are estimated.
In Phase 2, unrated player ratings are recalculated using Phase 1 ratings.
In Phase 3, rated player ratings are updated.

=cut

sub CalculateRating(\@$$) {
  my $datap = shift;
  my $id = shift;
  my $phase = shift;
  my $p = $datap->[$id];
  my $mr = $p->Rating();
  my $newr = 0;
  my $ngames = 0;
  for my $r0 (0..$#{$p->{'scores'}}) {
    my $oppid = $p->{'pairings'}[$r0];
    next unless $oppid;
    my $opp = $datap->[$oppid];
    my $ms = $p->{'scores'}[$r0];
    next unless $ms;
    my $os = $opp->{'scores'}[$r0];
    next unless $os;
    my $oppr;
    if ($phase == 1) {
      $oppr = $opp->Rating();
      }
    elsif ($phase == 2) {
      $oppr = $opp->Rating() || $opp->{'tmpr'};
      }
    elsif ($phase == 3) {
      $oppr = $opp->Rating() || $opp->{'newr'};
      }
    next unless $oppr;
    $ngames++;
    if ($mr) {
      if ($opp->Rating()) {
	  if ($oppr > $mr + 40) { $oppr = $mr + 40; }
	  elsif ($oppr < $mr - 40) { $oppr = $mr - 40; }
	}
      # TODO: implement novice ratings enhancement: a player cannot
      # have an effective rating of below 80 until they have finished
      # playing the tournament containing their 15th game.
      }
    $newr += $oppr;
    $newr += 50 * ($ms <=> $os);
    }
  if ($ngames) {
    $newr /= $ngames;
    }
  else {
    $newr = $mr;
    }
  if ($p->Rating()) {
    if ($newr < 40) { $newr = 40; }
    }
  else {
    if ($newr < 80) { $newr = 80; }
    }

  $newr = int($newr + 0.5);
  if ($phase == 1) {
    $p->{'tmpr'} = $newr;
    }
  else {
    $p->NewRating($newr);
    }
  }

=item ABSP::CalculateRatings(@players)

Updates a list of hashes of player information to include tournament ratings.

=cut

sub CalculateRatings(\@) {
  my $datap = shift;

  # calculate initial ratings for unrated players
  for my $id (1..$#$datap) {
    CalculateRating @$datap, $id, 1 unless $datap->[$id]->Rating();
    }
  # calculate final ratings for unrated players
  for my $id (1..$#$datap) {
    CalculateRating @$datap, $id, 2 unless $datap->[$id]->Rating();
    }
  # calculate initial ratings for previously rated players
  for my $id (1..$#$datap) {
    CalculateRating @$datap, $id, 3 if $datap->[$id]->Rating();
    }
  }

=back

=cut

1;
