#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::UPSETs;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::UPSETs - implement the C<tsh> UPSETs command

=head1 SYNOPSIS

  my $command = new TSH::Command::UPSETs;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::UPSETs is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the UPSETs command to list biggest ratings upsets in a division.
EOF
  $this->{'names'} = [qw(upset upsets)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;

  my (@unrated_upsets);
  my (@upsets);
  for my $p ($dp->Players()) {
    for my $r (1..$p->CountScores()) {
      my $r0 = $r-1;
      my $ms = $p->Score($r0);
      my $oid = $p->OpponentID($r0);
      next unless $oid;
      my $os = $p->OpponentScore($r0);
      next unless $ms > $os;
      my $o = $p->Opponent($r0);
      my $myrating = $p->Rating();
      my $diff = $o->Rating() - $myrating;
      next unless $diff > 0;
      my $target = $myrating ? \@upsets : \@unrated_upsets;
      push(@$target, [$diff, $r0, $o, $p]);
      if (@$target > 100) {
	@$target = (sort { $b->[0] <=> $a->[0] } @$target)[0..19];
        }
      }
    }
  @unrated_upsets = (sort { $b->[0] <=> $a->[0] } @unrated_upsets);
  splice(@unrated_upsets, 5) if @unrated_upsets > 5;
  @upsets = (sort { $b->[0] <=> $a->[0] } @upsets);
  splice(@upsets, 20) if @upsets > 20;

  my $logp = new TSH::Log($tournament, $dp, 'upsets');
  $logp->ColumnClasses([qw(rating rating rating round score score name name)]);
  $logp->ColumnTitles({
    'text' => [qw(Diff Rtg1 Rtg2 Rd Sc1 Sc2 Player1 Player2)],
    'html' => [
      'Rating<br>Difference',
      'Rating<br>#1','Rating<br>#2',
      'Round',
      'Score<br>#1','Score<br>#2',
      'Player<br>#1','Player<br>#2',
      ],
    });
  for my $upset (@upsets, @unrated_upsets) {
    my ($diff, $r0, @p) = @$upset;
    my (@rating, @score, @name);
    for my $i (0..1) { 
      my $p = $p[$i];
      $rating[$i] = $p->Rating();
      $score[$i] = $p->Score($r0);
      $name[$i] = $p->TaggedName();
      }
    $logp->WriteRow([
      $diff, $rating[0], $rating[1], $r0+1, $score[0], $score[1], 
      $name[0], $name[1]]);
    }
  $logp->Close();
  }

=back

=cut

1;
