#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::showWallChart;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::showWallChart - implement the C<tsh> showWallChart command

=head1 SYNOPSIS

  my $command = new TSH::Command::showWallChart;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::showWallChart is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display a version of the self-posting
wall chart suitable for checking what the players have
posted themselves.  If you specify a round number, the
wall chart starts with that round; if you don't, it starts
at round 1.
EOF
  $this->{'names'} = [qw(wc showwallchart)];
  $this->{'argtypes'} = [qw(OptionalRound Divisions)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();

  my $from = 1;
  if (ref($_[0]) eq '') {
    $from = shift;
    }
  my $from1 = $from;
  $from--;
  for my $dp (@_) {
    my $logp = new TSH::Log($tournament, $dp, 'wallchart', undef,
      { 'title' => 'Division ' . $dp->Name() . ' Wall Chart' },
      );
    my (@classes) = qw(name);
    my (@html_title) = qw(Player);
    my $max_rounds = $dp->MaxRound0();
    if (defined $max_rounds) { $max_rounds++; }
    else { $max_rounds = $dp->MostScores(); }
    for my $r ($from1..$max_rounds) {
      push(@html_title, "Round $r");
      push(@classes, 'round');
      }
    $logp->ColumnClasses(\@classes);
    $logp->ColumnTitles( {
      'text' => [],
      'html' => \@html_title,
      });
    for my $p ($dp->Players()) {
      my (@classes2) = @classes = qw(name);
      my (@classesh) = qw(name);
      my $pname = $p->PrettyName();
      my (@html) = ($p->ID() . '. ' . $pname);
      my (@text) = ($pname);
      my (@text2) = ('');
      my $spread = 0;
      my $wins = 0;
      my $losses = 0;
      my $cols = 0;
      my $id = $p->ID();
      for my $r (1..$p->CountScores()) {
	my $r0 = $r - 1;
        my $oppid = $p->OpponentID($r0);
	my $opp = $p->Opponent($r0);
	my $os = ($p->OpponentScore($r0) || 0);
	my $ms = $p->Score($r0);
        my $thisSpread = ($ms||0) - $os;
        $spread += $thisSpread;
	if (defined $ms) {
	  my $wtl = (1 + ($thisSpread <=> 0))/2;
	  $wins += $wtl;
	  $losses += 1-$wtl;
	  }
	if ($r0 >= $from) {
	  push(@text, sprintf("%.1f", $wins));
          push(@text2, sprintf("%+d", $spread));
	  push(@classes, 'wins');
	  push(@classes2, 'spread');

	  my $oppinfo = (defined $oppid) 
	    ? $dp->FormatPairing($r0, $id, 'brief') : '';
	  if (!defined $oppid) {
	    $oppinfo = qq(<div class=empty>$oppinfo</div>);
	    }
	  elsif ($oppid) { 
	    $oppinfo = qq(<div class=opp><a href="#$oppid.$r" name="$id.$r">$oppinfo</a></div>); 
	    }
	  else {
	    $oppinfo = qq(<div class=bye>$oppinfo</div>);
	    }
	  push(@classesh, (defined $oppid) ? $oppid ? $thisSpread > 0 ? 'win' :
	    $thisSpread < 0 ? 'loss' : 'tie' : 'bye' : 'empty');
	  push(@html, sprintf(qq(<div class=score>%d-%d</div><div class=pair>%s</div><div class=record>%.1f-%.1f</div><div class=spread>%+d = %+d</div>),
	    ($ms||0), $os,
	    $oppinfo,
	    $wins, $losses,
	    $thisSpread, $spread,
	    ));
	  }
        unless ((1+$cols-$from) % 10) {
	  if ($r0 >= $from) {
	    $logp->ColumnClasses(\@classes);
	    $logp->WriteRow(\@text, []);
	    $logp->ColumnClasses(\@classes2);
	    $logp->WriteRow(\@text2, []);
	    @text = ('');
	    @text2= ('');
	    @classes = @classes2 = qw(name);
	    }
          }
	$cols++;
        } # for $r0
      if (@text > 1) {
	$logp->ColumnClasses(\@classes);
	$logp->WriteRow(\@text, []);
	$logp->ColumnClasses(\@classes2);
	$logp->WriteRow(\@text2, []);
        }
      $logp->ColumnClasses(\@classesh);
      $logp->WriteRow([], \@html);
      } # for my $p
    $logp->Close();
    }
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;
