#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::PairingCommand;

use strict;
use warnings;
use threads::shared;
use TSH::Utility;

use TSH::Utility qw(Debug DebugOn DebugOff DebugDumpPairings);

our (@ISA) = qw(TSH::Command);

DebugOn('GIB');

=pod

=head1 NAME

TSH::PairingCommand - abstraction of a C<tsh> pairing command

=head1 SYNOPSIS

This class supports features common to pairings commands, and
is also used to test class membership.
See its parent class C<TSH::Command> for usage.
  
=head1 ABSTRACT

$setup = $command->SetupForPairing(%options);
$command->TidyAfterPairing($dp);

=cut

sub CalculateBPFs ($);
sub CalculateBestPossibleFinish ($$);
sub CountGibsons ($$);
sub FlightCapDefault ($);
sub FlightCapNSC ($);
sub MaxSpread ($);
sub PairAllGibsons ($$$$);
sub PairEvenGibsons ($$);
sub PairOneGibson ($$$$);
sub SetupForPairings ($%);
sub TidyAfterPairing ($$);

=head1 DESCRIPTION

=over 4

=item CalculateBPFs($psp)

Calculate best possible finishes for everyone in @$psp, which
must be sorted according to current standing.

=cut

sub CalculateBPFs($) {
  my $psp = shift;

  my $toprank = $psp->[0]->RoundRank($psp->[0]->CountScores()-1);
  for my $i (0..$#$psp) {
    my $r = CalculateBestPossibleFinish $psp, $i;
    $psp->[$i]->MaxRank($r+$toprank-1);
    }
  }

=item CalculateBestPossibleFinish($psp, $index);

Calculate what the highest possible finishing rank is for the
${index}th player in $psp.  
The rank is relative to the players in @$psp.

=cut

sub CalculateBestPossibleFinish ($$) {
  my $psp = shift;
  my $index = shift;

  return undef if $index > $#$psp;
  return 1 if @$psp == 1;
  my $me = $psp->[$index];
  my $dp = $me->Division();
  my $config = $dp->Tournament()->Config();
  my $max_rounds = $dp->MaxRound0() + 1;
  my $bye_spread = $config->Value('bye_spread');
  $bye_spread = 50 unless defined $bye_spread; # should not happen

# DebugOn('CBPF');
  # set scratch variables
  for my $p (@$psp) {
    # current wins
    $p->{'xw'} = $p->Wins();
    # current spread
    $p->{'xs'} = $p->Spread();
    # maximum spread per round, number of games left to play
    $p->{'xms'} = MaxSpread ($p->{'xleft'} = $p->UnscoredGames());
    }
  my $nrounds = 0;
  for my $p (@$psp) { $nrounds = $p->{'xleft'} if $nrounds < $p->{'xleft'} }
# Debug 'CBPF', "np=%d nrounds=%d.", $#$psp+1, $nrounds;
  # if no games left to play, rank won't change
  return 1+$index if $nrounds == 0;

  my $my_final_wins = $me->{'xw'} + $me->{'xleft'};
  my $my_final_spread = $me->{'xs'} + $me->{'xleft'} * $me->{'xms'};
  my @ps;
  # assign wins as favourably to me as possible
  for my $r (1..$nrounds) {
#   Debug 'CBPF', 'Round %d', $max_rounds-$nrounds+$r;
    # everyone ranked up to the cap must play each other
    my $rounds_left = $nrounds - $r + 1; # a rough approximation
    my $cap;
    {
      no strict 'refs';
      $cap = &$config::flight_cap($rounds_left) - 1; # 0-based
    }
    $cap = $#$psp if $cap > $#$psp;
    my $r0 = $max_rounds - $rounds_left; # +1 -1 = 0
    @ps = sort { $b->{'xw'}<=>$a->{'xw'} || $b->{'xs'}<=>$a->{'xs'}||($a eq $me ? -1 : $b eq $me ? 1 : 0)} @$psp;
    # divide players into five groups: preferred winners above and below
    # the cap line, losers (ditto) and me
    my $myrank = undef;
    my @hi_winners;
    my @lo_winners;
    my @hi_losers;
    my @lo_losers;
    for my $i (0..$#$psp) { 
      my $p = $ps[$i];
      if ($me eq $p) { $myrank = $i; next; } 
      next if defined $p->Score($r0); # already has a score this round
      my $w = $p->{'xw'}; 
      # If ranked below me
      if ($w < $my_final_wins - 1 
	|| ($w == $my_final_wins - 1 && $p->{'xs'} < $my_final_spread)) {
	# I want them to win
	if ($i <= $cap) { unshift(@hi_winners, $p); }
	else { unshift(@lo_winners, $p); }
        }
      # else ranked above me
      else { # build these lists in the other order.
	# I want them to lose
	if ($i <= $cap) { push(@hi_losers, $p); }
	else { push(@lo_losers, $p); }
        }
      }
#   Debug 'CBPF', 'MR: %d. HW: %s. LW: %s. HL: %s. LL: %s.', $myrank, (join(',',map {$_->{'name'}} @hi_winners)), (join(',',map {$_->{'name'}} @lo_winners)), (join(',',map {$_->{'name'}} @hi_losers)), (join(',',map {$_->{'name'}} @lo_losers));
    # if I can play, I always win with the largest possible spread
    if (!defined $me->Score($r0)) { 
      # find me an opponent
      # preference ranking for opponent category, according to where we stand
      my $grouppp = $myrank <= $cap 
	?  [ \@hi_losers, \@hi_winners, \@lo_losers, \@lo_winners]
	:  [ \@lo_losers, \@lo_winners, \@hi_losers, \@hi_winners];
      my $opp;
      for my $groupp (@$grouppp) 
        { if (@$groupp) { $opp = shift @$groupp; last; } }
      if ($opp) {
	$me->{'xw'}++; $me->{'xs'} += $me->{'xms'}; 
	# following is an approximation, should check when 
	# $me->{'xms'} != $opp->{'xms'} (TODO)
	# See several similar cases below. 
	$opp->{'xs'} += $opp->{'xms'}; 
        }
      else {
	$me->{'xw'}++; 
	$me->{'xs'} += $bye_spread;
	$my_final_spread += $bye_spread - $me->{'xms'};
        }
      }
    
    for my $group_data_p ((
      [\@hi_winners,[\@hi_losers,\@hi_winners,\@lo_winners,\@lo_losers]],
      [\@hi_losers,[\@hi_winners,\@hi_losers,\@lo_losers,\@lo_winners]],
      [\@lo_winners,[\@lo_losers,\@lo_winners,\@hi_losers,\@hi_winners]],
      [\@lo_losers,[\@lo_winners,\@lo_losers,\@hi_winners,\@hi_losers]],
      )) {
      my $tbps = $group_data_p->[0];
      my $p = shift @$tbps;
      next unless defined $p;
      my $grouppp = $group_data_p->[1];
      my $opp;
      for my $groupp (@$grouppp) 
        { if (@$groupp) { $opp = shift @$groupp; last; } }
      if (!$opp) { $p->{'xw'}++; $p->{'xs'} += $bye_spread; next; }
      # make sure $p is higher-ranked than $opp
      if (($p->{'xw'}<=>$opp->{'xw'}||$p->{'xs'}<=>$opp->{'xs'})<0)
        { ($p, $opp) = ($opp, $p); }
      # $p and $opp have more wins than me
      if ($opp->{'xw'} > $my_final_wins) 
        # who cares: give opp a big win
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
      # $p has more wins than me
      elsif ($p->{'xw'} > $my_final_wins) 
	# write $p off: give him a big win
        { $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
      # $p and $opp both have my wins
      elsif ($opp->{'xw'}== $my_final_wins) 
	# write $p off: give him a big win
	{ $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
      # $p has <= mine; $opp has mine-1
      elsif ($opp->{'xw'} == $my_final_wins-1) {
	# if $opp would overtake me by winning
	if ($opp->{'xs'} >= $my_final_spread) {
	  # give a big win to whichever one has more spread already
	  if ($p->{'xs'} > $opp->{'xs'}) 
        { $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
	  else 
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
	  }
	# give $opp as big a win as possible without overtaking
	else {
	  my $spread = $my_final_spread - $opp->{'xs'};
	  $opp->{'xw'}++; $opp->{'xs'} += $spread; $p->{'xs'} -= $spread;
	  }
	}
      # $p has my wins, $opp <= mine-2
      elsif ($p->{'xw'} == $my_final_wins) {
	my $delta = $p->{'xs'} - $my_final_spread;
	# $p is ahead on spread, but maybe we can bring him down
	if ($delta > 0 && $delta <= $p->{'xms'} * $rounds_left) 
	  # give opp a big win
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
	else 
	# write $p off: give him a big win
	{ $p->{'xw'}++;$p->{'xs'}+=$p->{'xms'};$opp->{'xs'}-=$opp->{'xms'}; }
        }
      # $p has <= mine-1, $opp <= mine-2
      else 
	# give opp a big win
        { $opp->{'xw'}++;$opp->{'xs'}+=$opp->{'xms'};$p->{'xs'}-=$p->{'xms'}; }
      }
    }
  
  my $rank = undef;
  @ps = sort { $b->{'xw'}<=>$a->{'xw'} || $b->{'xs'}<=>$a->{'xs'}||($a eq $me ? -1 : $b eq $me ? 1 : 0)} @$psp;
  for my $i (0..$#ps) {
    my $p = $ps[$i];
    next if $p ne $me;
    $rank = $i + 1;
#   my $behind = $i > 0 ? sprintf("%s %g %+d", $ps[$i-1]->TaggedName(), $ps[$i-1]->{'xw'}, $ps[$i-1]->{'xs'}) : 'nobody'; my $after = $i < $#ps ? sprintf("%s %g %+d", $ps[$i-1]->TaggedName(), $ps[$i+1]->{'xw'}, $ps[$i+1]->{'xs'}) : 'nobody'; Debug 'CBPF', '%s can finish ranked %d, %g %+d, behind %s, ahead of %s.', $me->TaggedName(), $rank, $my_final_wins, $my_final_spread, $behind, $after;
    last;
    }
  Debug 'GIB', "%3d=>%3d%5.1f%+5d %s #%d", $index+1, $rank, $my_final_wins, $my_final_spread, $me->Name(), $me->ID();
    return $rank;
  }

=item $n = CountGibsons($sr0, $psp)

Count the number of Gibsons in @$psp based on round $sr0 standings.
You must call Division::ComputeRanks($sr0) before calling this sub.
Gibsonization is done based on wins and spread, with the assumption
that a player can catch up 500 points of spread in one game, 700 in
two games, and 300 points per round in three or more games.

=cut

sub CountGibsons ($$) {
  my $sr0 = shift;
  my $psp = shift;
  my $sr = $sr0 + 1;
  my $p0 = $psp->[0];
  return 0 unless defined $p0;
  my $dp = $p0->Division();
  my $dname = $dp->Name();
  my $max_rounds = $dp->MaxRound0() + 1;
  my $rounds_left = $max_rounds - $sr;
  my $round_spread = 2 * MaxSpread($rounds_left);
  my $spread_allowed = $rounds_left * $round_spread;
  unless (@$psp) {
    Debug 'GIB', 'CountGibsons called with empty list';
    return 0;
    }

  my $gibson_equivalent = $config::gibson_equivalent{$dname};
  my $ngibsons = 1;
  if ($gibson_equivalent) {
    # $gibson_equivalent tells us which ranks are equivalent,
    # usually either #1 and #2 or none
    my $rank = $psp->[0]->RoundRank($sr0);
    my $here = $gibson_equivalent->[$rank] || $rank;
    if ($here) {
      while (1) {
	my $next = $gibson_equivalent->[++$rank];
	last unless $next && $next == $here;
	$ngibsons++;
        }
      }
    }
  Debug 'GIB', 'Gibson-equivalent ranks: %d starting at %d', $ngibsons, $psp->[0]->RoundRank($sr0);
  while ($ngibsons > 0) {
    my $win_diff = $psp->[$ngibsons-1]->RoundWins($sr0)
      - $psp->[$ngibsons]->RoundWins($sr0);
    if ($win_diff > $rounds_left) { last; }
    elsif ($win_diff < $rounds_left) { next; }
    my $spread_diff = $psp->[$ngibsons-1]->RoundSpread($sr0)
      - $psp->[$ngibsons]->RoundSpread($sr0);
    if ($spread_diff > $spread_allowed) { last; }
    }
  continue { $ngibsons--; }
  Debug 'GIB', 'Gibsons found: %d', $ngibsons;
  return $ngibsons;
  }

=item $cap = FlightCapDefault($rounds_left);

Apply the default algorithm for mapping rounds left to cap on
number of contenders.

=cut

sub FlightCapDefault ($) {
  my $rounds_left = shift;
  return $rounds_left <= 1 ? 2 :
    $rounds_left >= 4 ? 12 :
    ($rounds_left - 1) * 4;
  }

=item $cap = FlightCapNSC($rounds_left);

Apply the NSC algorithm for mapping rounds left to cap on
number of contenders.

=cut

sub FlightCapNSC ($) {
  my $rounds_left = shift;
  return $rounds_left > 3 ? 12 : $rounds_left * 4;
  }

=item $s = MaxSpread ($nr);

Return the maximum likely spread that a player can earn per round
over $nr rounds.

NSA rules are 500 points of spread in 1 round, 700 in 2, 900 in 3.
We generalise to 300*n for n>=3.

=cut

sub MaxSpread ($) {
  my $nr = shift;
  return $nr == 1 ? 250 : $nr == 2 ? 175 : 150;
  }

=item $parser->PairAllGibsons($psp, $sr0, $rounds_left, $last_prize_rank);

Pair all Gibsons at the top of @$psp.

=cut

sub PairAllGibsons ($$$$) {
  my $this = shift;
  my $tobepaired = shift;
  my $sr0 = shift;
  my $rounds_left = shift;
  my $last_prize_rank = shift;
  Debug 'GIB', 'PairAllGibsons(%d,%d,%d,%d)', scalar(@$tobepaired), $sr0, $rounds_left, $last_prize_rank;
  $tobepaired->[0]->Division()->ComputeRanks($sr0);
  while (my $ngibsons = CountGibsons($sr0, $tobepaired)) {
    Debug 'GIB', '%d gibson%s.', $ngibsons, ($ngibsons == 1 ? '' : 's');
    # if there is an even number of gibsons
    if ($ngibsons % 2 == 0) {
      # then pair them KOTH but minimize rematches
      PairEvenGibsons($tobepaired, $ngibsons); 
      }
    # else there is an odd number of gibsons
    else { 
      # pair all but the lowest ranked gibson KOTH minimizing rematches
      PairEvenGibsons($tobepaired, $ngibsons-1);
      # pair the last one with a low-ranked victim
      PairOneGibson($tobepaired, $last_prize_rank, $sr0, $rounds_left);
      }
    }
  }

=item PairEvenGibsons($psp, $ngibsons)

Pair an even number of gibsons with each other,
with decreasing priority given to

  - minimizing rematches
  - avoiding consecutive rematches (where possible)
  - pairing each player with the one ranked closest possible

=cut

sub PairEvenGibsons ($$) {
  my $psp = shift;
  my $ngibsons = shift;
  return unless $ngibsons && @$psp;
  Debug 'GIB', 'Pairing %d gibsons.', scalar($ngibsons);
  my $r0 = $psp->[0]->CountOpponents();
  if (
    TSH::Player::PairGRT([@$psp[0..$ngibsons-1]],
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs($_[1]-$_[2]);
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d svr=%d dist=%d', $pid, $oid, $repeats, $sameopp, $pairsvr, $distance;
	pack('NCCNN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance, # prefer closer-ranked opponents
	  $_[2], # to recover player ID
	  )
        },
      # allowed opponent filter
      sub {
	# allow any
	1,
        },
      [],
      $r0,
      )
    ) # if
    {
#   DebugDumpPairings 'GIB', $psp->[0]->CountOpponents()-1, [splice(@$psp, 0, $ngibsons)]; # splices are not thread-safe
    my @paired = TSH::Utility::SpliceSafely(@$psp, 0, $ngibsons);
    DebugDumpPairings 'GIB', $psp->[0]->CountOpponents()-1, \@paired; 
    }
  else
    {
    TSH::Utility::Error "Assertion failed: can't resolve even gibson pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairOneGibson($psp, $last_prize_rank, $sr0, $rounds_left)

Pair one gibson.  If the bottom prize band (of players who have 
nothing at stake but ratings and pride) is occupied, the victim is
the highest ranked among those who have played the gibson least often.
If not, then it's the lowest ranked overall (including higher ranked
players who may still be concerned with reaching prize money or
qualification status) among those who have played the gibson least often.

=cut

sub PairOneGibson ($$$$) {
  my $psp = shift;
  my $last_prize_rank = shift;
  my $sr0 = shift;
  my $rounds_left = shift;
  Debug 'GIB', 'Pairing one gibson from %d players, lpr=%d.', scalar(@$psp), $last_prize_rank;
  my $gibson = $psp->[0];
  my $gid = $gibson->ID();
  
  if (@$psp % 2) {
    Debug 'GIB', 'Gibson gets the bye.';
    $gibson->Division()->Pair($gid, 0, $gibson->CountOpponents());
    shift(@$psp);
    return;
    }

  CalculateBPFs $psp;

  my $victim = undef;
  my $minrep = undef;
  my $hopeless_victim = undef;
  for (my $i = $#$psp; $i > 0; $i--) {
    my $poss_victim = $psp->[$i];
    my $rep = $poss_victim->Repeats($gid);
    my $is_hopeless = $poss_victim->MaxRank() > $last_prize_rank;
    if (!defined $minrep) {
      Debug 'GIB', '... victim (rep=%d) = %s', $rep, $poss_victim->TaggedName();
      $minrep = $rep;
      $victim = $i;
      $hopeless_victim = $victim if $is_hopeless;
      }
    elsif ($minrep > $rep) {
      Debug 'GIB', '... better victim (rep %d<%d) = %s', $minrep, $rep, $poss_victim->TaggedName();
      $minrep = $rep;
      $victim = $i;
      $hopeless_victim = $victim if $is_hopeless;
      }
    elsif ($is_hopeless && $minrep == $rep) {
      Debug 'GIB', '... better victim (rep %d=%d) = %s', $minrep, $rep, $poss_victim->TaggedName();
      $hopeless_victim = $i;
      }
    }
   
  $victim = $hopeless_victim if defined $hopeless_victim;
  $gibson->Division()->Pair($gid, $psp->[$victim]->ID(),
    $gibson->CountOpponents());
  TSH::Utility::SpliceSafely(@$psp, $victim, 1);
  shift @$psp;
  return;
  }

=item $setup = $command->SetupForPairing(%options);

Check and set up variables prior to performing pairings. 
%options is modified as necessary and a reference is returned.
Supported options:

division: (input) reference to a TSH::Division

exagony: (output) true if players from same team should not play each other

filter: (output) reference to an opponent filter to pass to PairGRT

nobye: (input) true if odd player groups should be left as is without
  assigning a bye, say because of a subsequent complex Gibson calculation,
  or because we're getting ready to do multiround pairings

required: (input) names of required configuration options

repeats: (input) number of repeats allowed

source0: (input) 0-based round on which to base standings

target0: (input) 0-based round in which to store pairings

=cut

sub SetupForPairings ($%) {
  my $this = shift;
  my (%setup) = @_;
  my $dp = $setup{'division'};
  my $sr0 = $setup{'source0'};
  my $repeats = $setup{'repeats'};
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $dname = $dp->Name();

  $dp->CheckRoundHasResults($sr0) or return 0;
  $setup{'target0'} = $dp->FirstUnpairedRound0();
  my $max_round0 = $dp->MaxRound0();
  if ((defined $max_round0) && $max_round0 < $setup{'target0'}) {
    $tournament->TellUser('ebigrd', $setup{'target0'}+1, $max_round0+1);
    return 0;
    }

  # check for required configuration options
  if ($setup{'required'}) {
    for my $key (keys %{$setup{'required'}}) {
      next if defined $config->Value($key);
      $tournament->TellUser("eneed_$key");
      }
    }
  # check for wanted configuration options
  if ($setup{'wanted_div'}) {
    for my $key (keys %{$setup{'wanted_div'}}) {
      my $ref = $config->Value($key);
      next if defined $ref && $ref->{$dname};
      # if not found, warn and apply default value
      $tournament->TellUser("wwant_$key");
      if (!defined $ref) {
	$config->Value($key, ($ref = &share({})));
        }
      $ref->{$dname} = $setup{'wanted_div'}{$key};
      $config->Export();
      }
    }

  # Remind user what pairings these are
  $tournament->TellUser("i$this->{'names'}[0]ok", $dp->Name(), 
    $setup{'target0'}+1, $sr0+1, $repeats);
  
  # If exagony is set, players from the same team cannot play each other
  if ($setup{'exagony'} = 
    ($config->Value('exagony')
      || ($config->Value('initial_exagony') && $dp->LastPairedRound0() == -1))
    ) { $tournament->TellUser('irsem'); }

  # Build player list
  my $psp = $dp->GetUnpairedRound($setup{'target0'});
  unless (@$psp) { $tournament->TellUser('ealpaird'); return 0; }
  @$psp = TSH::Player::SortByCurrentStanding @$psp;
  # perform any necessary gibsonization
  if ($config->Value('gibson')) {
    if (!defined $max_round0) {
      $tournament->TellUser('eneed_max_rounds');
      return 0;
      }
    my $rounds_left = $max_round0 + 1 - $setup{'target0'};
    my $last_prize_rank = $config->Value('prize_bands');
    $last_prize_rank = $last_prize_rank->{$dname} if defined $last_prize_rank;
    $last_prize_rank = $last_prize_rank->[-1] if defined $last_prize_rank;
    if (!defined $last_prize_rank) {
      $tournament->TellUser('wwant_prize_bands');
      $last_prize_rank = int(@$psp/4) || 1;
      }
    $this->PairAllGibsons($psp, $sr0, $rounds_left, $last_prize_rank);
    }
  $dp->ChooseBye($sr0, $setup{'target0'}, $psp) 
    if @$psp % 2 && !$setup{'nobye'};
  $setup{'players'} = [TSH::Player::SortByStanding $sr0, @$psp];

  # Set up opponent filter
  $setup{'filter'} = sub {
    ($setup{'exagony'} ? 
      (
	$_[0][$_[1]]->Team() ne $_[0][$_[2]]->Team() 
	|| $_[0][$_[1]]->Team() eq ''
      )
      : 1
    ) 
    && $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $repeats
    };

  return \%setup;
  }

=over 4

=item $command->TidyAfterPairing($dp);

=cut

sub TidyAfterPairing ($$) {
  my $this = shift;
  my $dp = shift;

  $dp->Dirty(1);
  $dp->Update();
  $dp->Tournament()->TellUser('idone');
  }

=back

=cut

=head1 BUGS

SetupForPairings would be a good place to do some basic Gibsonization
detection.

Gibsonization should consider results from partially scored rounds.

The current algorithm assumes (e.g.) that if the top eight players
are paired with each other to contend for the top two places, then
the ninth-ranked player cannot finish higher than ninth place, and
is paired with other such players.

Gibsonization algorithm doesn't work well (or at all?) for more than
one final gibsonization position.

=cut

1;
