#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::AUPAIR;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::AUPAIR - implement the C<tsh> AUPAIR command

=head1 SYNOPSIS

  my $command = new TSH::Command::AUPAIR;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::AUPAIR is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to create a AUPAIR .TOU file,
to submit to a rating system requiring that input format.
EOF
  $this->{'names'} = [qw(aupair)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();
  my $event_date = $config->Value('event_date');
  my $event_name = $config->Value('event_name');
  unless ($event_date) { $tournament->TellUser('eneededate'); return; }
  unless ($event_name) { $tournament->TellUser('eneedename'); return; }
  for my $dp (@_) {
    my $dname = $dp->Name();
    my $fn = $config->MakeRootPath("$dname.TOU");
    my $fh = TSH::Utility::OpenFile(">", $fn);
    unless ($fh) {
      TSH::Utility::Error("Can't create $fn: $!");
      return;
      }
    my $nrounds = $dp->MostScores();
    if ($nrounds != $dp->LeastScores()) {
      TSH::Utility::Error "Not all results are in yet in round $nrounds.";
      return;
      }
    print $fh "*M$event_date $event_name\015\012";
    print $fh "*$dname\015\012";
    print $fh "                                      0\015\012"; # high word

    for my $p ($dp->Players()) {
      my $name = $p->Name();
      my ($surname, $given) = split(/, */, $p->Name(), 2);
      if (defined $given) {
	$name = "$given $surname";
        }
      elsif ($config->Value('realm') ne 'sgp') {
	TSH::Utility::Error "Can't find a comma separating surname from given name: $p->{'name'}\n";
	return 0;
        }
      printf $fh "%-20.20s", $name;
      for my $r (1..$nrounds) {
	my $r0 = $r - 1;
	if (defined $p->Opponent($r0)) {
	  printf $fh ($p->First($r0)==1) ? " %1s%3d%+4d" : " %1s%3d%4d", 
	    ('1','2',' ')[$p->Score($r0) <=> ($p->OpponentScore($r0)||0)],
	    $p->Score($r0),
	    $p->OpponentID($r0);
	  }
	else {
	  # Barry Harridge 2007-09-04:
	  # We overcome that limitation by representing a bye for
	  # player A as a tied game of A playing A with an arbitrary
	  # score. This should give a rating change of zero since his
	  # opponent has the same rating as himself.
	  printf $fh " %1s%3d%4d", 1, 350, $p->ID();
	  }
        } # for $r0
      print $fh "\015\012";
      } # for my $p
    print $fh "*** END OF FILE ***\015\012";
    close $fh;
    }
  $tournament->TellUser('idone');
  }

=back

=cut

=head1 BUGS

Makes extensive inappropriate use of Player.pm internals.

Currently only exports one division at a time.

=cut

1;
