#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::Addscore;

use strict;
use warnings;

use TSH::Utility;
use TSH::Tournament;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::Addscore - implement the C<tsh> Addscore command

=head1 SYNOPSIS

  my $command = new TSH::Command::Addscore;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::Addscore is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to enter player scores.  You must pair the round
(e.g. by autopairing or using the pm command) before you can enter
any scores.  Begin by specifying the round and division that you
are entering.  At the prompt, enter the first player's number and
score, then the second player's number and score, all on one line.
For a bye or forfeit, enter the player's number and the spread
adjustment.  If you don't know a player's number, try entering
part-of-their-last-name,part-of-their-first-name.  You may enter a
division name to switch divisions, or 'm' to see what scores are
still missing.  To correct a mistake in the game you just entered,
enter 'es'.  If you enter anything else, you will exit the command
and return to the main prompt.
EOF
  $this->{'names'} = [qw(a addscore)];
  $this->{'argtypes'} = [qw(Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round, $dp) = @_;
  my $round0 = $round - 1;
  if (!$config::allow_gaps) {
    if ($round0 <= $dp->{'mins'}) {
      $tournament->TellUser('eallsin', $dp->Name(), $round);
      return;
      }
    if ($round0 > $dp->{'mins'}+1) {
      $tournament->TellUser('emisss', $dp->Name(),
	$dp->{'mins'}+2, $dp->LeastScoresPlayer()->TaggedName());
      return;
      }
    }
  my $datap = $dp->{'data'};

  my $lastpn1 = 1;
  my $changed = 0;
prompt:while (1) {
    if ($config::entry eq 'spread') {
      my $left = scalar(grep { (!exists $_->{'etc'}{'off'}) 
	&& ! defined $_->{'scores'}[$round0] } 
	@$datap[1..$#$datap]);
      $left = $left == 1 ? "$left player" : "$left players";
      TSH::Utility::Prompt 
	"[$dp->{'name'}${round}]:winner loser spread ($left left)?";
      }
    else {
      my $left = scalar(grep { (!exists $_->{'etc'}{'off'}) 
        && ! defined $_->{'scores'}[$round0] } 
	@$datap[1..$#$datap]);
      $left = $left == 1 ? "$left score" : "$left scores";
      TSH::Utility::Prompt 
	"[$dp->{'name'}${round}]:pn1 ps1 pn2 ps2 ($left left)?";
      }
    local($_) = scalar(<STDIN>);
    last prompt unless defined $_;
    s/\s+$//;
    last if /^$/;
    if (my $newdp = $tournament->GetDivisionByName($_)) {
      $dp = $newdp;
      $datap = $dp->{'data'};
      last unless defined $datap;
      if (!$config::allow_gaps) {
	if ($round0 <= $newdp->{'mins'}) {
	  $tournament->TellUser('eallsin', $newdp->Name(), $round);
	  return;
	  }
	if ($round0 > $newdp->{'mins'}+1) {
	  $tournament->TellUser('emisss', $newdp->Name(),
	    $dp->{'mins'}+1, $newdp->LeastScoresPlayer()->TaggedName());
	  return;
	  }
	}
      next;
      }
    elsif (/^(?:m|miss|missing)(\s+\S+)?$/i) {
      my $div = $1;
      $div = '' unless defined $div;
      $this->Processor()->Process("missing $round$div");
      next;
      }
    elsif (/^(?:es|editscore)$/i) {
      $tournament->UpdateDivisions();
      my $dname = $dp->Name();
      $this->Processor()->Process("editscore $dname $lastpn1 $round");
      next;
      }
    elsif (/^(?:l|look)\s+[a-z]+[a-z\s]*$/i) {
      $tournament->UpdateDivisions();
      $this->Processor()->Process($_);
      next;
      }
    while (/^(.*?)(\w*),(\w*)(.*)$/) {
      my ($pre, $last, $first, $post) = ($1, $2, $3, $4);
      my $pp = $tournament->FindPlayer($last, $first, $dp);
      next prompt unless $pp;
      $_ = $pre . ($pp->ID()) . $post; 
      }
    while (/^(.*?)([a-z][a-z][-'a-z]*)(.*)$/i) {
      my ($pre, $name, $post) = ($1, $2, $3, $4);
      my $pp = $tournament->FindPlayer($name, '', $dp);
      next prompt unless $pp;
      $_ = $pre . ($pp->ID()) . $post; 
      }
    last if /[^-\d\s]/;
    my (@words) = split;
    # user entered spread for a bye
    if (@words == 2) {
      my ($pn1, $ps1) = @words;
      if ($pn1 < 1 || $pn1 > $#$datap) { 
	$tournament->TellUser('enosuchp', $pn1);
	next;
	}
      my $pp1 = $datap->[$pn1];
      my $opp1 = $pp1->{'pairings'}[$round0];
      unless ((defined $opp1) && $opp1 == 0) {
	$tournament->TellUser('enotabye', $pp1->TaggedName(), $round);
	next;
        }
      if ($pp1->{'scores'}[$round0] && $pp1->{'scores'}[$round0] != 9999) {
	$tournament->TellUser('ehass', $pp1->TaggedName(),
	  $pp1->Score($round0));
	next;
	}
      {
	my $wlt = (($ps1 <=> 0) + 1) / 2;
	printf "#%d %s %+d (%.1f %+d).\n",
	  $pp1->{id}, 
	  $pp1->{name}, 
	  $ps1,
	  $pp1->{wins} + $wlt,
	  $pp1->{spread} + $ps1,
	  ;
      }
      $lastpn1 = $pn1;
      $dp->Dirty(1);
      $changed++;
      my @time : shared = (time);
      $pp1->{'etc'}{'time'} = \@time;
      $pp1->{'scores'}[$round0] = $ps1;
      next;
      }
    # user entered scores for a game - parse them
    my ($pn1, $ps1, $pn2, $ps2);
    my $toohigh = 1499;
    my $toolow = -149;
    my $littlelow = 100;
    if ($config::entry eq 'spread') {
      # ABSP entry mode: winner loser spread
      # ... gets converted internally to...
      # standard entry mode: p1 s1 p2 s2
      if (@words != 3) { last; }
      ($pn1, $pn2, $ps1) = @words;
      $ps2 = 0;
      if ($ps1 !~ /^\d+$/) {
	$tournament->TellUser('ebadabspspread', $ps1);
	next;
	}
      $littlelow = -1;
      $toohigh = 999;
      }
    else {
      if (@words != 4) { last; }
      ($pn1, $ps1, $pn2, $ps2) = @words;
      }
    # range-check scores
    if ($pn1 < 1 || $pn1 > $#$datap) {
      $tournament->TellUser('enosuchp', $pn1);
      next;
      }
    if ($pn2 < 1 || $pn2 > $#$datap) {
      $tournament->TellUser('enosuchp', $pn2);
      next;
      }
    if ($ps1 !~ /^[-+]?\d+$/ || $ps1 < $toolow|| $ps1 > $toohigh) {
      $tournament->TellUser('ebadscore', $ps1);
      next;
      }
    if ($ps2 !~ /^[-+]?\d+$/ || $ps2 < $toolow || $ps2 > $toohigh) {
      $tournament->TellUser('ebadscore', $ps2);
      next;
      }
    if ($ps1 < $littlelow) {
      $tournament->TellUser('wlowscore', $ps1);
      }
    if ($ps2 < $littlelow) {
      $tournament->TellUser('wlowscore', $ps2);
      }
    my $pp1 = $datap->[$pn1];
    my $pp2 = $datap->[$pn2];
    if (($pp1->{'pairings'}[$round0]||0) ne $pn2) {
      TSH::Utility::Error "$pp1->{'name'} and $pp2->{'name'} did not play each other in round $round.\n";
      next;
      }
    if ((defined $pp1->Score($round0)) && $pp1->Score($round0) != 9999) {
      $tournament->TellUser('ehass', $pp1->TaggedName(),
	$pp1->Score($round0));
      next;
      }
    if ((defined $pp2->Score($round0)) && $pp2->Score($round-2) != 9999) {
      $tournament->TellUser('ehass', $pp2->TaggedName(),
	$pp2->Score($round0));
      next;   
      }
    # if as in current NSA use we ask players to indicate who went first
    # and do not determine it for them, we take the first player entered
    # on the line to be the player who played first, and complain if this
    # is inconsistent with past firsts and seconds
    if ($config::track_firsts && !$config::assign_firsts) {
      my $p12p = $pp1->{'etc'}{'p12'};
      my $old = $p12p->[$round0];
      if ($old && $old == 2) 
        { $tournament->TellUser('easbad12', $pp1->TaggedName(), 'second'); }
      $p12p->[$round0] = 1;

      $p12p = $pp2->{'etc'}{'p12'};
      $old = $p12p->[$round0];
      if ($old && $old == 1) 
        { $tournament->TellUser('easbad12', $pp2->TaggedName(), 'first'); }
      $p12p->[$round0] = 2;
      }
    if ($config::entry eq 'spread') {
      my $spread = $ps1 - $ps2;
      my $wlt = (($spread <=> 0) + 1) / 2;
      printf "#%d %s (%.1f %+d) - #%d %s (%.1f %+d).\n",
        $pp1->{id}, 
        $pp1->{name}, 
	$pp1->{wins} + $wlt,
	$pp1->{spread} + $spread,
        $pp2->{id}, 
        $pp2->{name}, 
	$pp2->{wins} + 1 - $wlt,
	$pp2->{spread} - $spread,
	;
      }
    else {
      my $spread = $ps1 - $ps2;
      my $wlt = (($spread <=> 0) + 1) / 2;
      printf "#%d %s %d (%.1f %+d) - #%d %s %d (%.1f %+d).\n",
        $pp1->{id}, 
        $pp1->{name}, 
	$ps1,
	$pp1->{wins} + $wlt,
	$pp1->{spread} + $spread,
        $pp2->{id}, 
        $pp2->{name}, 
	$ps2,
	$pp2->{wins} + 1 - $wlt,
	$pp2->{spread} - $spread,
	;
    }
    $lastpn1 = $pn1;
    $dp->Dirty(1);
    $changed++;
    my @time : shared = (time);
    $pp1->{'etc'}{'time'} = $pp2->{'etc'}{'time'} = \@time;
    $pp1->{'scores'}[$round0] = $ps1;
    $pp2->{'scores'}[$round0] = $ps2;
    }
  continue {
    if ($changed >= ($config'save_interval||10)) {
      $tournament->UpdateDivisions();
      $changed = 0;
      }
    }
  $tournament->UpdateDivisions();
  }

=back

=cut

=head1 BUGS

Should use a subprocessor rather than an event loop.

=cut

1;
