#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::HighLoss;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::HighLoss - implement the C<tsh> HighLoss command

=head1 SYNOPSIS

  my $command = new TSH::Command::HighLoss;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::HighLoss is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list the high losing scores in a division.
If you specify an integer, it indicates the number of scores you
would like to see.  If not, 20 scores are shown.
EOF
  $this->{'names'} = [qw(hl highloss)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $nscores = @_ ? shift @_ : 20;

  my $logp = new TSH::Log( $tournament, $dp, 'highloss', undef,
    {'title' => 'Division ' . $dp->Name() . ' High Losses'});
  my $entriesp = TSH::Division::FindExtremeGames::Search($dp, $nscores, 
    sub ($) { (defined $_[0][0]) && (defined $_[0][1]) && $_[0][0] < $_[0][1] }, # losses only
    sub ($$) { $_[1][0] <=> $_[0][0] }, # big losses first
    );
  $logp->ColumnClasses([ qw(round score score name name) ]);
  $logp->ColumnTitles(
    {
    'text' => [ qw(Rd LS WS Loser Winner) ],
    'html' => [ 'Round', 'Losing<br>Score', 'Winning<br>Score', 'Loser', 'Winner' ],
    }
    );
  for my $entry (@$entriesp) {
    $logp->WriteRow([
      $entry->[4]+1,
      $entry->[0],
      $entry->[1],
      $entry->[2]->TaggedName(),
      $entry->[3]->TaggedName(),
      ]);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

