#!/usr/bin/perl

# Copyright (C) 2005-2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::InitFontes;

use strict;
use warnings;

use TSH::PairingCommand;
use TSH::Player;
use TSH::Utility qw(Debug DebugOn);
use TSH::Division::Pairing::Berger;

our (@ISA) = qw(TSH::PairingCommand);

=pod

=head1 NAME

TSH::Command::InitFontes - implement the C<tsh> InitFontes command

=head1 SYNOPSIS

  my $command = new TSH::Command::InitFontes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::InitFontes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub ChooseGroups ($$);
sub initialise ($$$$);
sub MakeSchedule ($\@);
sub new ($);
sub PairGroups ($$);
sub PairGroups3 ($);
sub PairPartiallyPaired ($$$);
sub Run ($$@);

=item $parserp->ChooseGroups()

Calculate $this->{'if_groups'} based on $this->{'if_players'}.

=cut

sub ChooseGroups ($$) {
  my $this = shift;
  my $nrounds = shift;

# my $psp = $this->{'if_players'}; # splices are not thread-safe
  my @psp = @{$this->{'if_players'}};
  my $np = scalar(@psp);
  my @groups;
  # is everyone in one group?
  if ($np <= 2*$nrounds) { $this->{'if_groups'} = [\@psp]; return; }
  # do players not divide evenly?
  if (my $residue = $np % ($nrounds + 1)) { 
    my @oddballs;
    if ($nrounds == 3) { 
      # hard-coded, hand-optimised, the old-fashioned way
      if ($residue == 1) {
	# pick five players, one from each sextile except the top
	for (my $section=5; $section>=1; $section--) {
	  unshift(@oddballs,splice(@psp,int($np*($section+rand(1))/6),1));
	  }
	}
      elsif ($residue == 2) {
	# pick six players, one from each sextile
	for (my $section=5; $section>=0; $section--) {
	  unshift(@oddballs, splice(@psp, int($np*($section+rand(1))/6),1));
	  }
	}
      elsif ($residue == 3) {
	# pick three players, one from each quarter except the top
	for (my $section=3; $section>=1; $section--) {
	  unshift(@oddballs, splice(@psp, int($np*($section+rand(1))/4),1));
	  }
	}
      }
    else { # in this case $nrounds != 3
      my $parts = $nrounds + 1;
      my $top = $residue % 2;
      if ($residue != $nrounds) {
	$parts += $residue + $top;
        }
      # pick $parts-$top players, one from each $parts-ile except possibly the top
      for (my $section=$parts-1; $section>=$top; $section--) {
	unshift(@oddballs, splice(@psp, int($np*($section+rand(1))/$parts),1));
	}
      }
    Debug 'IF', 'Odd players: %s', join(',', map($_->{'id'}, @oddballs));
    push(@groups, \@oddballs);
    }
  # repeatedly pick one random player from each ($nrounds+1)-ile
  for (my $n4 = int(@psp/($nrounds+1)); $n4 > 0; $n4--) {
    my @rr = ();
    for (my $quartile = $nrounds; $quartile >= 0; $quartile--) 
      { unshift(@rr, splice(@psp, $quartile*$n4 + rand($n4), 1)); }
    Debug 'IF', 'Group: %s', join(',', map($_->{'id'}, @rr));
    push(@groups, \@rr);
    }
  $this->{'if_groups'} = \@groups;
  return;
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  DebugOn 'IF';
  $this->{'help'} = <<'EOF';
Use the InitFontes command to manually a fixed number of rounds
before starting a Swiss-paired tournament.
The field is divided into equal parts by pretournament ranking,
and round robin groups are formed by drawing one group member
from each part of the field.
EOF
  $this->{'names'} = [qw(if initfontes)];
  $this->{'argtypes'} = [qw(NumberOfIFRounds Division)];
  $this->{'if_groups'} = [];
  $this->{'if_players'} = [];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

=item $command->MakeSchedule(@players);

Return a pairings schedule for a group of players.

=cut

sub MakeSchedule ($\@) {
  my $this = shift;
  my $groupp = shift;
  # TODO: think about avoiding giving byes to unrated players
# warn scalar(@$groupp);
  my $schedule = (
    undef,
    [[undef],[undef],[undef]],
    [[1,1,1],[0,0,0]],
    [[2,undef,0],[1,0,undef],[undef,2,1]],
    [[3,2,1,0],[2,3,0,1],[1,0,3,2]],
    [[4,3,undef,1,0],[2,4,0,undef,1],[1,0,3,2,undef]],
    [[5,2,1,4,3,0],[3,4,5,0,1,2],[1,0,3,2,5,4]],
    [[2,undef,0],[1,0,undef],[undef,2,1]]
    )[scalar(@$groupp)];
# warn join(';', map { join (',', @$_) } @$schedule);
  return $schedule;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->PairGroups($nrounds);

Assign fixed pairings to the players listed in $command->{'if_groups'}.

=cut

sub PairGroups ($$) {
  my $this = shift;
  my $nrounds = shift;

  my (@rboard) = (0) x $nrounds;
  my $groups = $this->{'if_groups'};
  return unless @$groups;
  my $dp = $groups->[0][0]->Division();
  my %reserved = map { $_ => 1 } $dp->ReservedBoards();
  my $tournament = $dp->Tournament();
  my $reservations = $tournament->Config()->Value('reserved');
  $reservations = $reservations->{$dp->Name()} if $reservations;
  $reservations = [] unless $reservations;
# for my $gp (@$groups) { print $gp, "\n"; }
  for my $group (sort { ($a ? $a->[0]->ID() : 0) <=> ($b ? $b->[0]->ID() : 0); } @$groups) {
    for my $r0 (0..$nrounds-1) {
      if ($#$group == $nrounds) {
	TSH::Division::Pairing::Berger::PairGroup(
	  $group,
	  @$group - $r0,
	  {
	    'assign_firsts' => 1,
	    'koth' => 1,
	  },
	  );
	}
      else {
	my $rrround = int(2+(@$group+(@$group%2)-2)*(1-$r0/($nrounds-1)));
	# should be 2 when $r0 == $nrounds - 1
	# should be @$group (rounded up to even) when $r0 == 0
	warn "r0=$r0 rrr=$rrround\n";
	TSH::Division::Pairing::Berger::PairGroup(
	  $group,
	  $rrround,
	  {
	    'assign_firsts' => 0,
	    'koth' => 1,
	  },
	  );
        }

      for my $i (0..$#$group) {
	my $p = $group->[$i];
	my $pid = $p->ID();
	my $opp = $p->Opponent($r0);
	my $oid = $p->OpponentID($r0);
	next if $oid && $pid > $oid;
 	Debug 'IF2', "$pid vs $oid in %d at ?%d", $r0+1, $rboard[$r0]+1;
	if ($oid) {
	  my $board;
	  my $presboard = $reservations->[$pid];
	  my $oresboard = $reservations->[$oid];
	  if ($presboard) {
	    if ($oresboard && $presboard != $oresboard) {
	      $tournament->TellUser('eresconf', $p->PrettyName(), $opp->PrettyName());
	      }
	    $board = $presboard;
	    }
	  elsif ($oresboard) {
	    $board = $oresboard;
	    }
	  else {
	    while ($reserved{$board = ++$rboard[$r0]}) { }
	    }
	  $group->[$i]->Board($r0, $board);
	  $opp->Board($r0, $board) if $opp;
	  }
	}
      }
    }
  }

=item $command->PairGroups3();

Assign fixed pairings to the players listed in $command->{'if_groups'}.
This code is specific to the historically interesting case where
C<$nrounds == 3>.

=cut

sub PairGroups3 ($) {
  my $this = shift;
  my (@rboard) = (0,0,0);
  my $groups = $this->{'if_groups'};
  return unless @$groups;
  my $dp = $groups->[0][0]->Division();
  my %reserved = map { $_ => 1 } $dp->ReservedBoards();
  my $tournament = $dp->Tournament();
  my $reservations = $tournament->Config()->Value('reserved');
  $reservations = $reservations->{$dp->Name()} if $reservations;
  $reservations = [] unless $reservations;
# for my $gp (@$groups) { print $gp, "\n"; }
  for my $group (sort { ($a ? $a->[0]->ID() : 0) <=> ($b ? $b->[0]->ID() : 0); } @$groups) {
    my $schedule = $this->MakeSchedule(\@$group);
# warn "@$schedule";
    for my $r0 (0..2) {
      my $roundsched = $schedule->[$r0];
      for my $i (0..$#$roundsched) {
	my $opp;
	my $oid;
	my $p = $group->[$i];
	my $pid = $p->ID();
	if (defined(my $oii = $roundsched->[$i])) {
	  $opp = $group->[$oii];
	  $oid = $opp->ID();
	  }
	else {
	  $opp = undef;
	  $oid = 0;
	  }
	next if $oid && $pid > $oid;
 	Debug 'IF2', "$pid vs $oid in %d at ?%d", $r0+1, $rboard[$r0]+1;
	$dp->Pair($pid, $oid, $r0, 0);
	if ($oid) {
	  my $board;
	  my $presboard = $reservations->[$pid];
	  my $oresboard = $reservations->[$oid];
	  if ($presboard) {
	    if ($oresboard && $presboard != $oresboard) {
	      $tournament->TellUser('eresconf', $p->PrettyName(), $opp->PrettyName());
	      }
	    $board = $presboard;
	    }
	  elsif ($oresboard) {
	    $board = $oresboard;
	    }
	  else {
	    while ($reserved{$board = ++$rboard[$r0]}) { }
	    }
	  $group->[$i]->Board($r0, $board);
	  $opp->Board($r0, $board) if $opp;
	  }
	}
      }
    }
  }

=item $success = $command->PairPartiallyPaired($nrounds, $division);

Try to pair those players who have some pairings in the first $nrounds
rounds amongst themselves, so that the rest can be paired in quads.

=cut

sub PairPartiallyPaired ($$$) {
  my $this = shift;
  my $nrounds = shift;
  my $dp = shift;
  my $tournament = $dp->Tournament();
  # TODO: should check for active players
  my (@ps) = grep { 
    (defined $_->OpponentID(0))
    || (defined $_->OpponentID(1))
    || (defined $_->OpponentID(2)) 
    } $dp->Players();

  Debug 'IF', 'Partly paired: %s', join(',', map($_->{'id'}, @ps));
  for my $r0 (0..2) {
    my (@rps) = grep { !defined $_->OpponentID($r0) } @ps;
    Debug 'IF', 'Need to pair in Rd. %d: %s', $r0+1, join(',', map($_->{'id'}, @rps));
    if (@rps % 2) { $tournament->TellUser('eifpppo', 1); return 0; }
    while (@rps > 4) {
      my $p = shift @rps;
      my $offset = int((3-$r0)*@rps/4);
      my $found = 0;
      for my $i (1..@rps) {
	my $j = ($i + $offset) % @rps;
	my $opp = $rps[$j];
	next if $p->CountRepeats($opp) > 0;
	splice(@rps, $j, 1);
	$dp->Pair($p->ID(), $opp->ID(), $r0, 0);
	$found = 1;
	Debug 'IF', '%d vs %d on try %d, %d left', $p->ID(), $opp->ID(), $i, scalar(@rps);
	last;
	}
      unless ($found) {
	$tournament->TellUser('eifstuck', "can't pair #$p->{'id'} $p->{'name'}");
	return 0;
	}
      }
    Debug 'IF', 'Still left: %s', join(',', map($_->{'id'}, @rps));
    if (@rps == 2) {
      my $p = shift @rps;
      my $opp = shift @rps;
      if ($p->CountRepeats($opp) > 0) {
	$tournament->TellUser('eifstuck', "$p->{'id'} and $opp->{'id'} have already played");
	return 0;
        }
      $dp->Pair($p->ID(), $opp->ID(), $r0, 0);
      Debug 'IF', '%d vs %d ok', $p->ID(), $opp->ID();
      }
    # try 1-3 2-4
    if ($rps[0]->CountRepeats($rps[2]) == 0
     && $rps[1]->CountRepeats($rps[3]) == 0) {
      $dp->Pair($rps[0]->ID(), $rps[2]->ID(), $r0, 0);
      $dp->Pair($rps[1]->ID(), $rps[3]->ID(), $r0, 0);
      }
    # try 1-4 2-3
    elsif ($rps[0]->CountRepeats($rps[3]) == 0
     && $rps[1]->CountRepeats($rps[2]) == 0) {
      $dp->Pair($rps[0]->ID(), $rps[3]->ID(), $r0, 0);
      $dp->Pair($rps[1]->ID(), $rps[2]->ID(), $r0, 0);
      }
    # try 1-2 3-4
    elsif ($rps[0]->CountRepeats($rps[1]) == 0
     && $rps[2]->CountRepeats($rps[3]) == 0) {
      $dp->Pair($rps[0]->ID(), $rps[1]->ID(), $r0, 0);
      $dp->Pair($rps[2]->ID(), $rps[3]->ID(), $r0, 0);
      }
    else {
      $tournament->TellUser('eifstuck', "Can't pair last four");
      return 0;
      }
    }
  $dp->Dirty(1);
  $dp->Update();
  return 1;
  }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($nrounds, $dp) = @_;
  if ($dp->LastPairedRound0() != -1) {
    if ($config::allow_gaps) {
      if ($nrounds != 3) {
	$tournament->TellUser('eifnotodd', $nrounds);
	return 0;
        }
      if (!$this->PairPartiallyPaired($nrounds, $dp)) { return 0; }
      }
    else {
      $tournament->TellUser('ehaspair', $dp->Name());
      return 0; 
      }
    }
  unless ($nrounds % 2) { 
    $tournament->TellUser('eifnotodd', $nrounds);
    return 0;
    }
  $tournament->TellUser('iifok', $dp->Name());

  # calculate pairings
  my $sortedp = $dp->GetUnpairedRound(0);
  TSH::Player::SpliceInactive @$sortedp, $nrounds, 0;
  @$sortedp = TSH::Player::SortByInitialStanding @$sortedp;
  $this->{'if_players'} = $sortedp;
  $this->ChooseGroups($nrounds);
  if ($nrounds == 3) {
    $this->PairGroups3();
    }
  else {
    $this->PairGroups($nrounds);
    }
  $dp->Dirty(1);
  $dp->Update();
  $tournament->TellUser('idone');
  return;
  }

=back

=cut

=head1 BUGS

Makes some inappropriate use of TSH::Player internals.

Should check to see if the general Berger code works for $nrounds == 3,
and if so delete the code specific to that case.

Should check to see if the general Berger code works well in general, too.

=cut

1;
