#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundRATings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundRATings - implement the C<tsh> RoundRATings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundRATings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundRATings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub DoABSPRatings ($$);
sub DoNSARatings ($$);
sub initialise ($$$$);
sub new ($);
sub RenderTable ($$$);
sub Run ($$@);

=item DoABSPRatings($dp, $r0)

Display ratings estimates as calculated by the Association of
British Scrabble Players.

=cut

# if we don't include the following line, the parameter passing fails,
# because the ABSP 
sub ABSP::CalculateRatings (\@);

sub DoABSPRatings ($$) {
  my $dp = shift;
  my $r0 = shift;
  my $datap = $dp->{'data'};

  eval 'require ABSP';
  if ($@) {
    $dp->Tournament()->TellUser('enomod', 'ABSP.pm');
    return;
    }
  ABSP::CalculateRatings @$datap;

  RenderTable($dp, $r0,
    {
      'oldrh'=>'Input<br>Rating',
      'newrh'=>'Tourn.<br>Rating',
      'oldrt'=>'InR',
      'newrt'=>'ToR',
    });
  0;
  }

=item DoNSARatings($dp, $r0)

Display ratings estimates as calculated by the National Scrabble
Association.

=cut

sub DoNSARatings ($$) {
  my $dp = shift;
  my $r0 = shift;
  my $tournament = $dp->Tournament();

  # prepare CalculateRatings arguments - ugh
  my (@ps);
  for my $p ($dp->Players()) {
    my $id = $p->ID();
#   print $id-1, ' => ', $p->Name(), "\n";
    my $lifeg = $p->{'etc'}{'lifeg'};
    $lifeg = (defined $lifeg) ? ($lifeg->[0]||0) : 100;
    $ps[$id-1] = {
      'name' => $p->Name(),
      'oldr' => $p->Rating(),
      'pairings' => [ map { ($_||0)-1 } @{$p->{'pairings'}}[0..$r0] ],
      'scores' => [ map { $_||0 } @{$p->{'scores'}}[0..$r0] ],
      'lifeg' => $lifeg,
      'id' => $id,
      'p' => $id,
      };
    }
  for my $lib (qw(ratings ratings2)) {
    eval 'require "./$lib.pl"';
    if ($@) {
      $tournament->TellUser('enomod', "$lib.pl");
      return;
      }
    }
  if ($config::rating_system && $config::rating_system =~ /lct/i) {
    &ratings2::UseClubMultipliers(1);
    }
# &ratings2::CalculateRatings(\@ps, 'oldr', 1, 'newr', 10000, 'ewins');
# &ratings2::CalculateRatings(\@ps, 'oldr', 1, 'midr', $config::split1, 'ewins1');
# &ratings2::CalculateRatings(\@ps, 'midr', $config::split1+1, 'newr', 10000, 'ewins2');
  my $maxr = $dp->MaxRound0();
  if (defined $maxr) { 
    $maxr++;
    }
  else {
    $maxr = $dp->MostScores();
    }
  &ratings2::CalculateSplitRatings(\@ps, $maxr,
    {
      'ewins' => 'ewins',
      'lifeg' => 'lifeg',
      'newr' => 'newr',
      'oldr' => 'oldr',
      'rgames' => 'rgames',
      'pairings' => 'pairings',
      'scores' => 'scores',
      'splitr' => 'splitr',
    });
  for my $p ($dp->Players()) {
    $p->NewRating($ps[$p->ID()-1]{'newr'});
    }

  RenderTable($dp, $r0, {});
  0;
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display ratings estimates based on standings
in a given round within a division.
EOF
  $this->{'names'} = [qw(rrat roundratings)];
  $this->{'argtypes'} = [qw(Round0 Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item RenderTable($dp, $r0, $termsp)

Render the rows of the ratings table. This code is independent of
rating system.

=cut

sub RenderTable ($$$) {
  my $dp = shift;
  my $r0 = shift;
  my $terms = shift;
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $noshowlast = $config->Value('no_show_last') || $r0 < 0;
  my $showlastplayername = $config->Value('show_last_player_name');
  my $spreadentry = $config->Value('entry') eq 'spread';
  my $trackfirsts = $config->Value('track_firsts');

  my $has_classes = $dp->Classes();
  my $logp = new TSH::Log($tournament, $dp, 'ratings', $r0+1);
  my (@classes) = qw(rank wl spread rating rating rating);
  my (@html_titles) = (qw(Rank Won-Lost Spread),
    ($terms->{'oldrh'} || 'Old<br>Rating'),
    ($terms->{'newrh'} || 'New<br>Rating'),
    qw(Rating<br>Change));
  my (@text_titles) = (qw(Rnk Won-Lost Spread),
    ($terms->{'oldrt'} || 'OldR'),
    ($terms->{'newrt'} || 'NewR'),
    qw(+-R));
  if ($has_classes) {
    push(@classes, 'pclass');
    push(@html_titles, 'Class');
    push(@text_titles, 'C');
    }
  push(@classes, 'name');
  push(@html_titles, 'Player');
  push(@text_titles, 'Player');
  unless ($noshowlast) {
    push(@classes, qw(last));
    push(@html_titles, 'Last');
    push(@text_titles, 'Last');
    }
  if ($dp->LastPairedRound0() > $r0) {
    push(@classes, 'name');
    push(@html_titles, 'Next');
#   push(@text_titles, 'Next Opponent'); # too wide
    }
  $logp->ColumnClasses(\@classes);
  $logp->ColumnTitles(
    {
    'text' => \@text_titles,
    'html' => \@html_titles,
    }
    );
  $dp->ComputeRanks($r0);
  my $i = 0;
  for my $p (TSH::Player::SortByStanding($r0, $dp->Players())) {
    next unless $p->Active();
#   if ($config::pair_page_break && $i && ($i) % $config::pair_page_break == 0) { $logp->Write('',qq(</table><table class=ratings align=center cellspacing=0 style="page-break-before:always"><tr>$headings</tr>)); }
    my $rating = $p->Rating();
    my $newr = $p->NewRating();
    $i++;
    my (@fields) = (
      $p->RoundRank($r0),
      sprintf("%.1f-%.1f", $p->RoundWins($r0), $p->RoundLosses($r0)),
      sprintf("%+d ", $p->RoundSpread($r0)),
      $rating,
      $newr,
      $rating ? sprintf("%+d", $newr-$rating) : ''
      );
    push(@fields, $p->Class()) if $has_classes;
    push(@fields, $p->TaggedName());
    my (@text_fields) = @fields;
    unless ($noshowlast) {
      my $s = '';
      my $opp = $p->Opponent($r0);
      if ($opp) {
	my $oname = $showlastplayername ? $opp->TaggedName() : $opp->FullID();
	my $ms = $p->Score($r0);
	if (defined $ms) {
	  my $os = $p->OpponentScore($r0);
	  $os = 0 unless defined $os; # possible if pairing data bad
	  my $p12 = $p->First($r0);
	  $s = 
	    ($trackfirsts ? substr('B12??', $p12,1) : '')
	    .($spreadentry ? '' : ($ms > $os ? 'W' : $ms < $os ? 'L' : 'T'));
	  $s .= ':' if $s;
	  $s .=($spreadentry ? sprintf("%+d", $ms-$os) : "$ms-$os")
	    .':'
	    . $oname;
	  }
	else {
	  $s = 'pending:' . $oname;
	  }
        }
      else {
	if (defined $p->OpponentID($r0)) {
	  $s = 'bye';
	  }
	else {
	  $s = 'unpaired';
	  }
        }
      push(@fields, $s);
      push(@text_fields, $s);
      }
    # next round
    if (my $opp = $p->Opponent($r0+1)) {
      push(@fields, $opp ? $opp->FullID() : 'bye');
      }
    $logp->WriteRow(\@text_fields, \@fields);
    }
  $logp->Close();
  }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($r1, $dp) = @_;
  my $r0 = $r1 - 1;

  $dp->CheckRoundHasResults($r0) or return 0;
      
  if ($config::rating_system && $config::rating_system =~ /^absp$/i) {
    DoABSPRatings $dp, $r0;
    }
  else {
    DoNSARatings $dp, $r0;
    }
  return 0;
  }

=back

=cut

1;
