#!/usr/bin/perl

# regression test suite for tsh

use strict;
use warnings;
use POSIX ":sys_wait_h";

use IPC::Open3;

our($opt_p);

sub Clean ($);
sub CompareReceived ($\@$$);
sub Main ();
sub RunTSH ($$$$);
sub WriteConfig ($$);
sub WriteIFile ($$$);

Main;

sub Clean ($) {
  my $dir = shift;
  return if $opt_p;
  unlink <$dir/old/*>, <$dir/html/*>, <$dir/*>;
  rmdir "$dir/old";
  rmdir "$dir/html";
  rmdir $dir or die "Cannot rmdir $dir: $!";
  }

sub CompareReceived ($\@$$) {
  my $expect = shift;
  my $gotp = shift;
  my $nonl = shift;
  my $filename = shift;
  my $error = 0;

  my (@expect) = split(/\015?\012/, $expect . ' ');
  pop @expect;
  my $diff = @expect - @$gotp;
  if (@expect > @$gotp) {
    print STDERR "\n" unless $nonl;
    printf STDERR "* Expected %d more line%s than received (%d) for %s:\n", $diff,
      $diff > 1 ? 's' : '', scalar(@$gotp), $filename;
#   print STDERR map { "$_\n" } @expect[0..$#$gotp];
#   print STDERR "--- and then ---\n";
    print STDERR map { "$_\n" } @expect[@$gotp..$#expect];
    $error++;
    }
  elsif (@expect < @$gotp) {
    print STDERR "\n" unless $nonl;
    printf STDERR "* Received %d more line%s than expected (%d)  for %s:\n", -$diff,
      $diff < -1 ? 's' : '', scalar(@expect), $filename;
    print STDERR map { "$_\n" } @{$gotp}[@expect..$#$gotp];
    $error++;
    }
  for my $i (0..(@expect > @$gotp ? @$gotp : @expect)-1) {
    if ($expect[$i] ne $gotp->[$i]) {
      my $col = '?';
      my $gotchar = '[none]';
      my $expectchar = '';
      for my $j (0..length($expect[$i])) {
	if ($j > length($gotp->[$i])) { 
	  $col = $j + 1;
	  last;
	  }
	elsif (substr($expect[$i], $j, 1) ne substr($gotp->[$i], $j, 1)) {
	  $col = $j + 1;
	  $gotchar = substr($gotp->[$i], $j, 1);
	  $expectchar = substr($expect[$i], $j, 1);
	  last;
	  }
        }
      print STDERR "\n" unless $nonl;
      $gotchar = sprintf('0x%02x', ord($gotchar)) unless $gotchar =~ /[ -~]/;
      $expectchar = sprintf('0x%02x', ord($expectchar)) unless $expectchar =~ /[ -~]/;
      printf STDERR "* Did not receive expected text for %s at line %d column $col (got '%s', wanted '%s').\n", $filename,  $i+1, $gotchar, $expectchar;
      print STDERR "** Wanted:   ", $expect[$i], "\n";
      print STDERR "** Received: ", $gotp->[$i], "\n";
      $error++;
      last;
      }
    }
  return $error;
  }

sub Main () {
  my $testdir = 'test.' . time;
  my $testname = 'unnamed test';
  my %failed;
  my $config;
  my %ifiles;
  my $buffer;
  my $ifilename = undef;
  my $ofilename = undef;
  my $commands;
  my $stdout;
  my $stderr;
  my $error = 0;
  my %ofiles;
  if (@::ARGV) {
    if ($::ARGV[0] eq '-p') {
      $opt_p = 1;
      shift @::ARGV;
      }
    }
  unless (@::ARGV) {
    @::ARGV = glob('lib/test/*');
    }
  while (<>) {
    if (/^#begintest\s+(.*)/) { $testname = $1; goto resetall; }
    elsif (/^#endtest\s*$/) {
      print STDERR "+ Running test '$testname'... ";
      mkdir $testdir or die "Cannot create $testdir: $!";
      WriteConfig $testdir, $config;
      while (my ($iname, $idata) = each %ifiles) {
	WriteIFile $testdir, $iname, $idata;
        }
      my $thiserror = RunTSH $testdir, $commands, $stdout, $stderr;
      for my $ofile (sort keys %ofiles) {
	if (open my $fh, "<$testdir/$ofile") {
	  my (@got) = <$fh>;
	  for my $line (@got) { $line =~ s/[\015\012]+$//; }
	  close $fh;
	  $thiserror += CompareReceived $ofiles{$ofile}, @got, $thiserror, $ofile;
	  }
	else {
	  print STDERR "\n" unless $thiserror++;
	  print STDERR "cannot open '$ofile': $!\n";
	  }
        }
      if ($thiserror) {
	warn "failed\n";
	$failed{$testname}++;
	$error += $thiserror;
        }
      else {
	warn "ok\n";
        }

      Clean $testdir;
      goto resetall;
      }
    elsif (/^#beginconfig\s*$/) { $buffer = '';}
    elsif (/^#endconfig\s*$/) { $config = $buffer; }
    elsif (/^#begincommands\s*$/) { $buffer = ''; }
    elsif (/^#endcommands\s*$/) { $commands = $buffer; }
    elsif (/^#beginstdout\s*$/) { $buffer = ''; }
    elsif (/^#endstdout\s*$/) { $stdout = $buffer; }
    elsif (/^#beginstderr\s*$/) { $buffer = ''; }
    elsif (/^#endstderr\s*$/) { $stderr = $buffer; }
    elsif (/^#beginofile\s+(\S.*\S)$/) { $buffer = ''; $ofilename = $1; }
    elsif (/^#endofile\s*$/) { 
      $ofiles{$ofilename} = $buffer;
      $ofilename = undef;
      }
    elsif (/^#beginifile\s+(\S+)\s*$/) { $buffer = ''; $ifilename = $1; }
    elsif (/^#endifile\s*$/) {
      die "#endifile without matching #beginifile" unless defined $ifilename;
      $ifiles{$ifilename} = $buffer;
      $ifilename = undef;
      }
    elsif (s/^##/#/) {
      $buffer .= $_; 
      }
    elsif (/^#/) {
      die "Unknown directive in $::ARGV: $_\n";
      }
    else { $buffer .= $_; }
    next;
    resetall:
      $config = '';
      %ifiles = ();
      $buffer = '';
      %ofiles = ();
      $ifilename = undef;
      $ofilename = undef;
      $commands = '';
      $stdout = '';
      $stderr = '';
    }
  if ($error) {
    warn "+ Some tests failed: " . join(', ', keys %failed) . "\n";
    }
  else {
    warn "+ All tests successful.\n";
    }
  }

sub RunTSH ($$$$) {
  my $dir = shift;
  my $commands = shift;
  my $stdout = shift;
  my $stderr = shift;
  my $error = 0;
  my $command = $^O eq 'MSWin32' 
    ? "perl tsh.pl $dir"
    : "./tsh.pl $dir";
  my $pid = open3(*TOTSH, *FROMTSH, *ERRTSH, $command);
  print TOTSH $commands;
  my @got;
  my $timedout = 0;
  eval { 
    local $::SIG{'ALRM'} = sub { die "alarm\n" };
    alarm 5;
    while (<FROMTSH>) {
      chomp;
      s/\015?\012$//;
      push(@got, $_);
      }
    };
  if ($@) {
    die $@ unless $@ eq "alarm\n";
    $timedout = 1;
    print STDERR "\n" unless $error++;
    print STDERR "* Timed out waiting for stdout.\n";
    }
  if ($timedout && @got == 0) {
    print STDERR "\n" unless $error;
    print STDERR "** No data received.\n";
    }
  else {
    $error += CompareReceived $stdout, @got, $error, 'stdout';
    }
  @got = ();
  $timedout = 0;
  eval { 
    local $::SIG{'ALRM'} = sub { die "alarm\n" };
    alarm 2;
    while (<ERRTSH>) {
      chomp;
      push(@got, $_);
      }
    };
  if ($@) {
    $timedout = 1;
    $error++;
    print STDERR "\n" unless $error++;
    print STDERR "* Timed out waiting for stderr.\n";
    }
  if ($timedout && @got == 0) {
    print STDERR "\n" unless $error;
    print STDERR "** No data received.\n";
    }
  else {
    $error += CompareReceived $stderr, @got, $error, 'stderr';
    }
  close(TOTSH) || die "to-tsh exited $?";
  close(ERRTSH) || die "err-tsh exited $?";
  close(FROMTSH) || die "from-tsh exited $?";
  my $kid = waitpid($pid, WNOHANG);
  if ($kid != $pid) {
    print STDERR "\n" unless $error;
    if (kill 0, $pid) {
      print STDERR "* TSH process $pid is still running, trying to kill it.\n";
      if (kill 'TERM', $pid) {
	print STDERR "** Process killed.\n";
	$kid = waitpid($pid, 0);
	if ($kid != $pid) {
	  print STDERR "** Could not reap killed process: $!.\n";
	  if (kill 0, $pid) {
	    die "** Killed unreapable process still exists, giving up.\n";
	    }
	  }
	}
      else {
	print STDERR "** Kill failed: $!.\n";
	}
      }
    }
  return $error;
  }

sub WriteConfig ($$) {
  my $dir = shift;
  my $contents = shift;
  open my $fh, ">$dir/config.tsh" or die "Can't create $dir/config.tsh: $!";
  print $fh $contents or die "Can't write to $dir/config.tsh: $!";
  close $fh or die "Can't close $dir/config.tsh: $!";
  }

sub WriteIFile ($$$) {
  my $dir = shift;
  my $iname = shift;
  my $idata = shift;
  open my $fh, ">$dir/$iname" or die "Can't create $dir/$iname";
  print $fh $idata or die "Can't write to $dir/$iname";
  close $fh or die "Can't close $dir/$iname";
  }
