#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::PRiZes;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::PRiZes - implement the C<tsh> PRiZes command

=head1 SYNOPSIS

  my $command = new TSH::Command::PRiZes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::PRiZes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub ShowPrize ($$$$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display a prize table.
EOF
  $this->{'names'} = [qw(prz prizes)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();

  unless (@config::prizes) {
    # Dallas 2008
    push(@config::prizes, 
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>1, 'value'=>'$3000', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>2, 'value'=>'$900', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>3, 'value'=>'$700 and UWG', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>4, 'value'=>'$550', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>5, 'value'=>'$450', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>6, 'value'=>'$350', },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$325', 'groupname' => 'Group A', 'members' => [ 26..46 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$275', 'groupname' => 'Group B', 'members' => [ 47..58,60..67 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$225', 'groupname' => 'Group C', 'members' => [ 59, 68..85 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$175', 'groupname' => 'Group D', 'members' => [ 86..105 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$125', 'groupname' => 'Group E', 'members' => [ 106..125 ] },
#     { 'division'=>'A', 'type'=>'overseed', 'subtype'=>1, 'value'=>'$200', 'groupname' => 'Group AB', 'members' => [ 1..23] },
#     { 'type' => 'signup', 'subtype' => 'Group A High Play', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>121, 'value'=>'Tile bag', },
      { 'division'=>'A', 'type'=>'highwin', 'subtype'=>1, 'value'=>'Tile bag', },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'Tile bag', },
      { 'division'=>'A', 'type'=>'average', 'subtype'=>1, 'value'=>'Defalt Book', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'Wordiest Play 1', 'value'=>'Mike Baron Book', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'Wordiest Play 2', 'value'=>'Mike Baron Book', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'Wordiest Play 3', 'value'=>'Mike Baron Book', },
      { 'division'=>'1750+', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'B', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'C', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'D', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'E', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division' => 'A', 'type' => 'upset', 'subtype' => 1, 'value' => 'SamBoard' },
      { 'division' => 'A', 'type' => 'upset', 'subtype' => 2, 'value' => 'SamBoard' },
      { 'type' => 'signup', 'subtype' => 'High DO Word', 'value' => '$100' },
    );
    # Oshawa 2007
#   @config::prizes = (
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>17, 'value'=>'Rack/Bag', },
#     { 'division'=>'A', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#     { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division A High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division A Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>18, 'value'=>'Rack/Bag', },
#      { 'division'=>'B', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'B', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division B High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division B Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>20, 'value'=>'Rack/Bag', },
#      { 'division'=>'C', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'C', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division C High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division C Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>16, 'value'=>'Rack/Bag', },
#      { 'division'=>'D', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'D', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division D High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division D Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>18, 'value'=>'Rack/Bag', },
#      { 'division'=>'E', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'Portfolio Scrabble', 'groupname' => 'Newcomer', 'members' => [ 14..18 ] },
#      { 'division'=>'E', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'E', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division E High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division E Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'type' => 'signup', 'subtype' => 'Best Candy Word', 'value' => 'Scrabble chocolate' },
#      { 'type' => 'signup', 'subtype' => 'Best Dirty Word', 'value' => 'Bottle of wine' },
#      { 'type' => 'signup', 'subtype' => 'Early Bird', 'value' => 'cookies' },
#      );
    }
  if (!defined $config->Value('max_rounds')) {
    $tournament->TellUser('eneed_max_rounds');
    return;
    }
  my $partial = 0;
  for my $dp ($tournament->Divisions()) {
    unless ($dp->IsComplete()) {
      $partial = 1;
      }
    }

  my $logp = new TSH::Log($tournament, undef, 'prizes', undef);
  $logp->WritePartialWarning(4) if $partial;
  my $htmlheads = <<'EOF';
<tr class=top1>
<th class=number>#</th>
<th class=description>Description</th>
<th class=value>Value</th>
<th class=winner>Winner(s)</th>
</tr>
EOF
  $logp->Write(sprintf("%3s %s\n\n", '#', 'Prize, Winner'), $htmlheads);
  my $break = $config->Value('prizes_page_break') || 1000;
  for my $p0 (0..$#config::prizes) {
    if ($p0 && $p0 % $break == 0) {
      $logp->Write('', qq(</table><table class=prizes align=center cellspacing=0 style="page-break-before:always">\n).$htmlheads);
      }
    my $prize = $config::prizes[$p0];
    my $p1 = $p0+1;
    $logp->Write(sprintf("%3d ", $p1), "<td class=number>$p1</td>");
    $this->ShowPrize($tournament, $logp, $prize);
    $logp->Write("\n", "</tr>\n");
    }
  $logp->Close();
  return 0;
  }

=item $this->ShowPrize($tournament, $logp, $prize);

Add the data for the given prize to the log.

=cut

sub ShowPrize ($$$$) {
  my $this = shift;
  my $tournament = shift;
  my $logp = shift;
  my $prize = shift;
  my $type = $prize->{'type'} || '';
  if ($type eq 'rank') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $value = $prize->{'value'} || '?';
    my $seen = 0;
    for my $p ($dp->Players()) {
      next unless $p->Active();
      if ($p->RoundRank($lastr0) == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") if $seen++;
	my $pname = sprintf("%s %g-%g %+d", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread());
	$logp->Write("Division $dname Rank $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      }
    }
  elsif ($type eq 'average') { # Diane Firstman's prize
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastwl = 999;
    my $lastspread = 0;
    my $subtype = $prize->{'subtype'} || 1;
    my $i = 0;
    my $value = $prize->{'value'} || '?';
    my $rank = 0;
    for my $p (sort {
      abs($a->Wins()-$a->Losses()) <=> abs($b->Wins()-$b->Losses())
      || abs($a->Spread()) <=> abs($b->Spread())
      } $dp->Players()) {
      next unless $p->Active();
      my $changed = 0;
      $i++;
      my $wl = abs($p->Wins()-$p->Losses());
      my $spread = abs($p->Spread());
      if ($wl != $lastwl || $lastspread != $spread) {
	$changed++;
	$lastwl = $wl;
	$lastspread = $spread;
	$rank = $i;
#	warn "rank=$rank wl=$lastwl p=$p->{'name'}\n";
        }
      if ($rank == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	my $pname = sprintf("%s %g-%g %+d", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread());
	$logp->Write("Most Average $dname Rank $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Most Average $dname Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      }
    }
  elsif ($type eq 'grouprank') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [];
    my $groupname = $prize->{'groupname'} || 'Group ?';
    my $value = $prize->{'value'} || '?';
    my $lastw = 0;
    my $lastl = 0;
    my $lasts = 0;
    my $rank = 0;
    my (@sorted) = TSH::Player::SortByCurrentStanding grep { defined $_ } 
      map { $dp->Player($_) } @$members;
    for my $i (0..$#sorted) {
      my $changed = 0;
      my $p = $sorted[$i];
      my $w = $p->Wins();
      my $l = $p->Losses();
      my $s = $p->Spread();
      if ($lastw != $w || $lastl != $l || $lasts != $s) {
	$lastw = $w;
	$lastl = $l;
	$lasts = $s;
	$rank = $i+1;
	$changed++;
        }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %g-%g %+d (overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->RoundRank($lastr0));
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	$logp->Write("$groupname Rank: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>$groupname Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'highloss') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [1..$dp->CountPlayers()];
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
    for my $p (@players) {
      my $hl = -9999;
      for my $r0 (0..$lastr0) {
	my $ms = $p->Score($r0);
	my $os = $p->OpponentScore($r0);
	next unless defined $ms;
	next unless defined $os;
	next unless $ms < $os;
	$hl = $ms if $hl < $ms;
        }
      $p->{'xhl'} = $hl;
      }
    @players = sort { $b->{'xhl'} <=> $a->{'xhl'} } @players;
    my $lastv = -9999;
    my $rank = 0;
    for my $i (0..$#players) {
      my $changed = 0;
      my $p = $players[$i];
      my $v = $p->{'xhl'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	my $pname = sprintf("%s %d", $p->TaggedName(), $p->{'xhl'});
	$logp->Write("Division $dname ${groupname}High Loss: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname $groupname High Loss: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'highwin') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [1..$dp->CountPlayers()];
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
    for my $p (@players) {
      my $hw = -9999;
      for my $r0 (0..$lastr0) {
	my $ms = $p->Score($r0);
	my $os = $p->OpponentScore($r0);
	next unless defined $ms;
	next unless defined $os;
	next unless $ms > $os;
	$hw = $ms if $hw < $ms;
        }
      $p->{'xhw'} = $hw;
      }
    @players = sort { $b->{'xhw'} <=> $a->{'xhw'} } @players;
    my $lastv = -9999;
    my $rank = 0;
    for my $i (0..$#players) {
      my $changed = 0;
      my $p = $players[$i];
      my $v = $p->{'xhw'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	my $pname = sprintf("%s %d", $p->TaggedName(), $p->{'xhw'});
	$logp->Write("Division $dname ${groupname}High Win: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname $groupname High Win: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'lowwin') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [1..$dp->CountPlayers()];
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
    for my $p (@players) {
      my $lw = 9999;
      for my $r0 (0..$lastr0) {
	my $ms = $p->Score($r0);
	my $os = $p->OpponentScore($r0);
	next unless defined $ms;
	next unless defined $os;
	next unless $ms > $os;
	$lw = $ms if $lw > $ms;
        }
      $p->{'xlw'} = $lw;
      }
    @players = sort { $a->{'xlw'} <=> $b->{'xlw'} } @players;
    my $lastv = 9999;
    my $rank = 0;
    for my $i (0..$#players) {
      my $changed = 0;
      my $p = $players[$i];
      my $v = $p->{'xlw'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	my $pname = sprintf("%s %d", $p->TaggedName(), $p->{'xlw'});
	$logp->Write("Division $dname ${groupname}Low Win: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname $groupname Low Win: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'upset') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $subtype = $prize->{'subtype'} || 1;
    my $value = $prize->{'value'} || '?';
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my $count = $subtype + 20; # just to be safe
    my $entriesp = TSH::Division::FindExtremeGames::Search($dp, $count, 
      # Games are represented as lists: [$score1, $score2, $p1, $p2, $r0, $rat1, $rat2].
      sub ($) { 
	(defined $_[0][0]) # player one has a score
	&& (defined $_[0][1]) # player two has a score
	&& $_[0][0] > $_[0][1] # player one won
	&& $_[0][5] # player one is rated
	&& $_[0][5] < $_[0][6] # player one is rated lower than player two
	},
      sub ($$) { $_[1][6]-$_[1][5] <=> $_[0][6]-$_[0][5] }, # big ratings differences first
      );
    my $lastv = 9999;
    my $rank = 0;
    for my $i (0..$#$entriesp) {
      my $changed = 0;
      my $e = $entriesp->[$i];
      my $v = $e->[6]-$e->[5];
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	my $pname = sprintf("%s %d-%d=%d, %d-%d vs. %s", 
 	  $e->[2]->TaggedName(), 
 	  $e->[6], # opp rating
 	  $e->[5], # player rating
 	  $e->[6]-$e->[5], # rating difference
 	  $e->[0], # player score
 	  $e->[1], # opp score
 	  $e->[3]->TaggedName(), # opp name
	  );
	$logp->Write(
	  "Division $dname ${groupname}Upset: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname $groupname Upset: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>"
  	  );
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'overseed') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [map { $_->ID() } grep { defined $_ && $_->Active() } $dp->Players()];
    my $groupname = $prize->{'groupname'} ? "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = TSH::Player::SortByInitialStanding(
      map { $dp->Player($_) } @$members);
    my $lastrat = -1;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $rat = $p->Rating();
      if ($rat != $lastrat) {
	$rank = $i;
	$lastrat = $rat;
        }
      $p->{'xseed'} = $rank;
      }

    (@players) = sort { $a->RoundRank($lastr0)-$a->{'xseed'} <=>
      $b->RoundRank($lastr0)-$b->{'xseed'} }
      map { $dp->Player($_) } @$members;
# for my $p (sort { $a->RoundRank(-1) <=> $b->{'xseed'} } @players) { print $p->{'xseed'}, " ", $p->Name(), "\n"; }
    my $lastv = -@players;
    $rank = 0;
    for my $i (0..$#players) {
      my $changed = 0;
      my $p = $players[$i];
      next unless $p->Active();
      my $v = $p->RoundRank($lastr0) - $p->{'xseed'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
        $logp->Write("\n... ", "</tr><tr><td>...</td>") unless $changed;
	my $pname = sprintf("%s %g-%g %+d (seed %d overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->{'xseed'}, $p->RoundRank($lastr0));
	$logp->Write("${groupname}OverSeed: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>${groupname}OverSeed: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'signup') {
    my $subtype = $prize->{'subtype'} || '';
    my $value = $prize->{'value'} || '?';
    $logp->Write("$subtype; Value: $value; Winner: ",
      "<td class=description>$subtype</td>"
      ."<td class=value>$value</td>"
      ."<td class=name></td>");
    }
  }

=back

=cut

=head1 BUGS

Not easy to configure.

More use should be made of FindExtremeGames.

Should switch to using modern interface for Log.

Should be using the new Class() routine to designate prize classes.

=cut

1;
