#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ROSTERS;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ROSTERS - implement the C<tsh> ROSTERS command

=head1 SYNOPSIS

  my $command = new TSH::Command::ROSTERS;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ROSTERS is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list player rosters showing player numbers, ratings and names for
all divisions in your tournament, for your pretournament reference.
EOF
  $this->{'names'} = [qw(rosters)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $processor = $this->Processor();
  my $config = $tournament->Config();
  my (@divisions) = $tournament->Divisions();

  my $logp = $this->{'log'} = new TSH::Log( $tournament, undef, 'rosters', undef,
    {'title' => 'Player Rosters'});
  $this->{'divisions'} = \@divisions;
  $logp->ColumnAttributes([(q(colspan=3)) x @divisions]);
  $logp->ColumnClasses([(q(division)) x @divisions]);
  $logp->ColumnTitles(
    {
    'text' => [ map { ('','','Division '.$_->Name()) } @divisions ],
    'html' => [ map { "Division ".$_->Name() } @divisions ],
    }
    );
  $logp->ColumnAttributes([]);
  $logp->ColumnClasses([(qw(rank rating player)) x @divisions]);
  $logp->ColumnTitles(
    {
    'text' => [ ('#', qw(Rtng Player)) x @divisions],
    'html' => [ ('#', qw(Rating Player)) x @divisions],
    });
  my @players;
  my $rows = 0;
  for my $dp (@divisions) {
    push(@players, [$dp->Players()]);
    my $thisrows = scalar(@{$players[-1]});
    $rows = $thisrows if $rows < $thisrows;
    }
  for my $row (1..$rows) {
    my $pn0 = $row - 1;
    my @data;
    for my $i (0..$#players) {
      if (my $p = $players[$i][$pn0]) {
	push(@data, $p->ID(), $p->Rating(), $p->Name());
        }
      else {
	push(@data, '', '', '');
        }
      }
    $logp->WriteRow(\@data, \@data);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;
