#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ScoreBoard;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ScoreBoard - implement the C<tsh> ScoreBoard command

=head1 SYNOPSIS

  my $command = new TSH::Command::ScoreBoard;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ScoreBoard is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub AddTData1 ($$;$);
sub AddTData3 ($$$$;$);
sub AddTDataPerson ($$);
sub AddTFormat1 ($$;$);
sub AddTFormat3 ($$;$);
sub Columns ($);
sub ComputeSeeds ($);
sub HeadShot ($$);
sub initialise ($$$$);
sub new ($);
sub RenderPlayer ($$);
sub RenderTable ($$$);
sub Run ($$@);
sub SetThresholdAttribute($$$$);
sub SetupTableFormat ($);
sub TagName ($);

=item $command->AddTData1($data[, $threshold]);

Used internally to add single content data to table.

=cut

sub AddTData1($$;$) {
  my $this = shift;
  my $data = shift;
  my $threshold = shift;
  
  push(@{$this->{'table_data'}{'contents'}[0]}, $data);
  $this->SetThresholdAttribute(0, $data, $threshold);
  }

=item $command->AddTData3($new, $old, $delta[, $threshold]);

Used internally to add triple content data to table

=cut

sub AddTData3($$$$;$) {
  my $this = shift;
  my $new = shift;
  my $old = shift;
  my $delta = shift;
  my $threshold = shift;
  
  my $contentsp = $this->{'table_data'}{'contents'};
  push(@{$contentsp->[0]}, $new);
  push(@{$contentsp->[1]}, $old, ($delta && $delta > 0) ? "+$delta" : $delta);

  $this->SetThresholdAttribute(1, $delta, $threshold);
  }

=item $command->AddTDataPerson($player);

Used internally to add data for a person (the current player, or
an opponent) to the table.

=cut

sub AddTDataPlayer($$) {
  my $this = shift;
  my $p = shift;
  
  if (UNIVERSAL::isa($p, 'TSH::Player')) {
    $this->AddTData1($this->HeadShot($p));
    $this->AddTData1(TagName $p->Name());
    }
  else {
    $this->AddTData1('');
    $this->AddTData1($p);
    }
  }

=item $command->AddTFormat1($type[, $title]);
  
Used internally to add single table format data

=cut

sub AddTFormat1 ($$;$) {
  my $this = shift;
  my $type = shift;
  my $title = shift;
  my $tfhp = $this->{'table_format'}{'head'};
  my $tfdp = $this->{'table_format'}{'data'};
  
  # There are two types of rows appearing in alternation, whose information is
  # indexed here as [0] and [1].  For single data, we use rowspan=2 to combine
  # the two rows, so no [1] appears below.
  push(@{$tfhp->{'attributes'}[0]}, 'rowspan=2');
  push(@{$tfhp->{'classes'}[0]}, $type);
  push(@{$tfhp->{'titles'}[0]}, ($title || ucfirst $type));

  push(@{$tfdp->{'attributes'}[0]}, 'rowspan=2');
  push(@{$tfdp->{'classes'}[0]}, $type);
  }

=item $command->AddTFormat3($type, [@titles]);
  
Used internally to add triples of table format data.
If present, @titles specifies non-default titles for data.

=cut

sub AddTFormat3 ($$;$) {
  my $this = shift;
  my $type = shift;
  my $titles = shift;
  my $tfhp = $this->{'table_format'}{'head'};
  my $tfdp = $this->{'table_format'}{'data'};
  
  # There are two types of rows appearing in alternation, whose information is
  # indexed here as [0] and [1].  For triple data, the current information goes
  # in a cell in the upper row spanning two columns, above the old information
  # and the difference.
  push(@{$tfhp->{'attributes'}[0]}, 'colspan=2');
  push(@{$tfhp->{'classes'}[0]}, $type);
  push(@{$tfhp->{'titles'}[0]}, $titles ? $titles->[0] : "New \u$type");
  push(@{$tfhp->{'attributes'}[1]}, '');
  push(@{$tfhp->{'classes'}[1]}, "l$type", "d$type");
  push(@{$tfhp->{'titles'}[1]},
    ($titles ? $titles->[1] : "Old"),
    ($titles ? $titles->[2] : "Diff"));

  push(@{$tfdp->{'attributes'}[0]}, 'colspan=2');
  push(@{$tfdp->{'classes'}[0]}, $type);
  push(@{$tfdp->{'attributes'}[1]}, '');
  push(@{$tfdp->{'classes'}[1]}, "l$type", "d$type");
  }

=item $columns = $command->Columns()

Used internally to determine how many (meta)columns the table should have.

=cut

sub Columns ($) {
  my $this = shift;
  return $this->{'columns'};
  }

=item $command->ComputeSeeds()

Used internally to compute player seeds.

=cut

sub ComputeSeeds ($) {
  my $this = shift;
  my (@seeded) = (TSH::Player::SortByInitialStanding($this->{'dp'}->Players()));
  my @seed;
  my $lastrat = -1;
  my $rank = 1;
  for my $i (0..$#seeded) {
    my $p = $seeded[$i];
    my $rating = $p->Rating();
    if ($rating != $lastrat) {
      $rank = $i+1;
      $lastrat = $rating;
      }
    $seed[$i] = $rank;
    }
  $this->{'seed'} = \@seed;
  }

=item $html = $this->HeadShot($player)

Used internally to generate an IMG tag for a player headshot.

=cut

sub HeadShot ($$) {
  my $this = shift;
  my $p = shift;
  my $photo_size = $this->{'photo_size'};
  if ($this->Processor()->Tournament()->Config()->Value('player_photos')) {
    return sprintf(q(<img src="%s" alt="[head shot]" height="%d" width="%d">), $p->PhotoURL(), $photo_size, $photo_size);
    }
  else {
    return "&nbsp;";
    }
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to update a scoreboard suitable for displaying on a
second monitor or web kiosk.
You must specify a division, and may omit (beginning at the end) the remaining
optional arguments: first rank, last rank, pixel width of player photos, 
number of columns, seconds between refreshes.
EOF
  $this->{'names'} = [qw(sb scoreboard)];
  $this->{'argtypes'} = [qw(Division OptionalInteger OptionalInteger OptionalInteger OptionalInteger OptionalInteger)];
# print "names=@$namesp argtypes=@$argtypesp\n";
  $this->{'table_format'} = undef;
  $this->{'c_has_tables'} = undef;
  $this->{'columns'} = undef;
  $this->{'dp'} = undef;
  $this->{'first_rank'} = undef;
  $this->{'last_rank'} = undef;
  $this->{'n_subcolumns_2'} = undef;
  $this->{'photo_size'} = undef;
  $this->{'r0'} = undef;
  $this->{'r1'} = undef;
  $this->{'refresh'} = undef;
  $this->{'seed'} = undef;

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item ($html, $attribute) = $command->RenderPlayer($player);

Used internally to render a player.

=cut

sub RenderPlayer ($$) {
  my $this = shift;
  my $p = shift;
  my $r0 = $this->{'r0'};
  my $r1 = $this->{'r1'};
  my $seed = $this->{'seed'}[$p->ID()-1];
  {
    my ($crank, $lrank);
    if ($this->{'is_capped'}) {
      $crank = $p->RoundCappedRank($r0);
      $lrank = $r0 > 0 ? $p->RoundCappedRank($r0 - 1) : $seed;
      }
      else {
      $crank = $p->RoundRank($r0);
      $lrank = $r0 > 0 ? $p->RoundRank($r0 - 1) : $seed;
      }
    my $drank = $lrank - $crank;
    $this->AddTData3($crank, $lrank, $drank, 10);
  }
  $this->AddTData1($seed);
  $this->AddTData1(
    sprintf("%.1f-%.1f", $p->RoundWins($r0), $p->RoundLosses($r0))
    );
  {
    my $spread;
    if ($this->{'is_capped'}) {
      $spread = $p->RoundCappedSpread($r0);
      }
    else {
      $spread = $p->RoundSpread($r0);
      }
    $this->AddTData1($spread > 0 ? "+$spread" : $spread, 100*$r1);
  }
  # rating information
  { 
    my $rating = $p->Rating();
    my $newr = $p->NewRating();
    my $drat = ($rating && $newr) ? $newr-$rating : '';
    $this->AddTData3($newr, $rating, $drat, 5*$r1);
  }
  # record of firsts and seconds
  {
    # Not Firsts() and Seconds() as they include paired unscored future data
    # my $m1 = $p->Firsts();
    # my $m2 = $p->Seconds();
    my $p12s = $p->FirstVector();
    my $m1 = 0;
    my $m2 = 0;
    for my $p12 (@$p12s[0..$r0]) {
      $m1++ if $p12 == 1;
      $m2++ if $p12 == 2;
      }
    if ($r0 >= 0) {
    my $first = $p->First($r0);
      if ($first == 1) { $m1 = "<span class=first>$m1</span>"; }
      elsif ($first == 2) { $m2 = "<span class=first>$m2</span>"; }
      }
    $this->AddTData1("$m1-$m2");
  }
  $this->AddTDataPlayer($p);
  # last game
  {
    my $oppid = $p->OpponentID($r0);
    unless ($r0 >= 0 && defined $oppid) {
      $this->AddTData3('','','');
      $this->AddTDataPlayer('');
      }
    elsif (!$oppid) {
      my $ms = $p->Score($r0);
      $ms = (defined $ms) ? sprintf("%+d", $ms) : '';
      $this->AddTData3('', $ms, '');
      $this->AddTDataPlayer('(bye)');
      }
    else {
      my $op = $p->Opponent($r0);
      my $os = $op->Score($r0);
      if (!defined $os) {
	$this->AddTData3('', '', '');
	}
      else {
	my $ms = $p->Score($r0);
	my $spread = $ms - $os;
	$this->AddTData3($ms, $os, $spread, 100);
	}
      $this->AddTDataPlayer($op);
      }
  }
  # next game
  {
    my $oppid = $p->OpponentID($r0+1);
    unless (defined $oppid) {
      $this->AddTData1('');
      $this->AddTDataPlayer('');
      $this->AddTData1('');
      }
    elsif (!$oppid) {
      $this->AddTData1('');
      $this->AddTDataPlayer('(bye)');
      $this->AddTData1('');
      }
    else {
      my $op = $p->Opponent($r0+1);
      my $first = $p->First($r0+1);
      my $board = $p->Board($r0+1);
      $board = $this->{'dp'}->BoardTable($board) if $this->{'c_has_tables'};
      $first = $first == 1 ? '1st' : $first == 2 ? '2nd' : '';
      $this->AddTData1($first);
      $this->AddTDataPlayer($op);
      $this->AddTData1($board);
      }
  }
  } 

=item $command->RenderTable($logp, \@players);

Used internally to render the table.

=cut

sub RenderTable ($$$) {
  my $this = shift;
  my $logp = shift;
  my $psp = shift;
  my $dp = $this->{'dp'};
  my $r0 = $this->{'r0'};
  my $r1 = $this->{'r1'};
  my $columns = $this->Columns();
  my $rows = 1 + int($#$psp/$columns);
  my $c_has_tables = $this->{'c_has_tables'};
  for my $i (0..$rows-1) {
    my $j = $i;
    for my $subrow (0..1) {
      $this->{'table_data'}{'attributes'}[$subrow] 
	= [ @{$this->{'table_format'}{'data'}{'attributes'}[$subrow]} ];
      $this->{'table_data'}{'contents'}[$subrow] = [];
      }
    for my $column (0..$columns-1) {
      my $p = $psp->[$j];
      next unless $p;
#     warn "rendering player $j: $p->{'name'}";
      $this->RenderPlayer($p);
      unless ($column == $columns - 1) {
	push(@{$this->{'table_data'}{'contents'}[0]}, '');
        }
      }
    continue {
      $j += $rows;
      }
    for my $subrow (0..1) {
      $logp->ColumnClasses($this->{'table_format'}{'data'}{'classes'}[$subrow]);
      $logp->ColumnAttributes($this->{'table_data'}{'attributes'}[$subrow]);
      $logp->WriteRow([], $this->{'table_data'}{'contents'}[$subrow]);
      }
    }

  }

=item $command->Run($tournament, @parsed_args);

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = $this->{'dp'} = shift;
  my ($rank1, $rank2, $photo_size, $columns, $refresh) = @_;
  # This should all be in a sub called RunInit();
  $this->{'first_rank'} = ($rank1 || 1);
  $this->{'last_rank'} = ($rank2 || $dp->CountPlayers());
  $this->{'photo_size'} = ($photo_size || 32);
  $this->{'columns'} = ($columns || 1);
  $this->{'columns'} = ($this->{'last_rank'} - $this->{'first_rank'} + 1) if ($this->{'last_rank'} - $this->{'first_rank'} + 1) < $this->{'columns'};
  $this->{'refresh'} = ($refresh || 10);
  $this->{'table_format'} = {
    'head' => { 
      'attributes' => [ [], [] ],
      'classes' => [ [], [] ],
      'titles' => [ [], [] ],
      },
    'data' => { 
      'attributes' => [ [], [] ],
      'classes' => [ [], [] ],
      },
    };
  $this->{'table_data'} = {
    'attributes' => [ [], [] ],
    'contents' => [ [], [] ],
    };
  my $r1 = $this->{'r1'} = $dp->MostScores();
  my $r0 = $this->{'r0'} = $r1 - 1;
  my $config = $tournament->Config();
  my $c_spreadentry = $config->Value('entry') eq 'spread';
  my $c_trackfirsts = $config->Value('track_firsts');
  my $c_has_classes = $dp->Classes();
  my $c_has_tables = $this->{'c_has_tables'} = $dp->HasTables();
  my $c_is_capped = $this->{'is_capped'} = $config->Value('standings_spread_cap');

  my $logp = new TSH::Log($tournament, $dp, 'scoreboard', undef,
    {'no_console' => 1, 'notext' => 1, 'notitle' => 1, 'refresh' => $this->{'refresh'} });

  $this->SetupTableFormat();

  # store the titles
  for my $i (0..1) {
    $logp->ColumnAttributes($this->{'table_format'}{'head'}{'attributes'}[$i]);
    $logp->ColumnClasses($this->{'table_format'}{'head'}{'classes'}[$i]);
    $logp->ColumnTitles({
      'text'=>[],
      'html'=>$this->{'table_format'}{'head'}{'titles'}[$i]
    });
    }

  # compute data
  if ($c_is_capped) {
    $dp->ComputeCappedRanks($r0);
    $dp->ComputeCappedRanks($r0-1) if $r0 > 0;
    }
  else {
    $dp->ComputeRanks($r0);
    $dp->ComputeRanks($r0-1) if $r0 > 0;
    }
  $dp->ComputeRatings($r0);
  $this->ComputeSeeds();
# my (@ps) = (TSH::Player::SortByStanding($r0, $dp->Players()));
  my (@ps) = ($dp->Players());
  TSH::Player::SpliceInactive(@ps, 0, 0);
  if ($c_is_capped) {
    @ps = TSH::Player::SortByCappedStanding($r0, @ps);
    }
  else {
    @ps = TSH::Player::SortByStanding($r0, @ps);
    }
  if ($this->{'last_rank'} && $this->{'last_rank'} < @ps) {
    splice(@ps, $this->{'last_rank'});
    }
  if ($this->{'first_rank'} && $this->{'first_rank'} <= @ps) {
    splice(@ps, 0, $this->{'first_rank'}-1);
    }
  $this->RenderTable($logp, \@ps);

  $logp->Close();
  return 0;
  }

=item $command->SetThresholdAttribute($subrow, $value, $threshold);

Used internally to set the HTML attribute of the last stored content
according to its value and a threshold.

=cut

sub SetThresholdAttribute($$$$) {
  my $this = shift;
  my $subrow = shift;
  my $value = shift;
  my $threshold = shift;
  return unless $threshold;

  my $attributesp = $this->{'table_data'}{'attributes'}[$subrow];
  my $i = $#{$this->{'table_data'}{'contents'}->[$subrow]};
  $attributesp->[$i] .= ' ' if $attributesp->[$i];
  $attributesp->[$i] .= 'bgcolor="#' . (
    (!$value) ? 'ffffff' :
    $value >= $threshold ? '80ff80' : $value > 0 ? 'c0ffc0' :
    $value <= -$threshold ? 'ff8080' : 'ffc0c0'
    ) . '"';
  }

=item $command->SetupTableFormat();

Used internally to set up the table format.

=cut

sub SetupTableFormat ($) {
  my $this = shift;
  my $c_has_tables = $this->{'c_has_tables'};

  $this->AddTFormat3('rank');
  $this->AddTFormat1('seed');
  $this->AddTFormat1('wl', 'Won-Lost');
  $this->AddTFormat1('spread');
  $this->AddTFormat3('rating');
  $this->AddTFormat1('p12s', '1st-<br>2nd');
  $this->AddTFormat1('photo');
  $this->AddTFormat1('player');
  $this->AddTFormat3('score', [qw(For Agn Sprd)]);
  $this->AddTFormat1('photo');
  $this->AddTFormat1('opp', 'Last<br>Opponent');
  $this->AddTFormat1('p12', '1/2<br>Vs.');
  $this->AddTFormat1('photo');
  $this->AddTFormat1('opp', 'Next<br>Opponent');
  $this->AddTFormat1('table', ($c_has_tables ? 'Table' : 'Board'));
  $this->AddTFormat1('spacer');

  # measure the table so far
  $this->{'n_subcolumns_2'} 
    = scalar(@{$this->{'table_format'}{'head'}{'classes'}[1]});

  my $columns = $this->Columns();
  # replicate the table $columns times
  for my $head_or_data (values %{$this->{'table_format'}}) {
    for my $a_c_or_t (values %$head_or_data) { # attributes classes titles
      my $first = 1;
      for my $row (@$a_c_or_t) { 
	$row = [(@$row) x $columns];
	# omit the spacer at the end of the table
	pop @$row if $first;
	$first = 0;
	}
      }
    }
  }

=item $html = TagName($name)

Used internally to tag a player's given name and surname.

=cut

sub TagName ($) {
  my $name = shift;
  if ($name =~ /^(.*), (.*)$/) {
    my $given = $2;
    my $surname = $1;
    if ($surname =~ /^(.*)-(.*)$/) {
      my $surname1 = $1;
      my $surname2 = $2;
      if (length($surname1) < length($surname2) + length($given)) {
	$given = "$surname2, $given";
	$surname = "$surname1-";
        }
      }
    $name = "<span class=surname>$surname</span>"
      . "<span class=given>$given</span>";
    }
  return $name;
  }

=back

=cut

# sherrie's imac: 1920 x 1200

1;
