#!/usr/bin/perl -w

## tsh - tournament shell
## Copyright (C) 1998-2007 by John J. Chew, III.

=head1 NAME

B<tsh> - Scrabble tournament management shell

=head1 SYNOPSIS

B<tsh> [directory|configuration-file]

=head1 DESCRIPTION

For user information, see the accompanying HTML documentation.
This builtin (pod) documentation is intended for program maintenance 
and development.

=cut

## Version history: moved to doc/news.html

## public libraries

BEGIN { unshift(@::INC, "$::ENV{'HOME'}/lib/perl") if defined $::ENV{'HOME'}; }
use strict;
use lib './lib/perl';
use FileHandle;
BEGIN {
  if (-f 'lib/threads.txt') {
    require threads;
    import threads;
    }
  }

# use warnings FATAL => 'uninitialized';
# $::SIG{__WARN__} = sub { eval 'use Carp'; &confess($_[0]); };
# $::SIG{__DIE__} = sub { eval 'use Carp'; &confess($_[0]); };

## private libraries

use TSH::Command;
use TSH::Config;
use TSH::PairingCommand;
use TSH::Processor;
use TSH::Player;
use TSH::Tournament;
use TSH::XCommand;
use TSH::Utility qw(Debug DebugOn);

## global constants
our $gkVersion = '3.230';

## prototypes

# sub CheckGroupRepeats ($$);
# sub CmdFactorPairQuads ($$);
sub DefineExternal ($$$;$);
# sub DoFactor ($$$$);
# sub DoFactorGroup ($$$);
sub lint ();
sub LockFailed ($);
sub Main ();
sub Prompt ();
sub ReopenConsole ();
sub RunInteractive ($);
sub RunServer ($);
sub Use ($);

=head1 SUBROUTINES

=over 4

=cut

# Part of the not yet ready CmdFactorPairQuads
# sub CheckGroupRepeats ($$) {
#   my $psp = shift;
#   my $repeats = shift;
#   for my $i (0..$#$psp) {
#     my $repeatsp = $psp->[$i]{'repeats'};
#     die "Player $psp->[$i]{'name'} has no repeats information.\n"
#       unless ref($repeatsp) eq 'ARRAY';
#     for my $j ($i+1..$#$psp) {
#       if ((my $this_repeats = $repeatsp->[$psp->[$j]{'id'}]) > $repeats) {
# #	TSH::Utility::Error "Warning: $psp->[$i]{'name'} and $psp->[$j]{'name'} have played each other $this_repeats time(s).\n";
# 	return 0;
#         }
#       }
#     }
#   return 1;
#   }

# FactorPairQuads is not yet ready for public use
# sub CmdFactorPairQuads ($$) { my($argvp, $args) = @_;
#   my ($factor, $repeats, $sr, $dp) 
#     = ParseArgs $argvp, [qw(factor repeats based-on-round division)];
#   return 0 unless defined $dp;
#   my $sr0 = $sr-1;
#   $dp->CheckRoundHasResults($sr0) or return 0;
#   print "Calculating Factored Pairings for Division $dp->{'name'} based on round $sr, $repeats repeats allowed, factoring by $factor.\n";
#   DoFactor $dp, $repeats, $sr0, $factor;
#   return 0;
#   }
# 
  
sub DefineExternal ($$$;$) {
  my $name = lc shift;
  my $script = shift;
  my $template = shift;
  my $default_args = shift;

  my $command = new TSH::XCommand($main::processor, "$main::path/$script",
    [$name], $template, $default_args);
  $main::processor->AddCommand($command);
# print " $name";
  return 1;
  }

# DoFactor is not yet ready for public use
# sub DoFactor ($$$$) { my ($dp, $repeats, $sr0, $factor) = @_;
#   my $datap = $dp->{'data'};
#   my $theKeyRound = $sr0;
# 
#   my $tobepaired = $dp->GetRegularUnpaired($sr0, 'nobyes');
#   unless (@$tobepaired) {
#     TSH::Utility::Error "No players can be paired.\n";
#     return 0;
#     }
# # die "Assertion failed" unless @$tobepaired % 2 == 0;
#   my $minbyes = 0;
#   if (@$tobepaired % 2) {
#     $minbyes = CountByes $dp;
#     }
#   my (@ranked) = TSH::Player::SortByStanding $theKeyRound, @$tobepaired;
# 
#   my @pair_list;
#   my $group_number = 0;
#   while (@ranked) {
#     $group_number++;
#     my (@group) = @ranked > $factor + $factor
#       ? splice(@ranked, 0, $factor)
#       : splice(@ranked, 0);
#     my (@group_list) = DoFactorGroup \@group, $repeats, $minbyes;
#     unless (@group_list) {
#       TSH::Utility::Error "Can't factor group #$group_number. Division is partially paired.\n";
#       last;
#       }
#     push(@pair_list, @group_list);
#     }
# 
#   # store pairings
#   {
#     my $board = 1;
#     while (@pair_list) {
#       my $gp = shift @pair_list;
#       # make sure previous board numbers are set
#       for my $pp (@$gp) {
# 	my $tp = $pp->{'etc'}{'board'};
# 	my $pairingsp = $pp->{'pairings'};
# 	if (!defined $tp) {
# 	  $pp->{'etc'}{'board'} = [ (0) x @$pairingsp ];
# 	  }
# 	elsif ($#$tp < $#$pairingsp) {
# 	  push(@{$pp->{'etc'}{'board'}}, (0) x $#$pairingsp - $#$tp);
# 	  }
#         }
#       if (@$gp == 3) {
# 	# TODO: this somewhat duplicates InitFontes and should use a table
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[1]{'id'}, 0);
# 	push(@{$gp->[1]{'pairings'}},
# 	  0,              $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[1]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, 0);
# 	push(@{$gp->[1]{'etc'}{'board'}}, 0, $board, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board, 0, $board);
# 	$board += 1;
# 	}
#       elsif (@$gp == 4) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[3]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[1]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board, $board+1, $board+1);
# 	$board += 2;
# 	}
#       elsif (@$gp == 5) {
# 	# This table is not the one used in InitFontes
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, 0);
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[4]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  0,              $gp->[1]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board,   $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board,  0);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board,   0,      $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, 0,       $board+1, $board+1);
# 	$board += 2;
#         }
#       elsif (@$gp == 6) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[5]{'id'}, $gp->[3]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[5]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[4]{'id'}, $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[1]{'id'}, $gp->[5]{'id'});
# 	push(@{$gp->[5]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[2]{'id'}, $gp->[4]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board+2, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board+2, $board, $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, $board+2, $board+1, $board+2);
# 	push(@{$gp->[5]{'etc'}{'board'}}, $board, $board+2, $board+2);
# 	$board += 3;
# 	}
#       else { die "Assertion failed"; }
#       }
#       my $p1 = shift @pair_list;
#       my $p2 = shift @pair_list;
#       push(@{$p1->{'pairings'}}, $p2->{'id'});
#       push(@{$p2->{'pairings'}}, $p1->{'id'});
#   } # store pairings
# 
#   print "Done.\n";
#   $dp->Dirty(1);
#   $gTournament->UpdateDivisions();
#   }

# # TODO: this could be more efficient, but was written live at NSC 2005
# sub DoFactorGroup ($$$) {
#   my $psp = shift; # must not modify contents
#   # 0 indicates no repeats allowed, ..., 3 means up to 3 repeats = 4 pairings
#   my $repeats = shift;
#   # TODO: allow for possibility that we have to increase minbytes after 1st plr
#   my $minbyes = shift;
# 
#   print "DFG: " . (1+$#$psp) . ' ' . join(',', map { $_->{'id'} } @$psp) . "\n";
#   if (@$psp == 4 || @$psp == 6) {
#     if (CheckGroupRepeats $psp, $repeats) {
# #     print "DFG: returning $#$psp+1\n";
#       return ([@$psp]);
#       }
#     else {
# #     print "DFG: returning failure\n";
#       return ();
#       }
#     }
#   elsif (@$psp == 3) {
#     for my $p (@$psp) {
#       if ($p->{'byes'} != $minbyes) {
# #	print "DFG: $p->{'name'} already has $p->{'byes'} bye(s).\n";
# 	return ();
#         }
#       }
#     return ([@$psp]);
#     }
#   elsif (@$psp == 5) {
#     my $possible_bye_players = 0;
#     for my $p (@$psp) {
#       if ($p->{'byes'} == $minbyes) {
# 	$possible_bye_players++;
#         }
#       }
#     if ($possible_bye_players < 3) {
#       for my $p (@$psp) {
#         print "DFG5: $p->{'name'} already has $p->{'byes'} bye(s).\n";
#         }
#       return ([
# 	sort { $b->{'byes'} <=> $a->{'byes'} } @$psp
#         ]);
#       }
#     }
#   elsif (@$psp < 7) {
#     die "DoFactorGroup: bad group size: " . scalar(@$psp) . "\n";
#     }
#   my $s = int(@$psp/4);
#   my $p1 = $psp->[0];
#   my $j1 = 0;
#   # first try to pair within quartiles
#   for my $j2 ($s..$s+$s-1) {
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $j3 ($s+$s..$s+$s+$s-1) {
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $j4 ($s+$s+$s..$s+$s+$s+$s-1) {
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
#   # then try to pair anywhere within the group
#   for my $i2 (0..$#$psp) {
#     my $j2 = ($i2 + $s) % @$psp;
#     next if $j2 == $j1;
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $i3 (0..$#$psp) {
#       my $j3 = ($i3 + $s + $s) % @$psp;
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $i4 (0..$#$psp) {
# 	my $j4 = ($i4 + $s + $s + $s) % @$psp;
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
# # print "DFG: returning failure.\n";
#   return ();
#   }

sub lint () {
  lint;
  }

sub LockFailed ($) {
  my $reason = shift;
  print <<"EOF";
System call failed: $reason

You should not run more than one copy of tsh using the same 
configuration file at the same time.  tsh uses a "lock file" called
tsh.lock to keep track of when it is running.  This copy of tsh
was unable to get access to the lock file.  The most likely reason
for this is that tsh is already in use.
EOF
  exit 1;
  }

sub Main () {
  srand;

  DebugOn('CBPF');
  DebugOn('CP');
  DebugOn('GIB');
  ReopenConsole if $^O eq 'MacOS';
  my $dir = @::ARGV ? shift @::ARGV : undef;
  my $tournament = new TSH::Tournament($dir);
  {
    my $error = $tournament->Lock();
    LockFailed $error if $error;
  }
  eval { $tournament->LoadConfiguration(); };
  if ($@) {
    print "Configuration error: $@\n";
    print "Press enter (return) to exit.\n";
    scalar(<>);
    exit 1;
    }
  # must load processor after configuration
  my $processor = new TSH::Processor($tournament);
  my $config = $tournament->Config();
  $tournament->TellUser('iwelcome', $gkVersion);
  $tournament->LoadDivisions();
  if ($tournament->CountDivisions() == 0) {
    $tournament->TellUser('enodivs');
    exit(1);
    }
  $config->DivisionsLoaded();
  $config->Export(); # deprecated, but must come after DivisionsLoaded
  my $event_name = $config->Value('event_name');
  if (defined $event_name) {
    my $date = $config->Value('event_date');
    my $namedate = $event_name;
    $namedate .= ", " . $date if defined $date;
    $tournament->TellUser('ievtname', $namedate);
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/ && -t STDOUT) {
      print "\e]0;tsh $gkVersion - $event_name\a";
      }
    }
  if ($config->Value('port')) { RunServer $processor; }
  else { RunInteractive $processor; }
  $tournament->Unlock();
  if (defined $event_name) {
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/ && -t STDOUT) {
      print "\e]0;\a";
      }
    }
  }

sub Prompt () { 
  TSH::Utility::PrintColour 'yellow on blue', 'tsh>';
  print ' ';
  }

sub ReopenConsole () {
  close(STDOUT);
  close(STDERR);
  close(STDIN);
  open(STDOUT, "+>dev:console:tsh console") || die;
  open(STDERR, "+>dev:console:tsh console") || die;
  open(STDIN, "<dev:console:tsh console") || die;
  $| = 1;
  }

sub RunInteractive ($) {
  my $processor = shift;
  if (-t STDIN && -t STDOUT) {
    eval "use Term::ReadLine";
    my $term = new Term::ReadLine "tsh $gkVersion";
  # my $ofh = $term->OUT || \*STDOUT;
    while (1) {
      $_ = $term->readline($processor->Prompt());
      last unless defined $_;
      $processor->Process($_)
	or print "Enter 'help' for help.\n";
      last if $processor->QuittingTime();
      }
    }
  else {
    while (<>) {
      $processor->Process($_)
	or print "Enter 'help' for help.\n";
      last if $processor->QuittingTime();
      }
    }
  }

sub RunServer ($) {
  my $processor = shift;
  my $tournament = $processor->Tournament();
  Use "TSH::Server";
  my $server 
    = new TSH::Server($tournament->Config()->Value('port'), $tournament);
  unless ($server->Start()) {
    $tournament->TellUser('etshtpon', $server->Error());
    RunInteractive $processor;
    return;
    }
  $processor->Process('GUI');
  eval { threads->list(); };
  my $has_threads = $@ eq '';
  if ($has_threads) {
    my $serverthread = threads->new(sub {
      while ($server->Run()) { }
      $server->Stop();
      });
    RunInteractive $processor;
    print "\nShutting down server.\n";
    $server->Stop();
    $serverthread->join; # would block until server completed execution, should instead signal exit
    }
  else {
    while ($server->Run()) { }
    $server->Stop();
    }
  }

=item Use $module or die

Carefully evaluate "use $module" and try to explain to a naive user
the meaning of failure.

=cut

sub Use ($) {
  my $module = shift;
  eval "use $module";
  die "Couldn't load $module, most likely because your tsh software distribution\nis incomplete.  Please download it again.  Here is what Perl says caused\nthe problem:\n$@" if $@;
  return 1;
  }

=back

=cut

END { 
  sleep 10 if $^O eq 'MSWin32'; # to prevent error messages from disappearing
  }

## main code
Main;

