#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::PRiZes;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility qw(Debug DebugOn Ordinal);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::PRiZes - implement the C<tsh> PRiZes command

=head1 SYNOPSIS

  my $command = new TSH::Command::PRiZes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::PRiZes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub ShowPrize ($$$$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display a prize table.
EOF
  $this->{'names'} = [qw(prz prizes)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();

  unless (@config::prizes) {
    # Dallas 2008
    push(@config::prizes, 
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>1, 'value'=>'$3000', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>2, 'value'=>'$900', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>3, 'value'=>'$700 and UWG', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>4, 'value'=>'$550', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>5, 'value'=>'$450', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>6, 'value'=>'$350', },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$325', 'groupname' => 'Group A', 'members' => [ 26..46 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$275', 'groupname' => 'Group B', 'members' => [ 47..58,60..67 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$225', 'groupname' => 'Group C', 'members' => [ 59, 68..85 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$175', 'groupname' => 'Group D', 'members' => [ 86..105 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$125', 'groupname' => 'Group E', 'members' => [ 106..125 ] },
#     { 'division'=>'A', 'type'=>'overseed', 'subtype'=>1, 'value'=>'$200', 'groupname' => 'Group AB', 'members' => [ 1..23] },
#     { 'type' => 'signup', 'subtype' => 'Group A High Play', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>121, 'value'=>'Tile bag', },
      { 'division'=>'A', 'type'=>'highwin', 'subtype'=>1, 'value'=>'Tile bag', },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'Tile bag', },
      { 'division'=>'A', 'type'=>'average', 'subtype'=>1, 'value'=>'Defalt Book', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'Wordiest Play 1', 'value'=>'Mike Baron Book', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'Wordiest Play 2', 'value'=>'Mike Baron Book', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'Wordiest Play 3', 'value'=>'Mike Baron Book', },
      { 'division'=>'1750+', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'A', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'B', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'C', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'D', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division'=>'E', 'type'=>'signup', 'subtype'=>'High Word', 'value'=>'$20', },
      { 'division' => 'A', 'type' => 'upset', 'subtype' => 1, 'value' => 'SamBoard' },
      { 'division' => 'A', 'type' => 'upset', 'subtype' => 2, 'value' => 'SamBoard' },
      { 'type' => 'signup', 'subtype' => 'High DO Word', 'value' => '$100' },
    );
    }
  if (!defined $config->Value('max_rounds')) {
    $tournament->TellUser('eneed_max_rounds');
    return;
    }
  my $partial = 0;
  for my $dp ($tournament->Divisions()) {
    unless ($dp->IsComplete()) {
      $partial = 1;
      }
    }

  my $logp = new TSH::Log($tournament, undef, 'prizes', undef);
  $logp->WritePartialWarning(4) if $partial;
  my (@classes) = qw(number description value winner);
  my (@text_titles) = ('#', 'Prize, Winner');
  my (@html_titles) = ('#', qw(Description Value Winner(s)));
  $logp->ColumnClasses(\@classes, 'top1');
  $logp->ColumnTitles({'text' => \@text_titles, 'html' => \@html_titles});
  my $break = $config->Value('prizes_page_break') || 1000;
  for my $p0 (0..$#config::prizes) {
    if ($p0 && $p0 % $break == 0) {
      $logp->PageBreak();
      $logp->ColumnTitles({'text' => [], 'html' => \@html_titles});
      }
    my $prize = $config::prizes[$p0];
    my $p1 = $p0+1;
    $this->ShowPrize($tournament, $logp, $p1, $prize);
    }
  $logp->Close();
  return 0;
  }

my %gPrizeData;
BEGIN {
  # when 'unit' == 'game',
  #   games are represented as lists: [$score1, $score2, $p1, $p2, $r0, $rat1, $rat2].
  # when 'unit' == 'player', the objects passed to subs are of class TSH::Player
  %gPrizeData = (
    'brlh' => { # best record lower half, as per US NSC
      'initialise' => sub { 
	my $prize = shift;
	my $tournament = shift;
	my $dp = $tournament->GetDivisionByName($prize->{'division'});
	my $firstr0 = $prize->{'first'} - 1;
	$dp->ComputeRanks($prize->{'first'}-2);
	my (@players) = TSH::Player::SortByStanding($firstr0-1,
	  grep { $_->CountScores() > $firstr0 } # must have played at least one game today
	  $dp->Players());
        for my $i (0..@players/2-1) 
	  { $players[$i]->{'xr'} = 0; }
	for my $i (@players/2 .. $#players) {
	  { $players[$i]->{'xr'} = $i+1; }
	  }
        },
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $p = shift;
	my $prize = shift;
	my $firstr0 = $prize->{'first'} - 1;
	my $lastr0 = $prize->{'last'} - 1;
	return 0 unless $p->CountScores() > $firstr0; # must have played at least one game today
	return 0 unless $p->{'xr'};
	$p->{'xw'} = $p->RoundWins($lastr0)-$p->RoundWins($firstr0-1);
	$p->{'xl'} = $p->RoundLosses($lastr0)-$p->RoundLosses($firstr0-1);
	$p->{'xs'} = $p->RoundSpread($lastr0)-$p->RoundSpread($firstr0-1);
	return 1;
	},
      'compare' => sub ($$) { 
	$_[1]->{'xw'} <=> $_[0]->{'xw'}
	|| $_[0]->{'xl'} <=> $_[1]->{'xl'}
	|| $_[1]->{'xs'} <=> $_[0]->{'xs'}
	},
      'prizename' => sub {
	my $prize = shift;
	return "Division $prize->{'division'} Best Record Lower Half Rounds $prize->{'first'}-$prize->{'last'}";
        },
      'formatwinner' => sub {
	my $p = shift @_;
	return sprintf("%s %.1f-%.1f %+d starting %s", $p->TaggedName(), 
	  $p->{'xw'}, $p->{'xl'}, $p->{'xs'},
	  Ordinal($p->{'xr'}));
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return 
	join(';', @$p{qw(xw xl xs)}) 
        },
      },
    'highloss' => {
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $p = shift;
	$p->{'xhl'} = -9999;
	$p->{'xr0'} = undef;
	for my $r0 (0..$p->CountScores()-1) {
	  my $ms = $p->Score($r0);
	  my $os = $p->OpponentScore($r0);
	  next unless (defined $ms) && (defined $os) && $ms < $os;
	  if ($p->{'xhl'} < $ms) {
	    $p->{'xhl'} = $ms;
	    $p->{'xr0'} = $r0;
	    }
	  }
	return defined $p->{'xr0'};
	},
      'compare' => sub ($$) { $_[1]->{'xhl'} <=> $_[0]->{'xhl'} },
      'prizename' => 'High Loss',
      'formatwinner' => sub {
	my $p = shift @_;
	my $r0 = $p->{'xr0'};
	return sprintf("%s %d-%d vs. %s", $p->TaggedName(), $p->{'xhl'}, 
	  $p->OpponentScore($r0), $p->Opponent($r0)->TaggedName());
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return $p->{'xhl'}; },
      },
    'highroundloss' => {
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $p = shift;
	my $prize = shift;
	my $r1 = $prize->{'round'};
	my $r0 = $r1 - 1;
	my $ms = $p->Score($r0);
	return 0 unless defined $ms;
	my $os = $p->OpponentScore($r0);
	return 0 unless defined $os;
	$p->{'xhrl'} = $ms;
	return $ms < $os;
	},
      'compare' => sub ($$) { $_[1]->{'xhrl'} <=> $_[0]->{'xhrl'} },
      'prizename' => sub {
	my $prize = shift;
	return "Division $prize->{'division'} High Loss Round $prize->{'round'}";
        },
      'formatwinner' => sub {
	my $p = shift;
	my $prize = shift;
	my $r1 = $prize->{'round'};
	my $r0 = $r1 - 1;
	return sprintf("%s %d-%d vs. %s", $p->TaggedName(), $p->{'xhrl'}, 
	  $p->OpponentScore($r0), $p->Opponent($r0)->TaggedName());
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return $p->{'xhrl'}; },
      },
    'highroundwin' => {
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $p = shift;
	my $prize = shift;
	my $r1 = $prize->{'round'};
	my $r0 = $r1 - 1;
	my $ms = $p->Score($r0);
	return 0 unless defined $ms;
	my $os = $p->OpponentScore($r0);
	return 0 unless defined $os;
	$p->{'xhrw'} = $ms;
	return $ms > $os;
	},
      'compare' => sub ($$) { $_[1]->{'xhrw'} <=> $_[0]->{'xhrw'} },
      'prizename' => sub {
	my $prize = shift;
	return "Division $prize->{'division'} High Win Round $prize->{'round'}";
        },
      'formatwinner' => sub {
	my $p = shift;
	my $prize = shift;
	my $r1 = $prize->{'round'};
	my $r0 = $r1 - 1;
	return sprintf("%s %d-%d vs. %s", $p->TaggedName(), $p->{'xhrw'}, 
	  $p->OpponentScore($r0), $p->Opponent($r0)->TaggedName());
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return $p->{'xhrw'}; },
      },
    'highwin' => {
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $p = shift;
	$p->{'xhw'} = -9999;
	$p->{'xr0'} = undef;
	for my $r0 (0..$p->CountScores()-1) {
	  my $ms = $p->Score($r0);
	  my $os = $p->OpponentScore($r0);
	  next unless (defined $ms) && (defined $os) && $ms > $os;
	  if ($p->{'xhw'} < $ms) {
	    $p->{'xhw'} = $ms;
	    $p->{'xr0'} = $r0;
	    }
	  }
	return defined $p->{'xr0'};
	},
      'compare' => sub ($$) { $_[1]->{'xhw'} <=> $_[0]->{'xhw'} },
      'prizename' => 'High Win',
      'formatwinner' => sub {
	my $p = shift @_;
	my $r0 = $p->{'xr0'};
	return sprintf("%s %d-%d vs. %s", $p->TaggedName(), $p->{'xhw'}, 
	  $p->OpponentScore($r0), $p->Opponent($r0)->TaggedName());
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return $p->{'xhw'}; },
      },
    'lowwin' => {
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $p = shift;
	$p->{'xlw'} = 9999;
	$p->{'xr0'} = undef;
	for my $r0 (0..$p->CountScores()-1) {
	  my $ms = $p->Score($r0);
	  my $os = $p->OpponentScore($r0);
	  next unless (defined $ms) && (defined $os) && $ms > $os;
	  if ($p->{'xlw'} > $ms) {
	    $p->{'xlw'} = $ms;
	    $p->{'xr0'} = $r0;
	    }
	  }
	return defined $p->{'xr0'};
	},
      'compare' => sub ($$) { $_[0]->{'xlw'} <=> $_[1]->{'xlw'} },
      'prizename' => 'Low Win',
      'formatwinner' => sub {
	my $p = shift @_;
	my $r0 = $p->{'xr0'};
	return sprintf("%s %d-%d vs. %s", $p->TaggedName(), $p->{'xlw'}, 
	  $p->OpponentScore($r0), $p->Opponent($r0)->TaggedName());
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return $p->{'xlw'}; },
      },
    'rank' => {
      'unit' => 'player',
      'filter' => sub ($$) { 
#	$_[0]->Active();
	1;
        },
      # TODO: duplicated in TSH::Player::SortByCurrentStanding
      'compare' => sub ($$) { 
	$_[1]->Wins() <=> $_[0]->Wins()
	|| $_[0]->Losses() <=> $_[1]->Losses()
	|| $_[1]->Spread() <=> $_[0]->Spread()
	|| $_[1]->Rating() <=> $_[0]->Rating()
	|| $_[1]->Random() <=> $_[0]->Random()
        },
      'prizename' => 'Rank',
      'formatwinner' => sub {
	my $p = shift @_;
	return sprintf("%s %g-%g %+d", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread());
        },
      'rankcriterion' => sub ($) { 
	my $p = shift @_;
	return join(';', $p->Wins(), $p->Losses(), $p->Spread()); 
        },
      },
    'tuffluck' => {
      'initialise' => sub { 
	eval "use TSH::Player::TuffLuck";
	die $@ if $@;
        },
      'unit' => 'player',
      'filter' => sub ($$) { 
	my $pp = shift;
	my ($tl, $lossesp) = $pp->TuffLuck(6);
	$pp->{'xtl'} = $tl or return 0;
	$pp->{'xtls'} = $lossesp;
	return 1;
	},
      # big ratings differences first
      'compare' => sub ($$) { $_[0]{'xtl'} <=> $_[1]{'xtl'} },
      'prizename' => 'Tuff Luck',
      'formatwinner' => sub {
	my $p = shift @_;
	return sprintf("%s %d = %s",
	  $p->TaggedName(),
	  $p->{'xtl'},
	  join('+', @{$p->{'xtls'}}),
	  );
        },
      'rankcriterion' => sub ($) { my $p = shift @_; return $p->{'xtl'}; }
      },
    'upset' => {
      'unit' => 'game',
      'filter' => sub ($$) { 
	(defined $_[0][0]) # player one has a score
	&& (defined $_[0][1]) # player two has a score
	&& $_[0][0] > $_[0][1] # player one won
	&& $_[0][5] # player one is rated
	&& $_[0][5] < $_[0][6] # player one is rated lower than player two
	},
      # big ratings differences first
      'compare' => sub ($$) { $_[1][6]-$_[1][5] <=> $_[0][6]-$_[0][5] },
      'prizename' => 'Ratings Upset Win',
      'formatwinner' => sub {
	my $e = shift @_;
	return sprintf("%s %d-%d=%d, %d-%d vs. %s", 
 	  $e->[2]->TaggedName(), 
 	  $e->[6], # opp rating
 	  $e->[5], # player rating
 	  $e->[6]-$e->[5], # rating difference
 	  $e->[0], # player score
 	  $e->[1], # opp score
 	  $e->[3]->TaggedName(), # opp name
	  );
        },
      'rankcriterion' => sub ($) { my $e = shift @_; return $e->[6]-$e->[5]; },
      },
    );
  }

=item $this->ShowPrize($tournament, $logp, $prizeno, $prize);

Add the data for the given prize to the log.

=cut

sub ShowPrize ($$$$) {
  my $this = shift;
  my $tournament = shift;
  my $logp = shift;
  my $prizeno = shift;
  my $prize = shift;
  my $type = $prize->{'type'} || '';
  if (my $prize_data = $gPrizeData{$type}) {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    # subtype: typically the rank within the category being awarded
    my $subtype = $prize->{'subtype'} || 1;
    my $value = $prize->{'value'} || '?';
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my $count = $subtype + 20; # just to be safe
    my $initialiser = $prize_data->{'initialise'};
    if (ref($initialiser) eq 'CODE') { &$initialiser($prize, $tournament); }
    my $entriesp;
    my $unit = $prize_data->{'unit'};
    if ($unit eq 'game') {
      $entriesp = TSH::Division::FindExtremeGames::Search(
	$dp, $count, \&{$prize_data->{'filter'}}, \&{$prize_data->{'compare'}})
      }
    elsif ($unit eq 'player') {
      my $compare = $prize_data->{'compare'};
      my $members = $prize->{'members'}
	? eval($prize->{'members'}) : [1..$dp->CountPlayers()];
      my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
#     die "@$members" if ref($prize->{'members'}) eq 'ARRAY';
      $entriesp = 
        [sort $compare
        grep { &{$prize_data->{'filter'}}($_, $prize) } 
	@players];
      }
    my $lastv = undef;
    my $rank = 0;
    my $rankcriterion = $prize_data->{'rankcriterion'};
    my $prize_name = $prize_data->{'prizename'};
    if ($prize->{'prizename'}) {
      $prize_name = $prize->{'prizename'};
      }
    elsif (ref($prize_name) eq 'CODE') {
      $prize_name = &$prize_name($prize);
      }
    else {
      $prize_name = "Division $dname ${groupname}$prize_name";
      }
    if ($subtype != 1 || $prize_data->{'prizename'} eq 'Rank') 
      { $prize_name .= ": " . Ordinal($subtype); }

    for my $i (0..$#$entriesp) {
      my $changed = 0;
      my $e = $entriesp->[$i];
      my $v = &$rankcriterion($e);
      if ((!defined $lastv) || $v ne $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
	$prizeno = '...' unless $changed;
	my $pname = &{$prize_data->{'formatwinner'}}($e, $prize);
	$logp->WriteRow(
	  [$prizeno, "$prize_name; Value: $value; Winner: $pname"],
	  [$prizeno, $prize_name, $value, $pname]
	  );
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'average') { # Diane Firstman's prize
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastwl = 999;
    my $lastspread = 0;
    my $subtype = $prize->{'subtype'} || 1;
    my $i = 0;
    my $value = $prize->{'value'} || '?';
    my $rank = 0;
    for my $p (sort {
      abs($a->Wins()-$a->Losses()) <=> abs($b->Wins()-$b->Losses())
      || abs($a->Spread()) <=> abs($b->Spread())
      } $dp->Players()) {
      next unless $p->Active();
      my $changed = 0;
      $i++;
      my $wl = abs($p->Wins()-$p->Losses());
      my $spread = abs($p->Spread());
      if ($wl != $lastwl || $lastspread != $spread) {
	$changed++;
	$lastwl = $wl;
	$lastspread = $spread;
	$rank = $i;
#	warn "rank=$rank wl=$lastwl p=$p->{'name'}\n";
        }
      if ($rank == $subtype) {
	$prizeno = '...' unless $changed;
	my $pname = sprintf("%s %g-%g %+d", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread());
	my $prizename = "Division $dname Most Average Rank: $subtype";
	$logp->WriteRow([$prizeno, "$prizename; Winner: $pname"],
	  [$prizeno, $prizename, $value, $pname]);
        }
      }
    }
  elsif ($type eq 'overseed') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $dp->MaxRound0();
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [map { $_->ID() } grep { defined $_ && $_->Active() } $dp->Players()];
    my $groupname = $prize->{'groupname'} ? "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = TSH::Player::SortByInitialStanding(
      map { $dp->Player($_) } @$members);
    my $lastrat = -1;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $rat = $p->Rating();
      if ($rat != $lastrat) {
	$rank = $i;
	$lastrat = $rat;
        }
      $p->{'xseed'} = $rank;
      }

    (@players) = sort { $a->RoundRank($lastr0)-$a->{'xseed'} <=>
      $b->RoundRank($lastr0)-$b->{'xseed'} }
      map { $dp->Player($_) } @$members;
# for my $p (sort { $a->RoundRank(-1) <=> $b->{'xseed'} } @players) { print $p->{'xseed'}, " ", $p->Name(), "\n"; }
    my $lastv = -@players;
    $rank = 0;
    for my $i (0..$#players) {
      my $changed = 0;
      my $p = $players[$i];
      next unless $p->Active();
      my $v = $p->RoundRank($lastr0) - $p->{'xseed'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; $changed++; }
      if ($rank == $subtype) {
	$prizeno = '...' unless $changed;
	my $pname = sprintf("%s %g-%g %+d (seed %d overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->{'xseed'}, $p->RoundRank($lastr0));
	my $prizename = "${groupname}OverSeed: $subtype";
	$logp->WriteRow([$prizeno, "$prizename; Winner: $pname"],
	  [$prizeno, $prizename, $value, $pname]);
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'signup') {
    my $subtype = $prize->{'subtype'} || '';
    my $value = $prize->{'value'} || '?';
    $logp->WriteRow([$prizeno, "$subtype; Value: $value; Winner: "],
      [$prizeno, $subtype, $value, '']);
    }
  }

=back

=cut

=head1 BUGS

Not easy to configure.

More use should be made of FindExtremeGames and TSH::Utility::DoOneRank.

Should switch to using modern interface for Log.

Should be using the new Class() routine to designate prize classes.

Should allow multidivision syntax in prize data.

Should parse monetary value out of prize value, and use to automate 
class prizes.

The routines in gPrizeData should be moved to subclasses of 
TSH::Game::Compare or TSH::Player::Compare, so
that they can be shared with other commands.

=cut

1;
