#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ABSPgrid;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ABSPgrid - implement the C<tsh> ABSPgrid command

=head1 SYNOPSIS

  my $command = new TSH::Command::ABSPgrid;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ABSPgrid is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to create a ratings grid for submitting scores
to the ABSP Ratings Officer.
EOF
  $this->{'names'} = [qw(absp abspgrid)];
  $this->{'argtypes'} = [qw(Divisions)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();
  my $save_ntf = $config->Value('no_text_files');
  $config->Value('no_text_files', 0);
  for my $dp (@_) {
    my $logp = new TSH::Log($tournament, $dp, 'grid', '', {'notitle' => 1, 'noconsole' => 1});
    $logp->Write('','<tr class=top1>');
    my $nrounds = $dp->MostScores();
    if ($nrounds != $dp->LeastScores()) {
      $tournament->TellUser('eabspr', $nrounds);
      }
    my $s = sprintf("Results grid after round %d\n\nName              |",
      $nrounds);
    $logp->Write($s, "<td><pre>\n$s");
    my $hrule = '-' x 18 . '|';
    for my $r (1..$nrounds) {
      $s = sprintf("Rnd%2d|", $r);
      $logp->Write($s, $s);
      $hrule .= '-----|';
      }
    $hrule .= "\n";
    $s = "\n$hrule";
    $logp->Write($s, $s);
    
    for my $p ($dp->Players()) {
      next unless defined $p;
      my ($surname, $given) = split(/, */, $p->Name(), 2);
      if (!defined $given) {
	TSH::Utility::Error "Can't find a comma separating surname from given name: $p->{'name'}\n";
	return 0;
        }
      my $line1 = sprintf("%3d: %-13.13s", $p->ID(), $given);
      my $line2 = sprintf("%4g %-13.13s", $p->Wins(), $surname);
      for my $r (1..$nrounds) {
	my $r0 = $r - 1;
	$line1 .= '|'; $line2 .= '|';
	{
	  my $p12 = $p->First($r0);
	  $line1 .= $p12 == 1 ? 'S' : $p12 == 2 ? 'R' : ' ';
	}
        my $oppid = $p->OpponentID($r0);
	if ($oppid) {
	  my $opp = $p->Opponent($r0);
	  my $os = $p->OpponentScore($r0);
	  my $ms = $p->Score($r0);
	  if ((defined $ms) && defined $os) {
	    my $thisSpread = $ms - $os;
	    if ($thisSpread) {
	      $line1 .= sprintf("%+4d", $thisSpread);
	      $line2 .= $thisSpread > 0 ? '+ ' : '- ';
	      }
	    else {
	      $line1 .= '   =';
	      $line2 .= '= ';
	      }
	    }
	  else { # don't have both $ms and $os
	    $line1 .= '    ';
	    $line2 .= '= ';
	    }
	  $line2 .= sprintf("%3d", $oppid);
	  }
	else { # no $oppid
	  $line1 .= "   =";
	  $line2 .= "=   X";
	  }
        } # for $r0
      $s = "$line1|\n$line2|\n$hrule";
      $logp->Write($s, $s);
      } # for my $p
    $logp->Write('', "</pre></td>");
    $logp->Close();
    }
  $tournament->TellUser('iabspok');
  $config->Value('no_text_files', $save_ntf);
  }

=back

=cut

=head1 BUGS

None known.

=cut

1;
