#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::AVErages;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::AVErages - implement the C<tsh> AVErages command

=head1 SYNOPSIS

  my $command = new TSH::Command::AVErages;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::AVErages is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list average scores for each player in a division.
EOF
  $this->{'names'} = [qw(ave averages)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;

  my $logp = new TSH::Log( $tournament, $dp, 'averages', undef,
    {'title' => 'Division ' . $dp->Name() . ' Average Scores'});
  my @rows;
  for my $p ($dp->Players()) {
    my $sum;
    my $count;
    for my $r0 (0..$p->CountScores()-1) {
      next unless $p->Opponent($r0);
      $sum += $p->Score($r0);
      $count++;
      }
    push(@rows, [$count ? $sum/$count : 0, $p]);
    }

  $logp->ColumnClasses([ qw(rank score wl spread name) ]);
  $logp->ColumnTitles(
    {
    'text' => [ qw(Rank Average Won-Lost Spread Player) ],
    'html' => [ 'Rank', 'Average', 'Won-Lost', 'Spread', 'Player' ],
    }
    );
  my $lastave = -1;
  my $i = 0;
  my $rank = 0;
  for my $row (sort { $b->[0] <=> $a->[0] || $a->[1]->Name() cmp $b->[1]->Name() } @rows) {
    $i++;
    my $p = $row->[1];
    if ($row->[0] != $lastave) {
      $rank = $i;
      $lastave = $row->[0];
      }
    $logp->WriteRow([
      $rank,
      sprintf("%7.2f", $row->[0]),
      sprintf("%4.1f-%4.1f", $p->Wins(), $p->Losses()),
      sprintf("%+d", $p->Spread()),
      $p->TaggedName(),
      ]);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

