#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundRATings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundRATings - implement the C<tsh> RoundRATings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundRATings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundRATings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub RenderTable ($$$$);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display ratings estimates based on standings
in a given round or rounds within a division.  If you specify
just one round, the results will be displayed on the screen.
If you specify a range of rounds (e.g., 4-7), the results will
not be shown on your screen but report files will be updated.
EOF
  $this->{'names'} = [qw(rrat roundratings)];
  $this->{'argtypes'} = [qw(RoundRange Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item RenderTable($dp, $r0, $optionsp, $noconsole)

Render the rows of the ratings table. This code is independent of
rating system.

=cut

sub RenderTable ($$$$) {
  my $dp = shift;
  my $r0 = shift;
  my $optionsp = shift;
  my $noconsole = shift;
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $noshowlast = $config->Value('no_show_last') || $r0 < 0;
  my $showlastplayername = $config->Value('show_last_player_name');
  my $spreadentry = $config->Value('entry') eq 'spread';
  my $trackfirsts = $config->Value('track_firsts');
  my $is_capped = $config->Value('standings_spread_cap');

  my $has_classes = $dp->Classes() || defined $dp->Player(1)->Class();
  my $logp = new TSH::Log($tournament, $dp, 'ratings', $r0+1, {'titlename' => $optionsp->{'titlename'}, 'noconsole' => $noconsole});
  my (@classes) = qw(rank wl spread);
  my (@html_titles) = (qw(Rank Won-Lost Spread));
  my (@text_titles) = (qw(Rnk Won-Lost Spread));
  for my $key (qw(oldr newr difr)) {
    if ($optionsp->{"${key}h"}) {
      push(@html_titles, $optionsp->{"${key}h"});
      push(@text_titles, $optionsp->{"${key}t"});
      push(@classes, 'rating');
      }
    };
  if ($has_classes) {
    push(@classes, 'pclass');
    push(@html_titles, 'Class');
    push(@text_titles, 'C');
    }
  push(@classes, 'name');
  push(@html_titles, 'Player');
  push(@text_titles, 'Player');
  unless ($noshowlast) {
    push(@classes, qw(last));
    push(@html_titles, 'Last');
    push(@text_titles, 'Last');
    }
  if ($dp->LastPairedRound0() > $r0) {
    push(@classes, 'name');
    push(@html_titles, 'Next');
#   push(@text_titles, 'Next Opponent'); # too wide
    }
  $logp->ColumnClasses(\@classes);
  $logp->ColumnTitles(
    {
    'text' => \@text_titles,
    'html' => \@html_titles,
    }
    );
  my @ps;
  if ($is_capped) {
    $dp->ComputeCappedRanks($r0);
    @ps = TSH::Player::SortByCappedStanding($r0, $dp->Players());
    }
  else {
    $dp->ComputeRanks($r0);
    @ps = TSH::Player::SortByStanding($r0, $dp->Players());
    }
  my $i = 0;
  for my $p (@ps) {
    next unless $p->Active();
#   if ($config::pair_page_break && $i && ($i) % $config::pair_page_break == 0) { $logp->Write('',qq(</table><table class=ratings align=center cellspacing=0 style="page-break-before:always"><tr>$headings</tr>)); }
    my $rating = $p->Rating();
    my $newr = $p->NewRating();
    $i++;
    my $rank = $is_capped ? $p->RoundCappedRank($r0) : $p->RoundRank($r0);
    my (@fields) = (
      $rank,
      sprintf("%.1f-%.1f", $p->RoundWins($r0), $p->RoundLosses($r0)),
      sprintf("%+d ", $is_capped ? $p->RoundCappedSpread($r0) : $p->RoundSpread($r0)),
      );
    if ($optionsp->{'oldrh'}) {
      push(@fields, $p->Rating());
      }
    if ($optionsp->{'newrh'}) {
      push(@fields, $p->NewRating());
      }
    if ($optionsp->{'difrh'}) {
      my $rating = $p->Rating();
      my $newr = $p->NewRating();
      push(@fields, $rating ? sprintf("%+d", $newr-$rating) : '');
      }
    push(@fields, $p->Class()) if $has_classes;
    push(@fields, $p->TaggedName());
    my (@text_fields) = @fields;
    unless ($noshowlast) {
      my $s = '';
      my $opp = $p->Opponent($r0);
      if ($opp) {
	my $oname = $showlastplayername ? $opp->TaggedName() : $opp->FullID();
	my $ms = $p->Score($r0);
	if (defined $ms) {
	  my $os = $p->OpponentScore($r0);
	  $os = 0 unless defined $os; # possible if pairing data bad
	  my $p12 = $p->First($r0);
	  $s = 
	    ($trackfirsts ? substr('B12??', $p12,1) : '')
	    .($spreadentry ? '' : ($ms > $os ? 'W' : $ms < $os ? 'L' : 'T'));
	  $s .= ':' if $s;
	  $s .=($spreadentry ? sprintf("%+d", $ms-$os) : "$ms-$os")
	    .':'
	    . $oname;
	  }
	else {
	  $s = 'pending:' . $oname;
	  }
        }
      else {
	if (defined $p->OpponentID($r0)) {
	  $s = 'bye';
	  }
	else {
	  $s = 'unpaired';
	  }
        }
      push(@fields, $s);
      push(@text_fields, $s);
      }
    # next round
    if (my $opp = $p->Opponent($r0+1)) {
      push(@fields, $opp ? $opp->FullID() : 'bye');
      }
    $logp->WriteRow(\@text_fields, \@fields);
    }
  $logp->Close();
  }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($firstr1, $lastr1, $dp) = @_;
  my $firstr0 = $firstr1 - 1;
  my $lastr0 = $lastr1 - 1;
  my $rating_system = $tournament->Config()->Value('rating_system');

  $dp->CheckRoundHasResults($lastr0) or return 0;

  my (%options_by_rating_system) = (
    'absp' => {
      'titlename' => 'Ratings',
      'oldrh'=>'Input<br>Rating',
      'newrh'=>'Tourn.<br>Rating',
      'difrh'=>'Rating<br>Change',
      'difrt'=>'+-R',
      'oldrt'=>'InR',
      'newrt'=>'ToR',
      },
    'nsa' => {
      'titlename' => 'Ratings',
      'oldrh'=>'Old<br>Rating',
      'newrh'=>'New<br>Rating',
      'difrh'=>'Rating<br>Change',
      'difrt'=>'+-R',
      'oldrt'=>'OldR',
      'newrt'=>'NewR',
      },
    'nsa lct' => {
      'titlename' => 'Ratings',
      'oldrh'=>'Old<br>Rating',
      'newrh'=>'New<br>Rating',
      'difrh'=>'Rating<br>Change',
      'difrt'=>'+-R',
      'oldrt'=>'OldR',
      'newrt'=>'NewR',
      },
    'none' => {
      'titlename' => 'Rankings',
      },
    );
  
  for my $r1 ($firstr1..$lastr1-1) {
    my $r0 = $r1 - 1;
    $tournament->TellUser('irratok', $r1);
    $dp->ComputeRatings($r0);
    RenderTable $dp, $r0, $options_by_rating_system{lc $rating_system}, 1;
    }
  $dp->ComputeRatings($lastr0);
  RenderTable $dp, $lastr0, $options_by_rating_system{lc $rating_system}, 0;

  return 0;
  }

=back

=cut

1;
