#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundStandings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundStandings - implement the C<tsh> RoundStandings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundStandings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundStandings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings as of a specific round.
EOF
  $this->{'names'} = [qw(rs roundstandings)];
  $this->{'argtypes'} = [qw(BasedOnRound Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($r1, $dp) = @_;

  my $r0 = $r1 - 1;
  $dp->CheckRoundHasResults($r0) or return 0;
  my $has_classes = $dp->Classes();
  my $logp = new TSH::Log($tournament, $dp, 'standings', $r1);
  my $headings = <<'EOF';
<tr class=top1>
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
EOF
  $logp->Write("Rank  Won-Lost Spread ", '');
  my $text_data_format = "%4d %4.1f-%4.1f %+5d  ";
  my $html_data_format = <<'EOF';
<tr style="page-break-inside:avoid">
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
EOF
  if ($has_classes) {
    $logp->Write('C ',  '');
    $headings .= "<th class=pclass>Class</th>\n";
    $text_data_format .= "%1s ";
    $html_data_format .= "<td class=pclass>%s</td>\n";
    }
  $logp->Write("Rtng Player\n\n", '');
  $headings .= <<'EOF';
<th class=rating>Rating</th>
<th class=name>Name</th>
EOF
  $html_data_format .= <<'EOF';
<td class=rating>%d</td>
<td class=name>%s</td>
EOF
  $text_data_format .= "%4d %s\n";
  $logp->Write('', "$headings</tr>");

  my $dname = $dp->Name();

  $dp->ComputeRanks($r0);
  my $i = 0;
  for my $p (TSH::Player::SortByStanding $r0, $dp->Players()) {
    next unless $p->Active();
    my $rank = $p->RoundRank($r0);
    my $wins = $p->RoundWins($r0);
    my $losses = $p->RoundLosses($r0);
    my $spread = $p->RoundSpread($r0);
    # if ($config::pair_page_break && $i && ($i) % $config::pair_page_break == 0) { $logp->Write('',qq(</table><table class=ratings align=center cellspacing=0 style="page-break-before:always"><tr>$headings</tr>)); }
    $i++;
    my (@fields) = (
      $p->RoundRank($r0),
      $p->RoundWins($r0),
      $p->RoundLosses($r0),
      $p->RoundSpread($r0),
      );
    push(@fields, $p->Class()) if $has_classes;
    push(@fields, $p->Rating(), $p->TaggedName());
    $logp->Write(
      sprintf($text_data_format, @fields),
      sprintf($html_data_format, @fields));
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

Much of this should be moved to TSH::Report, and common elements
shared with RoundRATings.pm separated out.

Should use new TSH::Log table code.

=cut

1;

