# ratings2.pl - Additional Perl library of routines for manipulating NSA Elo ratings

# $Id: ratings2.pl,v 1.3 2005/10/05 12:37:24 jjc Exp jjc $
#
# $Log: ratings2.pl,v $
# Revision 1.3  2005/10/05 12:37:24  jjc
# minor bug fixes
#
# Revision 1.2  2005/01/06 22:56:56  jjc
# ! cleanup
#

# Copyright (C) 1997-2005 by John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

=pod

=head1 NAME

ratings2.pl - library of routines for calculating Scrabble ratings

=head1 SYNOPSIS

  ratings2::UseClubMultipliers(1);
  ratings2::CalculateSplitRatings(\@psp, $nrounds, \%key_map);

=head1 ABSTRACT

  This library contains code shared by the tourney.pl and tsh.pl
  scripts.  It is overburdened by legacy compatibility code which
  is gradually being pruned out.  Most routines are for internal
  use, and may change without notice in future releases.  The
  next major release of this library will likely be in the form
  of a Perl module with a more carefully delineated demarcation
  between internal and public routines.
  
  Many routines manipulate lists of player data hashes.  For the
  sake of generality, the keys used to find data in those hashes
  are passed in their own hashes.  Temporary data may be stored in
  player data hashes under keys beginning with 'xrat_'.

=cut

use warnings;
use strict;

package ratings2;

use Carp qw(confess);

our $gqAccelerationBonuses = 1;
our $gqClubMultipliers = 0;
# The NSA uses a noniterative approximation of performance ratings.
our $gqUseHomanPR = 1;
our $gMaximumIterations = 25;
# The NSA ratings code contains a bug affecting players whose rating
# drops from exactly 1800 or 2000.
our $gqUseHomanBoundaryBug = 1;

our $gVersion = '1.4';

sub CalculateAccelerationBonus ($$$$$);
sub CalculateExcess ($$$$$);
sub CalculateHomanPR ($$);
sub CalculateRatings ($$$$$$);
sub CalculateSegmentRatings ($$$$);
sub CalculateSegmentStandings ($$$$);
sub CalculateSplitRatings ($$$);
sub ConvertExcessToPoints ($$$);
sub CopyField ($$$);
sub CountAllEWins ($$$);
sub CountEWins ($$$$);
sub MakeRoundToSplit (@);
sub MakeSplits ($);
sub Multiplier ($$);
sub RateNewcomers ($$$$);
sub RateNewcomersCorrectly ($$$$$);
sub RateNewcomersHoman ($$$$$);
sub RateVeterans ($$$$);
sub SetFeedbackToZero ($);
sub SetMaximumIterations ($);
sub SetupNewcomerPR ($$$$$);
# sub trunc ($);
sub UseAccelerationBonuses ($);
sub UseClubMultipliers ($);

=head1 SUBROUTINES

=over 4

=item CalculateAccelerationBonus \@ps, $pn, $key_usage_p

Internal use.
Add on acceleration bonuses to the given player, based on how many
points they have gained in how many rounds.  Keys used in player hashes:

newr: post-segment ratings

oldr: pre-segment ratings

rgames: rated games in segment

=cut

sub CalculateAccelerationBonus ($$$$$) {
  my $psp = shift;
  my $pn = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $key_usage_p = shift;
  my $p = $psp->[$pn];
  my $oldr_k = $key_usage_p->{'oldr'};
  my $newr_k = $key_usage_p->{'newr'};
  my $rgames_k = $key_usage_p->{'rgames'};
  my $pairings_k = $key_usage_p->{'pairings'};

  my $accel = $p->{$newr_k} - $p->{$oldr_k} - 5 * $p->{$rgames_k};
  return unless $accel > 0;
# warn "acceleration: $accel\n" if $p->{'name'} =~ /GREMAUD|MCKENZIE/;
  $p->{$newr_k} += $accel; 
  if (my $ta_k = $key_usage_p->{'totalacc'}) {
    $p->{$ta_k} = ($p->{$ta_k}||0) + $accel;
    }
  # calculate feedback
  $accel = $accel/20; # 2002
  my $tf_k = $key_usage_p->{'totalfeed'};
  for my $r0 ($first_round_0..$last_round_0) {
    my $o = $p->{$pairings_k}[$r0];
#   die "r0=$r0 p=$p->{'name'}: @{$p->{$pairings_k}}." unless defined $o;
    next if $o == -1;
    next unless $psp->[$o]{$oldr_k}; # 2002
    $psp->[$o]{'xrat_feed'} += $accel;
    if ($tf_k) {
      $psp->[$o]{$tf_k} = ($psp->[$o]{$tf_k}||0) + $accel;
      }
    }
  }

=item CalculateExcess \@ps, $pn, $first_round_0, $last_round_0, $key_usage_p

Internal use.
Calculate how many games over their expectation the given player won
over the given range of rounds.  Keys used in player hashes:

ewins: earned wins in segment

pairings: 0-based list of opponents (-1 indicates bye)

=cut

sub CalculateExcess ($$$$$) {
  my $psp = shift;
  my $pn = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $key_usage_p = shift;
  my $p = $psp->[$pn];

  my $excess = $p->{$key_usage_p->{'ewins'}};
  my $pairings_k = $key_usage_p->{'pairings'};
  for my $r ($first_round_0..$last_round_0) {
    my $on = $p->{$pairings_k}[$r];
    next if (!defined $on) || $on < 0;
    $excess -= 
      &main::outcome_cached($p->{'xrat_effr'}-$psp->[$on]->{'xrat_effr'});
    }
  warn "excess: $excess\n" if $p->{'name'} =~ /GREMAUD|MCKENZIE/;
  return $excess;
  }

=item $pr = CalculateHomanPR $win_ratio, $average_opponent_rating;

Public use.
Calculate the Homan pseudo-performance rating.

=cut

sub CalculateHomanPR ($$) {
  my $wr = shift;
  my $aor = shift;
  if ($wr == 0.5) { return int(0.5+$aor); }
  my $low = 1;
  my $high = 700;
  my $target;
  if ($wr > 0.5) { $target = $wr; }
  else { $target = 1 - $wr; }
  while ($high - $low > 1) {
    my $mid = int(($low+$high)/2);
    if (&main::outcome_cached($mid) > $target) 
      { $high = $mid; }
    else 
      { $low = $mid; }
    }
  return int($aor+0.5) + ($wr > 0.5 ? $high : -$high);
  }

=item CalculateRatings $players, $oldr_key, $first_round, $newr_key, $last_round, $ewins_key;

Deprecated in favour of CalculateSplitRatings.

=cut

sub CalculateRatings ($$$$$$) {
  my $psp = shift; # list of players
  my $old_k = shift; # player hash key for old rating
  my $first_round_1 = shift; # first 1-based round to rate
  my $new_k = shift; # player hash key for new rating
  my $last_round_1 = shift; # last 1-based round to rate
  my $ewins_k = shift; # player hash key for earned wins
  my $first_round_0 = $first_round_1 - 1;
  my $last_round_0 = $last_round_1 - 1;

  if ($last_round_0 < $first_round_0) {
    CopyField $psp, $new_k, $old_k;
    return;
    }
  CountEWins $psp, $first_round_0, $last_round_0, {
    'ewins' => 'xrat_ewins',
    'rgames' => 'xrat_rgames',
    # remaining values are assumed by the old interface
    'pairings' => 'opps', 
    'scores' => 'scores', 
    };
  CalculateSegmentRatings $psp, $first_round_0, $last_round_0, {
    'ewins' => 'xrat_ewins',
    'newr' => $new_k,
    'oldr' => $old_k,
    'rgames' => 'xrat_rgames',
    # remaining values are assumed by the old interface
    'id' => 'id', 
    'lifeg' => 'totalg', 
    'pairings' => 'opps', 
    'scores' => 'scores', 
    };
  }

=item CalculateSegmentRatings \@ps, $first_round_0, $last_round_0, $key_usage_p

Internal use.
Calculate ratings for all players for the split segment covering the
designated 0-based rounds.  Player hash key usage:

ewins: earned wins in segment

id: player id (0-based index in \@ps)

lifeg: lifetime games played

newr: post-segment ratings

oldr: pre-segment ratings

pairings: 0-based list of opponents (-1 indicates bye)

rgames: rated games in segment

scores: 0-based list of scores

=cut

sub CalculateSegmentRatings ($$$$) { 
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $key_usage_p = shift;
  my $oldr_k = $key_usage_p->{'oldr'};
  my $newr_k = $key_usage_p->{'newr'};

  SetFeedbackToZero $psp;
  CopyField $psp, 'xrat_effr', $oldr_k;
  RateNewcomers $psp, $first_round_0, $last_round_0, $key_usage_p;
  RateVeterans $psp, $first_round_0, $last_round_0, $key_usage_p;
# for my $p (@$psp) { warn "$p->{'name'} $p->{$key_usage_p->{'newr'}}\n" if $p->{$key_usage_p->{'newr'}}; }

  # add on feedback
  for my $p (@$psp) { 
    warn "feedback: $p->{'xrat_feed'}\n" if $p->{'name'} =~ /GREMAUD|MCKENZIE/;
    $p->{$newr_k} += $p->{'xrat_feed'}; 
    }

  # 2002: round now
  for my $p (@$psp) { 
    warn "rounding: $p->{$newr_k}\n" if $p->{'name'} =~ /GREMAUD|MCKENZIE/;
    $p->{$newr_k} = int(0.5 + $p->{$newr_k});
    }

  if ($gqClubMultipliers) {
    for my $p (@$psp) {
      next unless $p->{$oldr_k};
      my $diff = $p->{$newr_k} - $p->{$oldr_k};
      if ($diff > 50) { $p->{$newr_k} = $p->{$oldr_k} + 50; }
      elsif ($diff < -50) { $p->{$newr_k} = $p->{$oldr_k} - 50; }
      }
    }
  }

=item CalculateSegmentStandings \@ps, $first_round_0, $last_round_0, $key_usage_p

Public use.
Calculate cume and wins for all players for the split segment covering the
designated 0-based rounds.  Player hash key usage:

cume: cume in segment

games: games in segment

pairings: 0-based list of opponents (-1 indicates bye)

scores: 0-based list of scores

wins: wins in segment (including unearned ones (byes))

=cut

sub CalculateSegmentStandings ($$$$) { 
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $key_usage_p = shift;
  my $cume_k = $key_usage_p->{'cume'};
  my $games_k = $key_usage_p->{'games'};
  my $pairings_k = $key_usage_p->{'pairings'};
  my $scores_k = $key_usage_p->{'scores'};
  my $wins_k = $key_usage_p->{'wins'};

  for my $p (@$psp) {
    my $opps = $p->{$pairings_k};
    my $scores = $p->{$scores_k};
    my $this_last_round_0 = $#$scores;
    $this_last_round_0 = $last_round_0 if $this_last_round_0 > $last_round_0;
    my $cume = 0;
    my $games = 0;
    my $wins = 0;
    for my $r0 ($first_round_0..$this_last_round_0) {
      $games++;
      my $ms = $scores->[$r0];
      my $on = $opps->[$r0];
      if ($on < 0) { $cume += $ms; $wins++ if $ms > 0; next; }
      my $os = $psp->[$on]{$scores_k}[$r0];
      unless (defined $os) { printf STDERR "In round %d player %d has no score but opponent does.\n", $r0+1, $on+1; next; }
      my $spread = $ms - $os;
      $cume += $spread;
      $wins += (($spread <=> 0) + 1)/2;
      }
    $p->{$cume_k} = $cume;
    $p->{$games_k} = $games;
    $p->{$wins_k} = $wins;
    }
  }

=item @splitlist = CalculateSplitRatings (\@players, $rounds, \%key_usage)

Public use.
@players is a 0-based list of players, each of which is a reference
to a hash.
$rounds specifies how many rounds the tournament lasts.
%key_usage specifies how keys are used in the player hashes, and
must include values corresponding to the following keys:

ewins: basename for keys in which to store earned wins by split segment

lifeg: lifetime games played

newr: posttournament rating

oldr: pretournament rating

pairings: 0-based list of opponents (-1 indicates bye)

rgames: basename for keys in which to store rated games by split segment

scores: 0-based list of scores

splitr: basename for keys in which to store intermediate ratings

It may also include the following keys:

totalacc: total acceleration points awarded

totalfeed: total feedback points awarded

Returns list of 0-based [first_round,last_round] pairs.

=cut

sub CalculateSplitRatings ($$$) {
  my $psp = shift;
  my $nrounds = shift;
  my $key_usage_p = shift;

  my $scores_key = $key_usage_p->{'scores'};
  my @splits = MakeSplits $nrounds;
  my @round_to_split = MakeRoundToSplit @splits;
  CountAllEWins $psp, \@splits, $key_usage_p;
  for my $si (0..$#splits) {
    my $split = $splits[$si];
    my $from = $si == 0 ? $key_usage_p->{'oldr'} 
      : "$key_usage_p->{'splitr'}$si";
    my $to = $si == $#splits ? $key_usage_p->{'newr'}
      : ($key_usage_p->{'splitr'}.($si+1));
    CalculateSegmentRatings $psp, $split->[0], $split->[1], {
      'ewins' => "$key_usage_p->{'ewins'}$si",
      'id' => $key_usage_p->{'id'},
      'lifeg' => $key_usage_p->{'lifeg'},
      'newr' => $to,
      'oldr' => $from,
      'pairings' => $key_usage_p->{'pairings'},
      'rgames' => "$key_usage_p->{'rgames'}$si",
      'scores' => $key_usage_p->{'scores'},
      'totalacc' => $key_usage_p->{'totalacc'},
      'totalfeed' => $key_usage_p->{'totalfeed'},
      };
    }
  return @splits;
  }

=item ConvertExcessToPoints $p, $excess, $key_usage_p

Internal use.
Convert excess wins to ratings points.  Keys used in player hashes:

lifeg: lifetime games played (read-only)

newr: post-segment rating (updated)

=cut

sub ConvertExcessToPoints ($$$) {
  my $p = shift;
  my $excess = shift;
  my $key_usage_p = shift;
  my $life_k = $key_usage_p->{'lifeg'} or die "need lifeg key";
  my $newr_k = $key_usage_p->{'newr'} or die "need newr key";

  while ($excess) {
    my $starting_rating = $p->{$newr_k};
    my $starting_multiplier = Multiplier $p->{$life_k}, $starting_rating;
#       $r = int(0.5 + $starting_rating + $multiplier * $excess);
    my $r = $starting_rating + $starting_multiplier * $excess; # 2002
# warn "$p->{'fname'}: $excess $starting_rating $r\n";
    # If the whole rating adjustment can be made within one multiplier band
    if ($starting_multiplier == Multiplier($p->{$life_k}, $r)) 
      # make the adjustment and return
      { $p->{$newr_k} = $r; last; }
    # Otherwise, compute just the adjustment to the boundary of the band.
    my $boundary;
    if ($starting_rating < 1800) { $boundary = 1800; }
    elsif ($starting_rating == 1800 && $excess < 0 && $gqUseHomanBoundaryBug) 
      { $p->{$newr_k} = $r; last; }
    elsif ($starting_rating < 2000) {
      $boundary = $excess > 0 ? 2000 : 1799;
      }
    elsif ($starting_rating == 2000 && $excess < 0 && $gqUseHomanBoundaryBug) {
      if ($r < 1799) { $boundary = 1799; }
      else { $p->{$newr_k} = $r; last; }
      }
    else {
      $boundary = 1999;
      }
    $excess -= ($boundary - $starting_rating) / $starting_multiplier;
    $p->{$newr_k} = $boundary;
    }
  }

=item CopyField $hashlistref, $destination_key, $oldr_key;

Public use.
Copies fields in each of a list of hashes.

=cut

sub CopyField ($$$) {
  my $hashlistp = shift;
  my $dst_k = shift;
  my $src_k = shift;
  for my $hashp (@$hashlistp) {
    if (exists $hashp->{$src_k}) { $hashp->{$dst_k} = $hashp->{$src_k}; }
    else { delete $hashp->{$dst_k}; }
    }
  }

=item CountAllEWins \@ps, \@splits, $key_usage_p;

Internal use.
Counts earned wins and rated games for all players.
Keys used in player hashes:

ewins: basename for keys in which to store earned wins by split segment

pairings: 0-based list of opponents (-1 indicates bye)

rgames: basename for keys in which to store rated games by split segment

scores: 0-based list of scores

spreads: basename for keys in which to store cume by split segment

=cut

sub CountAllEWins ($$$) {
  my $psp = shift;
  my $splitsp = shift;
  my $key_usage_p = shift;
  for my $si (0..$#$splitsp) {
    my $split = $splitsp->[$si];
    CountEWins $psp, $split->[0], $split->[1], {
      'ewins' => "$key_usage_p->{'ewins'}$si",
      'rgames' => "$key_usage_p->{'rgames'}$si",
      'pairings' => $key_usage_p->{'pairings'},
      'scores' => $key_usage_p->{'scores'},
      };
    }
  }

=item CountEWins \@ps, $first_round0, \$last_round0, $key_usage_p;

Internal use.
Count earned wins and rated games for each player over the given
range of rounds.  
Keys used in player hashes:

ewins: earned wins in segment

pairings: 0-based list of opponents (-1 indicates bye)

rgames: rated games in segment

scores: 0-based list of scores

=cut

sub CountEWins ($$$$) {
  my $psp = shift;
  my $first_round0 = shift;
  my $last_round0 = shift;
  my $key_usage_p = shift;
  my $ewins_k = $key_usage_p->{'ewins'};
  my $rgames_k = $key_usage_p->{'rgames'};
  my $opps_k = $key_usage_p->{'pairings'};
  my $scores_k = $key_usage_p->{'scores'};
  for my $pn (0..$#$psp) {
    my $p = $psp->[$pn];
    $p->{$ewins_k} = 0;
    $p->{$rgames_k} = 0;
    my $oppsp = $p->{$opps_k};
    my $scoresp = $p->{$scores_k};
    my $this_last_0 = $#$scoresp;
    $this_last_0 = $last_round0 if $last_round0 < $this_last_0;
    for my $r0 ($first_round0..$this_last_0) {
      my $on = $oppsp->[$r0];
      unless (defined $on) {
	my $pn1 = $pn+1;
	my $r1 = $r0+1;
	warn "Player $pn1 has no opponent in round $r1\n";
	next;
        }
      next unless $on >= 0;
      my $ms = $scoresp->[$r0];
      next unless defined $ms;
      my $os = $psp->[$on]{$scores_k}[$r0];
      next unless defined $os;
      $p->{$ewins_k} += (($ms<=>$os)+1)/2;
      $p->{$rgames_k} ++;
      }
    }
  }

=item @round_to_split = MakeRoundToSplit @splits;

Internal use.
Make a list mapping 0-based round numbers to 0-based split segment indices.

=cut

sub MakeRoundToSplit (@) {
  my @rts;
  for my $si (0..$#_) {
    my $sp = $_[$si];
    for my $r ($sp->[0]..$sp->[1]) {
      $rts[$r] = $si;
      }
    }
  return @rts;
  }

=item @splits = MakeSplits $nrounds;

Internal use.
Return a list of 0-based first and last rounds for each segment
of a tournament that must be split-rated under NSA rules.

=cut

sub MakeSplits ($) {
  my $nrounds = shift;
  my $nrounds0 = $nrounds - 1;
  if ($nrounds0 < 16) {
    return ([0,$nrounds0]);
    }
  if ($nrounds0 < 36) {
    my $s1 = int($nrounds0/2);
    return ([0,$s1],[$s1+1,$nrounds0]);
    }
  my $s1 = int($nrounds0/3);
  my $s2 = int((2*$nrounds0+1)/3);
  return ([0,$s1],[$s1+1,$s2],[$s2+1,$nrounds0]);
  }

=item $multiplier = Multiplier $games, $rating;

Return the appropriate multiplier to use for a player who has
played $games games and has a pretournament rating of $rating.

=cut

sub Multiplier ($$) { 
  my $games = shift;
  my $rating = shift;
  $games < 0 ? 0 : 
    ($games < 50
      ? $rating < 1800 ? 30 : $rating < 2000 ? 24 : 15
      : $rating < 1800 ? 20 : $rating < 2000 ? 16 : 10)
  / ($gqClubMultipliers ? 3 : 1);
  }

=item RateNewcomers \@ps, $first_round_0, $last_round_0, $key_usage_p;

Internal use.
Determine initial ratings for all unrated players, based on results
from the given rounds.
Keys used in player hashes:

id: player id (0-based index in \@ps)

newr: post-segment ratings

oldr: pre-segment ratings

ewins: earned wins in segment

pairings: 0-based list of opponents (-1 indicates bye)

rgames: rated games in segment

scores: 0-based list of scores

=cut

sub RateNewcomers ($$$$) {
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $key_usage_p = shift;
  if (my @unrated = grep { !$_->{$key_usage_p->{'oldr'}} } @$psp) {
    if ($gqUseHomanPR) {
      RateNewcomersHoman $psp, $first_round_0, $last_round_0, 
        \@unrated, $key_usage_p;
      }
    else {
      RateNewcomersCorrectly $psp, $first_round_0, $last_round_0,
        \@unrated, $key_usage_p;
      }
    # new rating and effective rating are last iterated performance rating
    CopyField \@unrated, 'xrat_effr', 'xrat_curr';
    CopyField \@unrated, $key_usage_p->{'newr'}, 'xrat_curr';
    }
  }

=item RateNewcomersCorrectly \@ps, $first_round_0, $last_round_0, $key_usage_p;

Internal use.
Determine initial ratings for all unrated players (saved in key
xrat_newpr), based on results
from the given rounds, using iterated performance ratings.
Keys used in player hashes:

id: player id (0-based index in \@ps)

oldr: pre-segment ratings

ewins: earned wins in segment

pairings: 0-based list of opponents (-1 indicates bye)

rgames: rated games in segment

scores: 0-based list of scores

=cut

sub RateNewcomersCorrectly ($$$$$) {
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $unratedp = shift;
  my $key_usage_p = shift;
  my $id_k = $key_usage_p->{'id'};
  SetupNewcomerPR $psp, $first_round_0, $last_round_0, $unratedp, $key_usage_p;
  
  # keep updating 'xrat_curr' until we attain stability
  for (my $i = 0; $i < $gMaximumIterations; $i++) {
    my $changed = 0;
    for my $p (@$unratedp) {
#	my $r = &main'search4("ipr$p->{$id_k}", 0, 3000, 0.1);
      my $r = &main::search("ipr$p->{$id_k}", 0, 3000);
      if ($r != $p->{'xrat_curr'}) {
	$p->{'xrat_newpr'} = $r;
	$changed = 1;
	}
      }
    CopyField $unratedp, 'xrat_curr', 'xrat_newpr';
    last unless $changed;
    }
  }

=item RateNewcomersHoman \@ps, $first_round_0, $last_round_0, $key_usage_p;

Internal use.
Determine initial ratings for all unrated players, based on results
from the given rounds, using a traditional NSA approximation.
Keys used in player hashes:

ewins: earned wins in segment

id: player id (0-based index in \@ps)

oldr: pre-segment ratings

pairings: 0-based list of opponents (-1 indicates bye)

rgames: rated games in segment

scores: 0-based list of scores

=cut

# calculate ratings for all previously unrated players, using
# a crude approximation
sub RateNewcomersHoman ($$$$$) {
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $unratedp = shift;
  my $key_usage_p = shift;
  my $oldr_k = $key_usage_p->{'oldr'};
  my $ewins_k = $key_usage_p->{'ewins'};
  my $pairings_k = $key_usage_p->{'pairings'};
  my $rgames_k = $key_usage_p->{'rgames'};
  my $scores_k = $key_usage_p->{'scores'};

  my $i = 0;
  my $changed = 1;
  for my $p (@$psp) {
    if ($p->{$oldr_k}) 
      { $p->{'xrat_curr'} = $p->{$oldr_k}; }
    else 
      { $p->{'xrat_curr'} = 1500; }
    }
  while ($changed && $i++ < $gMaximumIterations) {
    $changed = 0;
    for my $p (@$unratedp) {
      my $rgames = $p->{$rgames_k};
      next unless $rgames > 0;
      my $this_last_0 = $#{$p->{$scores_k}};
      $this_last_0 = $last_round_0 if $last_round_0 < $this_last_0;
      my $sumor = 0;
      my $nor = 0;
      my $maxor = 0;
      for my $r ($first_round_0..$this_last_0) {
	my $on = $p->{$pairings_k}[$r];
	next if $on == -1;
	my $op = $psp->[$on];
	die "Player #".($on+1)." has no current rating.\n" 
	  unless defined $op->{'xrat_curr'};
	$sumor += $op->{'xrat_curr'};
	$maxor = $op->{$oldr_k}+1 if $op->{$oldr_k}+1 > $maxor;
	$nor++;
	}
      my $aor = $nor ? $sumor/$nor : 500;
      my $w = $p->{$ewins_k};
      my $realwr = $w/$rgames;
      my $fakewr = $realwr;
      $fakewr = $fakewr == 0 ? 0.05 : $fakewr == 1 ? 0.95 : $fakewr;
      my $homanpr = CalculateHomanPR $fakewr, $aor; 
      my $max_allowed = int(0.5+$maxor+400*$realwr);
      $homanpr = $max_allowed if $homanpr > $max_allowed;
      if ($homanpr != $p->{'xrat_curr'}) {
	$p->{'xrat_newpr'} = $homanpr;
	$changed = 1;
	}
      }
    for my $p (@$unratedp) {
      next unless defined $p->{'xrat_newpr'};
      $p->{'xrat_newpr'} = 500 if $p->{'xrat_newpr'} < 500;
      $p->{'xrat_curr'} = $p->{'xrat_newpr'};
      }
    }
  }

=item RateVeterans \@ps, $first_round_0, $last_round_0, $key_usage_p

Calculate new ratings for all players who have pre-segment ratings,
based on results of the given rounds.  Keys used in player hashes:

lifeg: lifetime games played

newr: post-segment ratings

oldr: pre-segment ratings

rgames: rated games in segment

scores: 0-based list of scores

=cut

sub RateVeterans ($$$$) {
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $key_usage_p = shift;
  my $newr_k = $key_usage_p->{'newr'};
  my $oldr_k = $key_usage_p->{'oldr'};
  my $scores_k = $key_usage_p->{'scores'};
  for my $p (@$psp) {
    $p->{$newr_k} = $p->{$oldr_k} if $p->{$oldr_k};
    }
  for my $pn (0..$#$psp) {
    my $p = $psp->[$pn];
    next unless $p->{$oldr_k};
    my $this_last_0 = $#{$p->{$key_usage_p->{'scores'}}};
    $this_last_0 = $last_round_0 if $last_round_0 < $this_last_0;
    next unless $first_round_0 <= $this_last_0;
    my $excess = CalculateExcess $psp, $pn, $first_round_0, $this_last_0,
      $key_usage_p;
    ConvertExcessToPoints $p, $excess, $key_usage_p;
    if ($gqAccelerationBonuses) {
      CalculateAccelerationBonus $psp, $pn, $first_round_0, $this_last_0,
        $key_usage_p;
      }
    }
  }

sub SetFeedbackToZero ($) {
  my $psp = shift;
  for my $p (@$psp) { $p->{'xrat_feed'} = 0; }
  }

sub SetMaximumIterations ($) { $gMaximumIterations = shift; }

=item SetupNewcomerPR $psp, $first_round_0, $last_round_0, \@unrated, $key_usage_p;

Give each player an initial rating equal to the average of their opps
and set up their performance rating calculation sub.
Keys used in player hashes:

ewins: earned wins in segment

oldr: pre-segment ratings

pairings: 0-based list of opponents (-1 indicates bye)

rgames: rated games in segment

scores: 0-based list of scores

=cut

sub SetupNewcomerPR ($$$$$) {
  my $psp = shift;
  my $first_round_0 = shift;
  my $last_round_0 = shift;
  my $unratedp = shift;
  my $key_usage_p = shift;
  my $ewins_k = $key_usage_p->{'ewins'};
  my $id_k = $key_usage_p->{'id'};
  my $oldr_k = $key_usage_p->{'oldr'};
  my $pairings_k = $key_usage_p->{'pairings'};
  my $rgames_k = $key_usage_p->{'rgames'};
  my $scores_k = $key_usage_p->{'scores'};
  for my $p (@$unratedp) {
    my $this_last = $#{$p->{$scores_k}};
    $this_last = $last_round_0 if $last_round_0 < $this_last;
    my $sum = 0;
    my $n = 0;
    if ($this_last >= $first_round_0) {
      my $code = "sub main::ipr$p->{$id_k} {";
      for my $r0 ($first_round_0..$this_last) {
	my $on = $p->{$pairings_k}[$r0];
	next if $on < 0;
	my $op = $psp->[$on];
	$sum += $op->{$oldr_k};
	$n++ if $op->{$oldr_k};
	$code .= $op->{$oldr_k} 
	  ?"&main::outcome_cached(\$_[0]-$op->{$oldr_k})+"
	  :"&main::outcome_cached(\$_[0]-\$ps->[$op->{$id_k}]->{'xrat_curr'})+";
	}
      my $w = $p->{$ewins_k};
      if ($w == 0) { $w = $p->{$rgames_k}*0.05; } 
      elsif ($w == $p->{$rgames_k}) { $w = $p->{$rgames_k}*0.95;  }
      $code .= "-$w;} 1;";
      eval $code or die "eval failed for: $code\n";
      }

    if ($n) { $p->{'xrat_curr'} = $sum / $n; }
    else {
      $p->{'xrat_curr'} = 500;
      my $pn1 = $p->{$id_k}+1;
      warn "Player #$pn1 is unrated but did not play any rated players.\n";
      }
    }
  }

# sub trunc ($) { int(0.5+100000*shift)/100000; }

sub UseAccelerationBonuses ($) { 
  $gqAccelerationBonuses = shift;
  if ($gqAccelerationBonuses) {
    UseClubMultipliers 0;
    }
  }

sub UseClubMultipliers ($) { 
  $gqClubMultipliers = shift;
  if ($gqClubMultipliers) {
    UseAccelerationBonuses 0;
    }
  }

=back

=head2 BUGS

Check to make sure feedback is added before acceleration is calculated.

=cut

1;
