#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::Addscore;

use strict;
use warnings;

use TSH::Utility;
use TSH::Tournament;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::Addscore - implement the C<tsh> Addscore command

=head1 SYNOPSIS

  my $command = new TSH::Command::Addscore;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::Addscore is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub CheckRoundNumber ($$);
sub CheckScores ($);
sub Confirm ($);
sub ConfirmScores ($$$$$);
sub ConfirmSpread ($$$$$);
sub EscapeCommand ($$);
sub ExpandNames ($$);
sub Flush ($);
sub initialise ($$$$);
sub InputBye ($);
sub InputGame ($);
sub InputGameBoth ($);
sub InputGameScores ($);
sub InputGameSpread ($);
sub new ($);
sub PromptBoth ($$);
sub PromptScores ($$);
sub PromptSpread ($$);
sub ReadPromptedLine ($);
sub Run ($$@);

=item $ok = $parserp->CheckRoundNumber($dp)

Returns true if $parserp->{'a_round0'} is currently a valid zero-based round number
for data entry in division $dp.

=cut

sub CheckRoundNumber ($$) {
  my $this = shift;
  my $dp = shift;

  # If gaps are being allowed, anything goes
  return 1 if $this->{'a_config'}->Value('allow_gaps');

  my $round0 = $this->{'a_round0'};
  my $least_scores = $dp->LeastScores();

  # too early, round must already be complete
  if ($round0 <= $least_scores - 1) {
    $this->{'a_tournament'}->TellUser('eallsin', $dp->Name(), $round0+1);
    return 0;
    }
  # too late, must do earlier rounds first
  if ($round0 > $least_scores) {
    $this->{'a_tournament'}->TellUser('emisss', $dp->Name(),
      $dp->{'mins'}+2, $dp->LeastScoresPlayer()->TaggedName());
    return 0;
    }
  return 1;
  }

=item $ok = $parserp->CheckScores()

Returns true if the scores have just been entered are valid.

=cut

sub CheckScores ($) {
  my $this = shift;
  my $tournament = $this->{'a_tournament'};
  my $config = $tournament->Config();
  my $dp = $this->{'a_dp'};
  my $round0 = $this->{'a_round0'};
  my (@pn) = @$this{qw(a_pn1 a_pn2)};
  my (@ps) = @$this{qw(a_ps1 a_ps2)};
  my ($toohigh, $toolow, $lowish) = @$this{qw(a_too_high a_too_low a_lowish)};
  my (@pp) = map { $dp->Player($_) } @pn;
  my $dsize = $dp->CountPlayers();
  # check if players actually played each other
  if (($pp[0]{'pairings'}[$round0]||0) != $pn[1]) {
    my (@pname) = map { ((!defined $_) || ref($_) eq 'HASH') ? '?' : $_->Name() } @pp;
    $tournament->TellUser('eanotopp', $pname[0], $pname[1], $round0+1);
    return 0;
    }
  # check spread if entered
  if ($config->Value('entry') eq 'both') {
    if ($this->{'a_ps1'} - $this->{'a_ps2'} != $this->{'a_spread'}) {
      $this->{'a_tournament'}->TellUser('easa', $this->{'a_ps1'}, $this->{'a_ps2'}, $this->{'a_ps1'}-$this->{'a_ps2'}, $this->{'a_spread'});
      return 0;
      }
    }
  my $check_firsts = ($config->Value('track_firsts') 
    && !$config->Value('assign_firsts'));
  # check each player's data
  for my $i (0..1) {
    my $pn = $pn[$i];
    my $ps = $ps[$i];
    my $pp = $pp[$i];
    # bad player number
    if ($pn < 1 || $pn > $dsize) {
      $tournament->TellUser('enosuchp', $pn);
      return 0;
      }
    # bad score
    if ($ps !~ /^[-+]?\d+$/ || $ps < $toolow|| $ps > $toohigh) {
      $tournament->TellUser('ebadscore', $ps);
      return 0;
      }
    # suspicious score
    if ($ps < $lowish) {
      $tournament->TellUser('wlowscore', $ps);
      }
    # duplicate score
    if (defined $pp->Score($round0)) {
      $tournament->TellUser('ehass', $pp->TaggedName(), $pp->Score($round0));
      return 0;
      }
    # wrong player went first
    if ($check_firsts) {
      my $old = $pp->First($round0);
      if ($old && $old == 2 - $i) { # 2, 1
	$tournament->TellUser('easbad12', $pp->TaggedName(), 
	  (qw(second first))[$i]); 
        }
      $pp->First($round0, 1 + $i); # 1, 2
      }
    }
  return 1;
  }

=item $parserp->Confirm()

Confirms to the user what was just entered.

=cut

sub Confirm ($) {
  my $this = shift;
  my $dp = $this->{'a_dp'};
  my $config = $this->{'a_tournament'}->Config();
  my $entry = $config->Value('entry');
  my $spread = $this->{'a_ps1'} - $this->{'a_ps2'};
  my $wlt = (($spread <=> 0) + 1) / 2;
  my $pp1 = $dp->Player($this->{'a_pn1'});
  my $pp2 = $dp->Player($this->{'a_pn2'});
  if ($entry eq 'spread') {
    $this->ConfirmSpread($spread, $wlt, $pp1, $pp2);
    }
  else { # scores or both
    $this->ConfirmScores($spread, $wlt, $pp1, $pp2);
    }
  }

=item $parserp->ConfirmScores($spread, $wlt, $pp1, $pp2)

Confirms to the user what was just entered, when in scores or both mode.

=cut

sub ConfirmScores ($$$$$) {
  my $this = shift;
  my $spread = shift;
  my $wlt = shift;
  my $pp1 = shift;
  my $pp2 = shift;
  printf "#%d %s %d (%.1f %+d) - #%d %s %d (%.1f %+d): %+d.\n",
    $pp1->ID(), 
    $pp1->Name(), 
    $this->{'a_ps1'},
    $pp1->Wins() + $wlt,
    $pp1->Spread() + $spread,
    $pp2->ID(), 
    $pp2->Name(), 
    $this->{'a_ps2'},
    $pp2->Wins() + 1 - $wlt,
    $pp2->Spread() - $spread,
    $spread,
    ;
  }

=item $parserp->ConfirmSpread($spread, $wlt, $pp1, $pp2)

Confirms to the user what was just entered, when in spread mode.

=cut

sub ConfirmSpread ($$$$$) {
  my $this = shift;
  my $spread = shift;
  my $wlt = shift;
  my $pp1 = shift;
  my $pp2 = shift;
  printf "#%d %s (%.1f %+d) - #%d %s (%.1f %+d).\n",
    $pp1->ID(), 
    $pp1->Name(), 
    $pp1->Wins() + $wlt,
    $pp1->Spread() + $spread,
    $pp2->ID(), 
    $pp2->Name(), 
    $pp2->Wins() + 1 - $wlt,
    $pp2->Spread() - $spread,
    ;
  }

=item $s = $parserp->DivisionRound();

Return a string showing the current division name (if there's more
than one division) and round number, for use in a prompt.

=cut

sub DivisionRound ($) {
  my $this = shift;
  my $dp = $this->{'a_dp'};
  my $round = $this->{'a_round0'} + 1;
  my $tournament = $dp->Tournament();
  my $s = '';
  if ($tournament->CountDivisions() > 1) {
    $s .= $dp->Name();
    }
  $s .= $round;
  return $s;
  }

=item $ran = $parserp->EscapeCommand($command)

If $command is a valid escape command, run it and return true.

=cut

sub EscapeCommand ($$) {
  my $this = shift;
  local($_) = shift;
  my $dp = $this->{'a_dp'};
  my $round = $this->{'a_round0'} + 1;
  if (/^(?:m|miss|missing)(\s+\S+)?$/i) {
    my $div = $1;
    $div = '' unless defined $div;
    $this->Flush();
    $this->Processor()->Process("missing $round$div");
    return 1;
    }
  elsif (/^(?:es|editscore)$/i) {
    my $dname = $dp->Name();
    my $lastpn1 = $this->{'a_lastpn1'};
    my $round = $this->{'a_round0'}+1;
    $this->Flush();
    $this->Processor()->Process("editscore $dname $lastpn1 $round");
    return 1;
    }
  elsif (/^(?:l|look)\s+([a-z]+[a-z\s]*)$/i) {
    $this->Processor()->Process("look $1");
    return 1;
    }
  return 0;
  }

=item $ok = $parserp->ExpandNames($string)

Used internally to expand names of the form surname,given or name to their player
number.  Return false on failure, if an ambiguous name was given.

=cut

sub ExpandNames ($$) {
  my $this = shift;
  my $tournament = $this->{'a_tournament'};
  my $dp = $this->{'a_dp'};
  while ($_[0] =~ /^(.*?)(\w*),(\w*)(.*)$/) {
    my ($pre, $last, $first, $post) = ($1, $2, $3, $4);
    my $pp = $tournament->FindPlayer($last, $first, $dp);
    return 0 unless $pp;
    $_[0] = $pre . ($pp->ID()) . $post; 
    }
  while ($_[0] =~ /^(.*?)([a-z][a-z][-'a-z]*)(.*)$/i) {
    my ($pre, $name, $post) = ($1, $2, $3, $4);
    my $pp = $tournament->FindPlayer($name, '', $dp);
    return 0 unless $pp;
    $_[0] = $pre . ($pp->ID()) . $post; 
    }
  return 1;
  }

=item $command->Flush()

Used internally to flush division data and reset the private count.

=cut

sub Flush ($) {
  my $this = shift;
  if ($this->{'a_changed'}) {
    my $tournament = $this->{'a_tournament'};
    my $config = $tournament->Config();
    $tournament->UpdateDivisions();
    $this->{'a_changed'} = 0;
    if (my $cmds = $config->Value('hook_addscore_flush')) {
      $this->Processor()->Process($cmds, 'nohistory');
      }
    }
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to enter player scores.  You must pair the round
(e.g. by autopairing or using the pm command) before you can enter
any scores.  Begin by specifying the round and division that you
are entering.  At the prompt, enter the first player's number and
score, then the second player's number and score, all on one line.
For a bye or forfeit, enter the player's number and the spread
adjustment.  If you don't know a player's number, try entering
part-of-their-last-name,part-of-their-first-name.  You may enter a
division name to switch divisions, or 'm' to see what scores are
still missing.  To correct a mistake in the game you just entered,
enter 'es'.  If you enter anything else, you will exit the command
and return to the main prompt.
EOF
  $this->{'names'} = [qw(a addscore)];
  $this->{'argtypes'} = [qw(Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

=item $success = $command->InputBye()

Try to parse input data for a bye.  Return success if the input resembled
bye data sufficiently that either it was processed or an error message was emitted,
but in either case no further parsing is required.

=cut

sub InputBye ($) {
  my $this = shift;
  my (@words) = @{$this->{'a_words'}};
  return 0 unless @words == 2;
  my $tournament = $this->{'a_tournament'};
  my $dp = $this->{'a_dp'};
  my $round0 = $this->{'a_round0'};
  my $round = $round0 + 1;
  my ($pn1, $ps1) = @words;
  if ($pn1 < 1 || $pn1 > $dp->CountPlayers()) {
    $tournament->TellUser('enosuchp', $pn1);
    return 1;
    }
  my $pp1 = $dp->Player($pn1);
  my $opp1 = $pp1->OpponentID($round0);
  unless ((defined $opp1) && $opp1 == 0) {
    $tournament->TellUser('enotabye', $pp1->TaggedName(), $round);
    return 1;
    }
  if (my $s = $pp1->Score($round0)) {
    $tournament->TellUser('ehass', $pp1->TaggedName(), $s);
    return 1;
    }
  {
    my $wlt = (($ps1 <=> 0) + 1) / 2;
    printf "#%d %s %+d (%.1f %+d).\n",
      $pp1->ID(),
      $pp1->Name(),
      $ps1,
      $pp1->Wins() + $wlt,
      $pp1->Spread() + $ps1,
      ;
  }
  $this->{'a_lastpn1'} = $pn1;
  $dp->Dirty(1);
  $this->{'a_changed'}++;
  $pp1->Time(time);
  $pp1->Score($round0, $ps1);
  return 1;
  }

=item $success = $command->InputGame()

Try to parse input data for a game.  Return success if we think we
read things that looked like scores.  As a side effect, update
range-checking bounds for scores.

=cut

sub InputGame ($) {
  my $this = shift;
  my $config = $this->{'a_tournament'}->Config();
  my $entry = $config->Value('entry');
  if ($entry eq 'spread') { return $this->InputGameSpread(); }
  elsif ($entry eq 'both') { return $this->InputGameBoth(); }
  else { return $this->InputGameScores(); }
  }

=item $success = $command->InputGameBoth()

Try to parse input data for a game when both spread and scores are
being entered.  Return success if we think we read things that looked
like scores.

=cut

sub InputGameBoth ($) {
  my $this = shift;
  my (@words) = @{$this->{'a_words'}};
  return 0 unless @words == 5;
  @$this{qw(a_pn1 a_ps1 a_pn2 a_ps2 a_spread)} = @words;
  return 1;
  }

=item $success = $command->InputGameScores()

Try to parse input data for a game when only scores are being
entered.  Return success if we think we read things that looked
like scores.

=cut

sub InputGameScores ($) {
  my $this = shift;
  my (@words) = @{$this->{'a_words'}};
  return 0 unless @words == 4;
  @$this{qw(a_pn1 a_ps1 a_pn2 a_ps2)} = @words;
  return 1;
  }

=item $success = $command->InputGameSpread()

Try to parse input data for a game when only spread is being entered.
Return success if we think we read things that looked like scores.
As a side effect, update range-checking bounds for scores.

=cut

sub InputGameSpread ($) {
  my $this = shift;
  my $tournament = $this->{'a_tournament'};
  my (@words) = @{$this->{'a_words'}};
  return 0 unless @words == 3;
  @$this{qw(a_pn1 a_pn2 a_ps1)} = @words;
  $this->{'a_ps2'} = 0;
  $this->{'a_too_low'} = 0;
  $this->{'a_lowish'} = -1;
  $this->{'a_too_high'} = 999;
  return 1;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->PromptBoth($left)

Prompt for combined scores and spread.

=cut

sub PromptBoth ($$) {
  my $this = shift;
  my $left = shift;
  my $divrd = $this->DivisionRound();
  TSH::Utility::Prompt(
    $left == 0 ? "[$divrd]:ES|L words|M|division|<return> (no scores left)?"
    : $left == 1 ? "[$divrd]:pn spr (1 score left)?"
    : "[$divrd]:pn1 ps1 pn2 ps2 spr ($left scores left)?"
    );
  }
=item $command->PromptScores()

Prompt for scores.

=cut

sub PromptScores ($$) {
  my $this = shift;
  my $left = shift;
  my $divrd = $this->DivisionRound();
  TSH::Utility::Prompt(
    $left == 0 ? "[$divrd]:ES|L words|M|division|<return> (no scores left)?"
    : $left == 1 ? "[$divrd]:pn spread ($left score left)?"
    : "[$divrd]:pn1 ps1 pn2 ps2 ($left scores left)?"
    );
  }

=item $command->PromptSpread()

Prompt for spread.

=cut

sub PromptSpread ($$) {
  my $this = shift;
  my $left = shift;
  my $divrd = $this->DivisionRound();
  TSH::Utility::Prompt(
    $left == 0 ? "[$divrd]:ES|L words|M|division|<return> (no scores left)?"
    : $left == 1 ? "[$divrd]:player spread (1 player left)?"
    : "[$divrd]:winner loser spread ($left players left)?"
    );
  }

=item $command->ReadPromptedLine()

Prompt for and read one line of input for Addscore.

=cut

sub ReadPromptedLine ($) {
  my $this = shift;
  my $entry = $this->{'a_config'}->Value('entry');
  my $round0 = $this->{'a_round0'};
  my $dp = $this->{'a_dp'};
  my $left = scalar(grep { (!exists $_->{'etc'}{'off'}) 
    && ! defined $_->{'scores'}[$round0] } 
    $dp->Players());
  if ($entry eq 'spread') { $this->PromptSpread($left); }
  elsif ($entry eq 'both') { $this->PromptBoth($left); }
  else { $this->PromptScores($left); }
  local($_) = scalar(<STDIN>);
  return '' unless defined $_;
  s/\s+$//;
  return $_;
  }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = $this->{'a_tournament'} = shift;
  my $round = shift;
  my $dp = $this->{'a_dp'} = shift;
  my $round0 = $this->{'a_round0'} = $round - 1;
  my $config = $this->{'a_config'} = $tournament->Config();
  my $save_interval = ($config->Value('save_interval') || 10);

  return unless $this->CheckRoundNumber($dp);

  $this->{'a_lastpn1'} = 1;
  $this->{'a_changed'} = 0;
# my $entry = $config->Value('entry'};
prompt:while (1) {
    local($_) = $this->ReadPromptedLine();
    last if /^$/;
    if (my $newdp = $tournament->GetDivisionByName($_)) {
      $this->{'a_dp'} = $dp = $newdp;
      return unless $this->CheckRoundNumber($newdp);
      next;
      }
    next if $this->EscapeCommand($_);
    next unless $this->ExpandNames($_);
    last if /[^-.\d\s]/;
    $this->{'a_words'} = [split(/[\s.]+/)];
    # user entered spread for a bye
    next if $this->InputBye();
    # user entered data for a game played
    $this->{'a_pn1'} = $this->{'a_ps1'} =
    $this->{'a_pn2'} = $this->{'a_ps2'} = undef;
    $this->{'a_too_high'} = 1499;
    $this->{'a_too_low'} = -149;
    $this->{'a_lowish'} = 100;
    last unless $this->InputGame();
    next unless $this->CheckScores();
    $this->Confirm();
    $this->{'a_lastpn1'} = $this->{'a_pn1'};
    $dp->Dirty(1);
    $this->{'a_changed'}++;
    my $pp1 = $dp->Player($this->{'a_pn1'});
    my $pp2 = $dp->Player($this->{'a_pn2'});
    {
      my $now = time;
      $pp1->Time($now);
      $pp2->Time($now);
    }
    $pp1->Score($round0, $this->{'a_ps1'});
    $pp2->Score($round0, $this->{'a_ps2'});
    }
  continue {
    if ($this->{'a_changed'} >= $save_interval) {
      $this->Flush();
      }
    }
  $this->Flush();
  }

=back

=cut

=head1 BUGS

Should use a subprocessor rather than an event loop.

=cut

1;

