#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ScoreBoard;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn Ordinal);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ScoreBoard - implement the C<tsh> ScoreBoard command

=head1 SYNOPSIS

  my $command = new TSH::Command::ScoreBoard;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ScoreBoard is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub AddTDataPerson ($$);
sub ComputeSeeds ($);
sub FormatPlayerPhotoName($$;$);
sub FormatHalfInteger ($);
sub HeadShot ($$;$);
sub initialise ($$$$);
sub new ($);
sub RenderPlayer ($$$);
sub RenderTable ($$$);
sub Run ($$@);
sub SetThresholdAttribute($$$$);
sub TagName ($;$);

=item $command->ComputeSeeds()

Used internally to compute player seeds.

=cut

sub ComputeSeeds ($) {
  my $this = shift;
  my (@seeded) = (TSH::Player::SortByInitialStanding($this->{'dp'}->Players()));
  my @seed;
  my $lastrat = -1;
  my $rank = 1;
  for my $i (0..$#seeded) {
    my $p = $seeded[$i];
    my $rating = $p->Rating();
    if ($rating != $lastrat) {
      $rank = $i+1;
      $lastrat = $rating;
      }
    $seed[$i] = $rank;
    }
  $this->{'seed'} = \@seed;
  }

=item $command->FormatPlayerPhotoName($logp, $player[, $optionsp]);

Used internally to render a headshot and name for a player:
the current player, an opponent, or maybe a bye.

=cut

sub FormatPlayerPhotoName($$;$) {
  my $this = shift;
  my $p = shift;
  my $optionsp = shift;
  
  if (UNIVERSAL::isa($p, 'TSH::Player')) {
    return $this->HeadShot($p,$optionsp->{'size'}) . TagName($p, $optionsp);
    }
  else {
    return '<div class=nohead>$p</div>';
    }
  }

=item $html = FormatHalfInteger($n);

Format a half integer.

=cut

sub FormatHalfInteger ($) {
  my $x = shift;
  if ($x == int($x)) {
    return $x;
    }
  else {
    return int($x) . "&frac12;";
    }
  }

=item $html = $this->HeadShot($player)

Used internally to generate an IMG tag for a player headshot.

=cut

sub HeadShot ($$;$) {
  my $this = shift;
  my $p = shift;
  my $photo_size = shift || $this->{'photo_size'};
  if ($this->Processor()->Tournament()->Config()->Value('player_photos')) {
    return sprintf(q(<div class=head><img class=head src="%s" alt="[head shot]" height="%d" width="%d"></div>), $p->PhotoURL(), $photo_size, $photo_size);
    }
  else {
    return "&nbsp;";
    }
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to update a scoreboard suitable for displaying on a
second monitor or web kiosk.
You must specify a division, and may omit (beginning at the end) the remaining
optional arguments: first rank, last rank, pixel width of player photos, 
seconds between refreshes.
EOF
  $this->{'names'} = [qw(sb scoreboard)];
  $this->{'argtypes'} = [qw(Division OptionalInteger OptionalInteger OptionalInteger OptionalInteger)];
# print "names=@$namesp argtypes=@$argtypesp\n";
  $this->{'table_format'} = undef;
  $this->{'c_has_tables'} = undef;
  $this->{'dp'} = undef;
  $this->{'first_rank'} = undef;
  $this->{'last_rank'} = undef;
  $this->{'photo_size'} = undef;
  $this->{'r0'} = undef;
  $this->{'r1'} = undef;
  $this->{'refresh'} = undef;
  $this->{'seed'} = undef;

  return $this;
  }

sub InTheMoney ($$$) {
  my $this = shift;
  my $p = shift;
  my $dp = $p->Division();
  my $r0 = $this->{'r0'};
  my $config = $dp->Tournament()->Config();
  my $crank;
  if ($this->{'is_capped'}) {
    $crank = $p->RoundCappedRank($r0);
    }
  else {
    $crank = $p->RoundRank($r0);
    }
  my $is_in_money = 0;
  if (my $prize_bands = $config->Value('prize_bands')) {
    if (my $prize_band = $prize_bands->{$dp->Name()}) {
      if ($crank <= $prize_band->[-1]) {
	$is_in_money = 1;
	}
      }
    }
  return $is_in_money;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->RenderPlayer($logp, $player);

Used internally to render a player.

=cut

sub RenderPlayer ($$$) {
  my $this = shift;
  my $logp = shift;
  my $p = shift;
  my $r0 = $this->{'r0'};
  my $r1 = $this->{'r1'};
  my $seed = $this->{'seed'}[$p->ID()-1];
  my $html = '';
  my $dp = $p->Division();
  my $config = $dp->Tournament()->Config();
  {
    my ($crank, $lrank, $spread);
    if ($this->{'is_capped'}) {
      $crank = $p->RoundCappedRank($r0);
      $lrank = $r0 > 0 ? $p->RoundCappedRank($r0 - 1) : $seed;
      $spread = $p->RoundCappedSpread($r0);
      }
    else {
      $crank = $p->RoundRank($r0);
      $lrank = $r0 > 0 ? $p->RoundRank($r0 - 1) : $seed;
      $spread = $p->RoundSpread($r0);
      }
    my $is_in_money = $this->InTheMoney($p);
    $html .= '<div class="sbp' . ($is_in_money ? ' money' : '') .'"'
#     . ' style="max-width:' . int(2.5*$this->{'photo_size'}) . 'px"'
      . '>';
    $html .= '<div class=me>';
    $html .= "<div class=rank>"
      . Ordinal($crank)
      . "</div>";
    $html .=
      "<div class=old>was<br>"
      . Ordinal($lrank)
      ."</div>\n" if defined $p->Score($r0);
    $html .= "<div class=money>\$</div>" if $is_in_money;
    $html .= "<div class=wl>"
	. FormatHalfInteger($p->RoundWins($r0))
	. "&ndash;"
	. FormatHalfInteger ($p->RoundLosses($r0)) 
	. ', ' 
	. ($spread >= 0 ? "+$spread" : "&minus;" . -$spread)
      . "</div>";
  }
  $html .= $this->FormatPlayerPhotoName($p,{'size'=>$this->{'photo_size'},'id'=>1});
  # rating information
  { 
    my $oldr = $p->Rating();
    my $newr = $p->NewRating();
    my $delta = $newr-$oldr;
    $delta = "+$delta" if $delta >= 0;
    $delta =~ s/-/&minus;/;
    $html .= "<div class=newr>$newr</div>\n";
    if ($oldr) {
      $html .= "<div class=oldr>=$oldr<br>$delta</div>\n";
      }
    else {
      $html .= "<div class=oldr>was<br>unrated</div>\n";
      }
  }
  $html .= "</div>\n"; # me
  $html .= "<div class=opp>\n"; # opp
  { # last game
    my $last = '';
    my $oppid = $p->OpponentID($r0);
    if ($oppid) {
      my $op = $p->Opponent($r0);
      my $os = $op->Score($r0);
      if (defined $os) {
	my $ms = $p->Score($r0);
	my $spread = $ms - $os;
	$spread = $spread >= 0 ? "+$spread" : "&minus;" . -$spread;
	$last .= "<div class=gs>$ms&minus;$os=$spread</div>";
	}
      else {
	$last .= "<div class=gsn>no score yet</div>";
	}
      my $first = $p->First($r0);
      $first = $first == 1 ? '1st' : $first == 2 ? '2nd' : '';
      my $board = $p->Board($r0);
      $board = $this->{'dp'}->BoardTable($board) if $this->{'c_has_tables'};
      $last .= "<div class=where>$first \@$board vs.</div>\n";
      $last .= "<div class=hs>" 
        .$this->FormatPlayerPhotoName($op,{'size'=>int($this->{'photo_size'}/2)}) 
	. "</div>";
      }
    else { # bye
      my $ms = $p->Score($r0);
      $ms = (defined $ms) ? sprintf("%+d", $ms) : '';
      $last .= "<div class=bye>bye $ms</div>";
      }
    if ($last) {
      $html .= "<div class=last><div class=title>Last:</div>$last</div>\n";
      }
  }
  { # next game
    my $next = '';
    my $oppid = $p->OpponentID($r0+1);
    if ($oppid = $p->OpponentID($r0+1)) {
      my $op = $p->Opponent($r0+1);
      my $first = $p->First($r0+1);
      my $board = $p->Board($r0+1);
      my $repeats = $p->Repeats($op->ID());
      $board = $this->{'dp'}->BoardTable($board) if $this->{'c_has_tables'};
      $first = $first == 1 ? '1st' : $first == 2 ? '2nd' : '';
      if ($repeats > 1) {
	$next .= "<div class=repeats>"
	  .($repeats==2?"repeat":$repeats."peat")
	  ."</div>";
        }
      $next .= "<div class=where>$first \@$board vs.</div>\n";
      $next .= "<div class=hs>" 
	.$this->FormatPlayerPhotoName($op,{'size'=>int($this->{'photo_size'}/2)}) 
	. "</div>";
      }
    elsif (defined $oppid) {
      $next .= "<div class=bye>bye</div>";
      }
    if ($next) {
      $html .= "<div class=next><div class=title>Next:</div>$next</div>\n";
      }
  }
  { # record
    my $record = '';
    my $nscores = $p->CountScores();
    if ($nscores > 1) {
      for my $r0 (0..$nscores-1) {
	my $ms = $p->Score($r0);
	$record .= "<div class=rd>";
	if ($p->Opponent($r0)) {
	  my $os = $p->OpponentScore($r0);
	  if (defined $os) {
	    $record .= $ms > $os ? "<div class=win>W</div>"
	      : $ms < $os ? "<div class=loss>L</div>"
	      : "<div class=tie>T</div>";

	    }
	  else {
	    $record .= "<div class=unknown>?</div>";
	    }
	  }
	elsif (defined $ms) {
	  $record .= $ms > 0 ? "<div class=bye>B</div>"
	    : $ms < 0 ? "<div class=forfeit>F</div>"
	    : "<div class=missed>&ndash;</div>";
	  }
	my $p12 = $p->First($r0);
	if ($p12 && $p12 =~ /^[12]$/) {
	  $record .= "<div class=p$p12>$p12</div>";
	  }
	else {
	  $record .= "<div class=p0>&ndash;</div>";
	  }
	$record .= "</div>";
	}
      }
    if ($record) {
      $html .= "<div class=record><div class=title>Rec<br>ord</div>$record</div>\n";
      }
  }
  $html .= '</div>'; # opp
  $html .= '</div>'; # sbp
  $logp->Write('', $html);
  return;
  # record of wins and losses
  # record of firsts and seconds
  # player number
  } 

=item $command->RenderTable($logp, \@players);

Used internally to render the table.

=cut

sub RenderTable ($$$) {
  my $this = shift;
  my $logp = shift;
  my $psp = shift;
  $logp->Write('', '<div id=sbs>');
  for my $p (@$psp) {
    $this->RenderPlayer($logp, $p);
    }
  $logp->Write('', '</div><br clear=all />');
  $logp->Write('', <<'EOF');
<script language="JavaScript"><!--
  function fix_sizes () {
  var p = document.getElementById('sbs').firstChild
  var maxh, minh;
  maxh = minh = p.offsetHeight;
  while (p = p.nextSibling) {
    if (p.className != 'sbp' && p.className != 'sbp money') { continue; }
    if (maxh < p.offsetHeight) { maxh = p.offsetHeight; }
    if (minh > p.offsetHeight) { minh = p.offsetHeight; }
    }
  p = document.getElementById('sbs').firstChild;
  p.style.height = maxh;
  while (p = p.nextSibling) {
    if (p.className != 'sbp' && p.className != 'sbp money') { continue; }
    p.style.height = maxh;
    }
  }
  setTimeout('fix_sizes()', 1000);
--></script>
EOF
  }

=item $command->Run($tournament, @parsed_args);

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = $this->{'dp'} = shift;
  my ($rank1, $rank2, $photo_size, $refresh) = @_;
  # This should all be in a sub called RunInit();
  $this->{'first_rank'} = ($rank1 || 1);
  $this->{'last_rank'} = ($rank2 || $dp->CountPlayers());
  $this->{'photo_size'} = ($photo_size || 32);
  $this->{'refresh'} = ($refresh || 10);
  $this->{'table_format'} = {
    'head' => { 
      'attributes' => [ [], [] ],
      'classes' => [ [], [] ],
      'titles' => [ [], [] ],
      },
    'data' => { 
      'attributes' => [ [], [] ],
      'classes' => [ [], [] ],
      },
    };
  $this->{'table_data'} = {
    'attributes' => [ [], [] ],
    'contents' => [ [], [] ],
    };
  my $r1 = $this->{'r1'} = $dp->MostScores();
  my $r0 = $this->{'r0'} = $r1 - 1;
  my $config = $tournament->Config();
  my $c_spreadentry = $config->Value('entry') eq 'spread';
  my $c_trackfirsts = $config->Value('track_firsts');
  my $c_has_classes = $dp->Classes();
  my $c_has_tables = $this->{'c_has_tables'} = $dp->HasTables();
  my $c_is_capped = $this->{'is_capped'} = $config->Value('standings_spread_cap');

  my $logp = new TSH::Log($tournament, $dp, 'scoreboard', undef,
    {'noconsole' => 1, 'notext' => 1,  'refresh' => $this->{'refresh'},
    'titlename' => 'scoreboard',});
  $logp->Write('', '</table>');

  if ($c_is_capped) {
    $dp->ComputeCappedRanks($r0);
    $dp->ComputeCappedRanks($r0-1) if $r0 > 0;
    }
  else {
    $dp->ComputeRanks($r0);
    $dp->ComputeRanks($r0-1) if $r0 > 0;
    }
  $dp->ComputeRatings($r0);
  $this->ComputeSeeds();
  my (@ps) = ($dp->Players());
  TSH::Player::SpliceInactive(@ps, 0, 0);
  if ($c_is_capped) {
    @ps = TSH::Player::SortByCappedStanding($r0, @ps);
    }
  else {
    @ps = TSH::Player::SortByStanding($r0, @ps);
    }
  if ($this->{'last_rank'} && $this->{'last_rank'} < @ps) {
    splice(@ps, $this->{'last_rank'});
    }
  if ($this->{'first_rank'} && $this->{'first_rank'} <= @ps) {
    splice(@ps, 0, $this->{'first_rank'}-1);
    }
  $this->RenderTable($logp, \@ps);
  $logp->Write('', '<table>');

  $logp->Close();
  return 0;
  }

=item $command->SetThresholdAttribute($subrow, $value, $threshold);

Used internally to set the HTML attribute of the last stored content
according to its value and a threshold.

=cut

sub SetThresholdAttribute($$$$) {
  my $this = shift;
  my $subrow = shift;
  my $value = shift;
  my $threshold = shift;
  return unless $threshold;

  my $attributesp = $this->{'table_data'}{'attributes'}[$subrow];
  my $i = $#{$this->{'table_data'}{'contents'}->[$subrow]};
  $attributesp->[$i] .= ' ' if $attributesp->[$i];
  $attributesp->[$i] .= 'bgcolor="#' . (
    (!$value) ? 'ffffff' :
    $value >= $threshold ? '80ff80' : $value > 0 ? 'c0ffc0' :
    $value <= -$threshold ? 'ff8080' : 'ffc0c0'
    ) . '"';
  }

=item $html = TagName($p, [$optionsp])

Used internally to tag a player's given name and surname.

=cut

sub TagName ($;$) {
  my $p = shift;
  my $optionsp = shift || {};
  my $name = $p->Name();
  if ($name =~ /^(.*), (.*)$/) {
    my $given = $2;
    my $surname = $1;
    if ($surname =~ /^(.*)-(.*)$/) {
      my $surname1 = $1;
      my $surname2 = $2;
      if (length($surname1) < length($surname2) + length($given)) {
	$given = "$surname2, $given";
	$surname = "$surname1-";
        }
      }
    else {
#      $surname =~ s!Apissoghomian!Apisso-<br>ghomian!;
      }
#   else {
#      $surname =~ s!Apissoghomian!<span class=narrow>$&</span>!;
#     }
    if ($optionsp->{'id'}) {
      if (length($surname) > length($given)) {
	$given .= " (#". $p->ID().")" if $optionsp->{'id'};
        }
      else {
	$surname .= " (#". $p->ID().")" if $optionsp->{'id'};
	}
      }
    $name = "<div class=name>"
      . "<span class=given>$given</span>"
      . "<span class=surname>$surname</span>"
      . "</div>";
#   $name = "<div class=name><span class=surname>$surname</span>"
#     . "<span class=given>$given</span></div>";
    }
  return $name;
  }

=back

=cut

# sherrie's imac: 1920 x 1200

1;
