#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::TeamRoundRobin;

use strict;
use warnings;

use TSH::PairingCommand;
use TSH::Utility;
use TSH::Division::Team;

our (@ISA) = qw(TSH::PairingCommand);
our @Pairings;
our @Starts;

=pod

=head1 NAME

TSH::Command::TeamRoundRobin - implement the C<tsh> TeamRoundRobin command

=head1 SYNOPSIS

  my $command = new TSH::Command::TeamRoundRobin;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::TeamRoundRobin is a subclass of TSH::PairingCommand.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the TeamRoundRobin command to manually add a set of full team 
"round robin" pairings
to a division, where each player plays every player on every other
team.  When you use this command, the last paired round must
be fully, not partially paired.  If you specify an integer before the
division name, each pairing will be repeated that number of times.
EOF
  $this->{'names'} = [qw(trr teamroundrobin)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $count = @_ ? pop @_ : 1;
  my $config = $tournament->Config();

  my $datap = $dp->{'data'};
# my $nplayers = $#$datap;

  # check that division is not partially paired
  my $round0 = $dp->FirstUnpairedRound0();
  if ($round0 != $dp->LastPairedRound0()+1) {
    $tournament->TellUser('errpartp');
    return;
    }

  my $psp = $dp->GetUnpairedRound($round0);
  my $teamsp = TSH::Division::Team::GroupByTeam($psp);
  my (@team_names) = sort keys %$teamsp;
  unless (@team_names == 2
    && abs(@{$teamsp->{$team_names[0]}} - @{$teamsp->{$team_names[1]}}) <= 1
    ) {
    $tournament->TellUser('etrrlame');
    return;
    }

  my $swap = int(rand(2));
  my (@team1) = TSH::Player::SortByInitialStanding @{$teamsp->{$team_names[$swap]}};
  my (@team2) = TSH::Player::SortByInitialStanding @{$teamsp->{$team_names[1-$swap]}};
  my $team_size = @team1 > @team2 ? @team1 : @team2;
  $tournament->TellUser('itrrok', $dp->{'name'});

  for my $i (0..$team_size-1) {
    for my $k (1..$count) {
      for my $j (0..$team_size-1) {
	my $p1 = $team1[$j];
	my $p2 = $team2[($team_size*2-1-$i+$j) % $team_size];
        my $pn1 = $p1 ? $p1->ID() : 0;
        my $pn2 = $p2 ? $p2->ID() : 0;
	$dp->Pair($pn1, $pn2, $round0, 0);
        }
      $dp->GetUnpairedRound($round0++); # side-effect: assign byes to inactive players
      }
    }

  $tournament->TellUser('idone');

  $dp->Dirty(1);
  $dp->Update();
  }

=back

=cut

=head1 BUGS

None known.

=cut

1;
