#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::luckySTIFF;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::luckySTIFF - implement the C<tsh> luckySTIFF command

=head1 SYNOPSIS

  my $command = new TSH::Command::luckySTIFF;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::luckySTIFF is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings for a "Lucky Stiff" prize, 
typically awarded to a player who wins six games by the smallest
combined spread.  The optional second argument, if given, specifies
a value other than "six".
EOF
  $this->{'names'} = [qw(luckystiff stiff)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $nscores = @_ ? shift @_ : 6;

  my $logp = new TSH::Log( $tournament, $dp, 'luckystiff', undef,
    {'title' => 'Division ' . $dp->Name() . ' Lucky Stiff'});
  $logp->ColumnClasses([ qw(round name scores) ]);
  $logp->ColumnTitles(
    {
    'text' => [ qw(Sum Player Wins) ],
    'html' => [ 'Sum', 'Player', 'Winning Spreads'],
    }
    );
  my @data;
  for my $pp ($dp->Players()) {
    my @wins;
    for my $r0 (0..$pp->CountScores()-1) {
      my $ms = $pp->Score($r0);
      next unless defined $ms;
      my $os = $pp->OpponentScore($r0);
      next unless defined $os;
      next unless $ms > $os;
      push(@wins, $ms-$os);
      }
    next if @wins < $nscores;
    (@wins) = sort { $a <=> $b } @wins;
    my $tuffness = 0;
    splice(@wins, $nscores);
    for my $loss (@wins) { $tuffness += $loss; }
    push(@data, [$tuffness, $pp->TaggedName(), \@wins]);
    }
  for my $data (sort {
    $a->[0] <=> $b->[0] || do {
      for (my $i=0; $i<$nscores; $i++) {
	my $cmp = $a->[2][$i] <=> $b->[2][$i];
	return $cmp if $cmp;
	}
      0;
      }
    } @data) {
    $data->[2] = join(' ', @{$data->[2]});
    $logp->WriteRow($data, $data);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

