#!/usr/bin/perl

# Copyright (C) 2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package WebUpdate;

use strict;
use warnings;

use File::Path;
use File::Spec;
BEGIN {
  if ($^O eq 'MSWin32') {
    eval "use Win32::Internet";
    }
  else {
    eval "use HTTP::Client";
    }
  }


=pod

=head1 NAME

WebUpdate - update a set of files from a web server

=head1 SYNOPSIS

  my $wup = new WebUpdate(
    'server' => $server,
    'basepath' => $basepath,
    'manifest' => $manifest_name,
    );
  $wup->Update();

  
=head1 ABSTRACT

This Perl module is used to update a set of local files from
content served on a web server.

=head1 DESCRIPTION

=over 4

=cut

sub CheckDirectory ($$);
sub GetFile ($$);
sub GetLocalManifest ($$);
sub GetRemoteManifest ($$);
sub GetHTTP ($$$);
sub new ($@);
sub ParseManifest ($$);
sub Update ($);
sub UpdateFile ($$$$$);

sub CheckDirectory ($$) {
  my $this = shift;
  my $file = shift;
  my ($v, $d, $f) = File::Spec->splitpath($file);
  my $dir = File::Spec->catpath($v, $d, '');
  return unless $dir =~ /\S/;
  mkpath $dir, 0, 0755;
  }

sub GetFile ($$) {
  my $this = shift;
  my $path = shift;
  if ($^O eq 'MSWin32') {
    my $file = $this->{'win32'}->FetchURL(
      "http://$this->{'server'}/$this->{'basepath'}/$path"
      );
    unless ($file) {
      print "Server sent us an empty file for $path.  You may have briefly lost your Internet connection.  Check it and rerun the update.\n";
      return undef;
      }
    if ($file =~ /<title>(\d\d\d .*?)<\/title>/i) {
      print "This looks like a server problem: $1\n";
      return undef;
      }
    return $file;
    }
  return $this->GetHTTP($this->{'server'}, "$this->{'basepath'}/$path");
  }

sub GetHTTP($$$) {
  my $this = shift;
  my $server = shift;
  my $path = shift;
  $path = "/$path" unless $path =~ m!^/!;

  my $http = eval "new HTTP::Client";
  die "can't new HTTP::Client: $@" if $@;
  $http->Connect($server);
  my $request = ''
    ."GET $path HTTP/1.1\015\012"
    ."Host: $server\015\012"
    ."Accept-Charset: iso-8859-1\015\012"
    ."Accept-Encoding:\015\012"
    ."User-Agent: tsh-update/$main::gkVersion\015\012"
    ."\015\012";
  $http->SendRaw($request);
  my $response = $http->Receive();
  $http->Close();
  if ($response->StatusCode && $response->StatusCode =~ /^301$/) {
    my $location = $response->Header('location');
    ($server, $path) = $location 
      =~ /^http:\/\/([^\/]+)(.*)/;
    unless (defined $path) {
      print "Bad redirect: $location\n";
      return undef;
      }
    return $this->GetHTTP($server, $path);
    }
  if (my $error = $response->Error()) {
    print "Can't get http://$server$path: $error\n";
    return undef;
    }
  my $html = $response->Body();
  return $html;
  }
  
sub GetLocalManifest ($$) {
  my $this = shift;
  my $fn = $this->MakeLocalPath(shift);
  local($/) = undef;
# warn "opening local manifest $fn";
  open my $fh, "<$fn" or return {};
  my $text = <$fh>;
  close $fh;
# print $text;
  return $this->ParseManifest($text);
  }

sub GetRemoteManifest ($$) {
  my $this = shift;
  my $fn = shift;
  print "Fetching remote manifest file.\n";
  my $text = $this->GetFile($fn);
  unless (defined $text) { return undef; }
  return $this->ParseManifest($text);
  }

sub MakeLocalPath ($$) {
  my $this = shift;
  my $path = shift;
  my $localpath = $this->{'localpath'};
  $path = File::Spec->catfile($localpath, $path) if $localpath;
  return $path;
  }

=item $wup = new WebUpdate('server' => $server,
  'basepath' => $basepath, 'manifest' => $manifest_name);

Create a new WebUpdate object or return undef on failure.

=cut

sub new ($@) {
  my $proto = shift;
  my (%args) = @_;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    %args,
    };
  if ($^O eq 'MSWin32') {
    $this->{'win32'} = new Win32::Internet();
    }
  bless($this, $class);
  return $this;
  }

sub ParseManifest ($$) {
  my $this = shift;
  my $lines = shift;
  my %data;

  for my $line (split(/\n/, $lines)) {
    next if $line =~ /^(?:Archive| +Length +Date|[- ]+$| +\d+ +\d+ files$)/;
    $line =~ s/^\s+//; $line =~ s/\+$//;
    my ($length, $date, $time, $file) = split(/\s+/, $line, 4);
    unless (defined $file) {
      print "Can't parse: $line\n";
      return undef;
      }
    $data{$file} = "$length $date $time";
    }
  return \%data;
  }

sub Update ($) {
  my $this = shift;

  my $localsp = $this->GetLocalManifest($this->{'manifest'});
  unless (keys %$localsp) {
    print "No local manifest.\n";
    }
# warn join("\n", keys %$localsp);
  my $remotesp = $this->GetRemoteManifest($this->{'manifest'});
  unless ($remotesp) {
    print "No remote manifest.\n";
    return -1;
    }
  print "Checking manifest file.\n";
  my $found_something;
  my $did_something = 0;
  for my $file (sort keys %$remotesp) {
    $found_something++;
    my $path = $this->MakeLocalPath($file);
    if ((!-e $path) || (!exists $localsp->{$file}) 
      || $localsp->{$file} ne $remotesp->{$file}) { 
#     warn "$file: $localsp->{$file} ne $remotesp->{$file}";
      if ($path =~ /\/$/) {
	unless (-d $path) {
	  print "Creating directory $path\n";
	  mkdir $path || warn "mkdir() failed: $!\n";
	  $did_something++;
	  }
	}
      else {
	$this->UpdateFile($file, $remotesp->{$file}, $localsp, $path) 
	  or return -1;
	$did_something++;
	}
      }
    }
  unless ($found_something) {
    print "Empty manifest file received: are you behind a firewall?\n";
    return -1;
    }
  return $did_something;
  }

sub UpdateFile ($$$$$) {
  my $this = shift;
  my $file = shift;
  my $remotedata = shift;
  my $localsp = shift;
  my $path = shift;
  
  print "Updating $file.\n";
  my $content = $this->GetFile($file);
  $localsp->{$file} = $remotedata;
  $this->CheckDirectory($path);
  my $fh;
  unless (open $fh, ">$path-new") {
    print "Can't create $path-new: $!\n";
    return 0;
    }
  if ($path =~ /\.(?:doc|dwg|gif|ico|jpg|jpeg|zip)$/) {
    binmode $fh;
    }
  unless (print $fh $content) {
    print "Can't write to $path-new: $!\n";
    return 0;
    }
  unless (close $fh) {
    print "Can't close $path-new: $!\n";
    return 0;
    }
  unless (rename "$path-new", $path) {
    print "Can't update $path\n";
    return 0;
    }
  if ($path =~ /^(?:bin|util)\/|^(?:get-tsh|tourney|tsh)\.pl$|^osx-tsh\.command$/ && $path !~ /\.(?:hlp|txt)$/) {
    unless (chmod 0755, $path) {
      print "chmod $path failed: $!\n";
      return 0;
      }
    }
  $fh = undef;
  $path = $this->MakeLocalPath($this->{'manifest'});
  unless (open $fh, ">$path") {
    print "Can't creat $path: $!\n";
    return 0;
    }
  unless (print $fh map { qq($localsp->{$_} $_\n) } sort keys %$localsp) {
    print "Can't write to $path: $!\n";
    return 0;
    }
  unless (close $fh) {
    print "Can't close $path: $!\n";
    return 0;
    }
  return 1;
  }

=back

=cut

1;
