#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::TeamStandings;

use strict;
use warnings;

use TSH::Log;
# use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::TeamStandings - implement the C<tsh> TeamStandings command

=head1 SYNOPSIS

  my $command = new TSH::Command::TeamStandings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::TeamStandings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings as of a specific round.
EOF
  $this->{'names'} = [qw(ts teamstandings)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;

  my $rankround0 = $dp->LeastScores()-1;
  $dp->ComputeRanks($rankround0);
  my $round = $dp->MostScores();
  my $logp = new TSH::Log($tournament, $dp, 'teams', $round);
  $logp->ColumnClasses([qw(rank wl spread wl spread name ranks)]);
  $logp->ColumnTitles(
    {
      'text' => ['Rnk', '  Won-Lost', 'Spread', ' Mean-WL',  qw(Mean-Spr Team Ranks)],
      'html' => [qw(Rank Won-Lost Spread Mean-WL Mean-Spread Team Ranks)],
    }
    );

  my %teams;
  for my $p ($dp->Players()) {
    next unless $p->Active();
    my $team = $p->Team();
    next unless length($team);
    $teams{$team}{'wins'} += $p->Wins();
    $teams{$team}{'losses'} += $p->Losses();
    $teams{$team}{'spread'} += $p->Spread();
    $teams{$team}{'count'} ++;
    unless (defined $teams{$team}{'ranks'}) {
      $teams{$team}{'ranks'} = [];
      }
    push(@{$teams{$team}{'ranks'}}, $p->RoundRank($rankround0));
    }

  my $lastw = -1; my $lasts = 0; my $rank = 0; my $i = 0; my $lastl = -1;
  for my $teamname (sort { 
    my $ac = $teams{$a}{'count'};
    my $bc = $teams{$b}{'count'};
    $teams{$b}->{'wins'}/$bc <=> $teams{$a}->{'wins'}/$ac
    || $teams{$a}->{'losses'}/$ac <=> $teams{$b}->{'losses'}/$bc
    || $teams{$b}->{'spread'}/$bc <=> $teams{$a}->{'spread'}/$ac
    } keys %teams) {
    my $teamdata = $teams{$teamname};
    my $wins = $teamdata->{'wins'};
    my $losses = $teamdata->{'losses'};
    my $spread = $teamdata->{'spread'};
    my $count = $teamdata->{'count'};
    my $ranks = $teamdata->{'ranks'};
    $i++;
    if ($wins != $lastw || $spread != $lasts || $losses != $lastl) {
      $lastl = $losses;
      $lastw = $wins; $lasts = $spread;
      $rank = $i;
      }
    $logp->WriteRow([
      $rank,
      sprintf("%5.1f-%5.1f", $wins, $losses),
      sprintf("%+7.1f", $spread),
      sprintf("%5.2f-%5.2f", $wins/$count, $losses/$count),
      sprintf("%+6.1f", $spread/$count),
      $teamname,
      join(' ', sort { $a <=> $b } @$ranks),
      ]
    );
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known

=cut

1;
