#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Report::CSC;

use strict;
use warnings;
use Carp;
use TSH::Log;

=pod

=head1 NAME

TSH::Report::CSC - utility for generating a contestant scorecard

=head1 SYNOPSIS

  $c = new TSH::Report::CSC($player);
  ($text, $html) = $c->GetBoth(\%options);
  $s = $c->GetHTML(\%options);
  $s = $c->GetText(\%options);

=head1 ABSTRACT

This Perl module is used to create contestant scorecards.
Note that all of the C<Get*()> methods generate printable
files as a side effect.

=head1 DESCRIPTION

A CSC has (at least) the following member fields, none of which
ought to be accessed directly from outside the class.

  player  reference to TSH::Player object

The following member functions are currently defined.

=over 4

=cut

sub GetBoth ($$);
sub GetHTML ($$);
sub GetText ($$);
sub RenderTextTitle ($);
sub Compose ($);
sub new ($$);

=item $logp = $csc->Compose(\%options);

Store CSC in a TSH::Log object in preparation for rendering it as
text and/or HTML.  No options are currently supported.

=cut

sub Compose ($) {
  my $this = shift;
  my $optionsp = shift;
  my $p = $this->{'player'};
  my $dp = $p->Division();
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my $tables = $config->Value('tables');
  my $noboards = $config->Value('no_boards');
  $tables = $tables->{$dp->Name()} if $tables;
  my $has_rating = $dp->RatingSystem() ne 'none';

  my @html_cells1;
  my @html_cells2;
  my @text_cells;
  my $logp = new TSH::Log($tournament, $dp, 'scorecard', 
    'p' . $p->ID(), {
      'title' => "Contestant Scorecard",
      'htmltitle' => '',
      'texttitle' => $this->RenderTextTitle(),
      # no file I/O, request rendered string
      'noconsole' => 1,
      'nohtml' => 1,
      'notext' => 1,
      'nowrapper' => 1,
#     'notitle' => 1,
      'notop' => 1,
    });

  # HTML banner head
  {
    $logp->ColumnAttributes($this->{'html_attributes'}[0]);
    $logp->ColumnClasses($this->{'html_classes'}[0], 'top1');
    my @html_cells0 = (
      $p->FullID(), 
      '<span class="label">Name:</span> ' . $p->PrettyName(),
      $has_rating ? '<span class="label">Rating:</span> ' . $p->Rating() : '',
      );
    $logp->WriteTitle([], \@html_cells0);
  }

  # column titles
  $logp->ColumnClasses($this->{'text_classes'});
  $logp->WriteTitle($this->{'text_titles'}, []);
  $logp->ColumnAttributes([]);
  $logp->ColumnClasses($this->{'html_classes'}[1], 'top2');
  $logp->WriteTitle([], $this->{'html_titles'});
  
  # print one line for each paired or played round
  my $max_round0 = $dp->MaxRound0();
  my $maxround1 = (defined $max_round0) 
    ? $max_round0 + 1
    : ($p->CountOpponents() > $p->CountScores() 
    ? $p->CountOpponents() 
    : $p->CountScores());
  for my $r1 (1..$maxround1) {
    push(@html_cells1, $r1);

    @html_cells2 = ('1st 2nd');
    push(@html_cells2, '<span class="label">cumulative</span>')
      unless $this->{'spread_entry'};
    push(@html_cells2, '<span class="strut">&nbsp;</span>');

    my $r0 = $r1 - 1;
    my $oid = $p->OpponentID($r0);
    unless (defined $oid) {
      push(@html_cells1, '', '', '', '');
      push(@html_cells1, '') unless $noboards;
      push(@html_cells1, '', '') unless $this->{'spread_entry'};
      push(@html_cells1, '');
      next;
      }
    push(@text_cells, $r1);
    my $opp = $p->Opponent($r0);
    if ($oid && !$opp) {
      $dp->Tournament()->TellUser('ebadtpn', $p->ID(), $oid);
      return;
      }
    my $score = $p->Score($r0);
    my $p12 = $p->First($r0);
    if ($this->{'track_firsts'}) {
      if (defined $p12) {
	push(@text_cells, ' '.('-',"$p12","$p12",'?', '??')[$p12]);
	$html_cells2[0] = ('-','1st&nbsp;&nbsp;&nbsp;&nbsp;',
	  '&nbsp;&nbsp;&nbsp;&nbsp;2nd','1st 2nd','1st 2nd')[$p12];
	}
      else {
	push(@text_cells, '');
	}
      }
    unless ($noboards) {
      my $bort = $p->Board($r0);
      if ($bort && $tables) {
	$bort = $tables->[$bort-1];
	}
      $bort ||= '';
      push(@text_cells, $bort);
      push(@html_cells1, $bort);
      }
    if ($oid) {
      my $repeats = $p->CountRoundRepeats($opp, $r0);
      $repeats = $repeats > 1 ? "*$repeats" : '';
      my $oname = $opp->PrettyName() . $repeats;
      push(@text_cells, $opp->ID());
      push(@text_cells, $opp->Rating()) if $has_rating;
      push(@text_cells, $oname);
      push(@html_cells1, $opp->ID());
      push(@html_cells1, $opp->Rating()) if $has_rating;
      push(@html_cells1, $oname);
      }
    else {
      push(@text_cells, '', '', 'bye');
      push(@html_cells1, '-', '-', 'bye');
      }
    if (defined $score) {
      my $oscore = $p->OpponentScore($r0) || 0;
      my $wins = $p->RoundWins($r0);
      my $losses = $p->RoundLosses($r0);
      push(@text_cells, sprintf("%.1f-%.1f", $wins, $losses));
      push(@html_cells1, $wins, $losses);
      unless ($this->{'spread_entry'}) {
	push(@text_cells, $score, $oscore);
	push(@html_cells1, $score, $oscore);
        }
      my $spread = sprintf("%+d", $score - $oscore);
      my $cume = sprintf("%+d", $p->RoundSpread($r0));
      push(@text_cells, $spread, $cume);
      push(@html_cells1, $spread);
      $html_cells2[-1] .= $cume;
      }
    else {
      push(@html_cells1, '', '');
      push(@html_cells1, '', '') unless $this->{'spread_entry'};
      push(@html_cells1, '');
      }
    } # for $r0
  continue {
    $logp->ColumnClasses($this->{'text_classes'});
    $logp->WriteRow(\@text_cells, []);
    $logp->ColumnAttributes($this->{'html_attributes'}[1]);
    $logp->ColumnClasses($this->{'html_classes'}[1], 'round1');
    $logp->WriteRow([], \@html_cells1);
    $logp->ColumnAttributes($this->{'html_attributes'}[2]);
    $logp->ColumnClasses($this->{'html_classes'}[2], 'round2');
    $logp->WriteRow([], \@html_cells2);
    @text_cells = ();
    @html_cells1 = ();
    @html_cells2 = ();
    }

  $logp->ColumnAttributes(
    [$this->{'spread_entry'} ? 'colspan=9' : 'colspan=10']);
  $logp->ColumnClasses([],'bottom');
  $logp->WriteRow([], ['&nbsp;']);
  return $logp;
  }

=item my ($text, $html) = $csc->GetBoth(\%options);

Return text and HTML rendering of the CSC.
No options are currently supported.

=cut

sub GetBoth ($$) {
  my $this = shift;
  my $optionsp = shift;
  my $logp = $this->Compose($optionsp);
  return ($logp->RenderText(), $logp->RenderHTML());
  }

=item $html = $csc->GetHTML(\%options);

Return HTML rendering of the CSC as the rows of a table.
No options are currently supported.

=cut

sub GetHTML ($$) {
  my $this = shift;
  my $optionsp = shift;
  return $this->Compose($optionsp)->RenderHTML();
  }

=item $html = $csc->GetText(\%options);

Return text rendering of the CSC.
No options are currently supported.

=cut

sub GetText ($$) {
  my $this = shift;
  my $optionsp = shift;
  return $this->Compose($optionsp)->RenderText();
  }

sub initialise ($$) {
  my $this = shift;
  my $p = shift;
  my $dp = $p->Division();
  my $config = $dp->Tournament()->Config();
  my $has_rating = $config->Value('rating_system') ne 'none';
  # TODO: should cache the above value
  my $noboards = $config->Value('no_boards');
  my $tables = $config->Value('tables');
  # store some configuration variables to avoid possible future race conditions
  $tables = $tables->{$dp->Name()} if $tables;
  $this->{'tables'} = $tables;
  $this->{'spread_entry'} = $config->Value('entry') eq 'spread';
  $this->{'track_firsts'} = $config->Value('track_firsts');
  $this->{'player'} = $p;
  # set up and store table layout variables
  my @html_attributes0; # banner at top of CSC
  my @html_attributes1; # column headings
  my @html_attributes2; # second line of data rows
  my @html_classes0; # banner at top of CSC
  my @html_classes1; # column headings
  my @html_classes2; # second line of data rows
  my @html_titles; # column headings
  my @text_classes;
  my @text_titles;

  push(@html_attributes0, 'colspan=3');
  push(@html_classes0, qw(number name rating));

  push(@text_classes, q(round));
  push(@text_titles, q(Rnd));
  push(@html_titles, q(Round));
  push(@html_attributes1, '');
  push(@html_classes1, qw(round));
  push(@html_classes2, qw(p12));

  if ($this->{'track_firsts'}) {
    push(@text_titles, q(1/2));
    push(@text_classes, q(p12));
    }

  unless ($noboards) {
    push(@html_attributes1, 'rowspan=2');
    if ($tables) {
      push(@text_classes, 'table');
      push(@text_titles, 'Tbl');
      push(@html_classes1, qw(table));
      push(@html_titles, $config->Value('table_title'));
      }
    else {
      push(@text_classes, 'board');
      push(@text_titles, 'Bd');
      push(@html_classes1, qw(board));
      push(@html_titles, 'Board');
      }
    }

  push(@text_classes, qw(onum));
  push(@text_classes, qw(orat)) if $has_rating;
  push(@text_classes, qw(onam wl));
  push(@text_titles, qw(Opp));
  push(@text_titles, qw(Rtng)) if $has_rating;
  push(@text_titles, q(Opponent Name), qw(Won-Lost));
  push(@html_attributes1, ('rowspan=2') x (4 + ($has_rating ? 1 : 0)));
  push(@html_classes1, qw(onum));
  push(@html_classes1, qw(orat)) if $has_rating;
  push(@html_classes1, qw(onam won lost));
  push(@html_titles, 'Opp. #');
  push(@html_titles, 'Rating') if $has_rating;
  push(@html_titles, 'Opponent Name', 'Won', 'Lost');

  if ($this->{'spread_entry'}) {
    push(@html_attributes0, 'colspan=3');
    push(@html_classes2, qw(cume2));
    }
  else {
    push(@text_titles, qw(For Agn));
    push(@text_classes, qw(score score));
    push(@html_attributes0, 'colspan=5');
    push(@html_attributes1, 'rowspan=2', '');
    push(@html_classes1, qw(for against));
    push(@html_titles, 'Player<br>Score', 'Opponent<br>Score');
    push(@html_classes2, qw(cumelabel cume));
    }

  push(@text_titles, qw(Sprd Cumul));
  push(@text_classes, qw(spread spread));
  push(@html_attributes0, 'colspan=2');
  push(@html_classes1, qw(spread));
  push(@html_titles, qw(Spread));

  $this->{'html_attributes'} 
    = [\@html_attributes0, \@html_attributes1, \@html_attributes2];
  $this->{'html_classes'} 
    = [\@html_classes0, \@html_classes1, \@html_classes2];;
  $this->{'html_titles'} = \@html_titles;
  $this->{'text_classes'} = \@text_classes;
  $this->{'text_titles'} = \@text_titles;
  }

sub new ($$) { return TSH::Utility::new(@_); }

=item $text = $this->RenderTextTitle();

Used internally to render the title area in text mode.

=cut

sub RenderTextTitle ($) {
  my $this = shift;
  my $p = $this->{'player'};
  my $config = $p->Division()->Tournament()->Config();
  my $has_rating = $config->Value('rating_system') ne 'none';
  my $text_title = 'Player Scorecard: '. $p->TaggedName()
    . ($has_rating ? " (" . $p->Rating() . ")" : '');
  if ($this->{'track_firsts'}) {
    my $firsts = $p->Firsts()|| 0;
    my $seconds = $p->Seconds() || 0;
    $text_title .= " (S${firsts}:R$seconds)";
    }
  if (my $lifeg = $p->LifeGames()) {
    $text_title .= " (LG$lifeg)";
    }
  if (!$p->Active()) {
    my $spread = $p->OffSpread();
    $text_title .= " INACTIVE";
    $text_title .= "[$spread]" unless $spread == -$config->Value('bye_spread');
    }
  $text_title .= "\n";
  if ($p->{'etc'}{'time'}) {
    my $age = int((time - $p->{'etc'}{'time'}[0])/60);
    if ($age < 200) {
      my $s = $age == 1 ? '' : 's';
      $text_title .= "- Last score was entered $age minute$s ago.\n";
      }
    }
  return $text_title;
  }

=back

=cut

1;
