#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RANDomscores;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RANDomscores - implement the C<tsh> RANDomscores command

=head1 SYNOPSIS

  my $command = new TSH::Command::RANDomscores;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RANDomscores is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to add a round of random scores to a division
for testing purposes.
EOF
  $this->{'names'} = [qw(rand randomscores)];
  $this->{'argtypes'} = [qw(Divisions)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();

  for my $dp (@_) {
    my $distrib = $dp->RatingSystem() =~ /^absp$/i ? 'absp' : 'nsa';
    $tournament->TellUser('irandok', $dp->Name());
    my $round1 = $dp->LeastScores() + 1;
    for my $p ($dp->Players()) {
      next if $p->CountScores() >= $round1;
      next if $p->CountScores() >= $p->CountOpponents();
      if ($p->{'pairings'}[$#{$p->{'scores'}}+1]) {
	my $rating = $p->Rating();
	push(@{$p->{'scores'}}, 
	  $distrib eq 'nsa'
	    # based on the NSA Club #3 2004-2005 season
	    ? int(130*log(($rating||1000)/75)
	      +rand(200)-100)
	    # based on nothing
            : int(250+rand(50+($rating>200?200:$rating))));
	if ($config::track_firsts) {
	  my $r0 = $#{$p->{'scores'}};
	  my $first = int(rand(2));
	  my $p12p = $p->{'etc'}{'p12'};
	  if ($p12p->[$r0] != 1 && $p12p->[$r0] != 2) {
	    $p12p->[$r0] = 1 + $first;
	    }
	  $p12p = $p->Opponent($r0)->{'etc'}{'p12'};
	  if ($p12p->[$r0] != 1 && $p12p->[$r0] != 2) {
	    $p12p->[$r0] = 2 - $first;
	    }
	  }
	}
      else {
	push(@{$p->{'scores'}}, $config->Value('bye_spread'));
        }
      printf "%s: @{$p->{'scores'}}\n", (TSH::Utility::TaggedName $p);
      }
    $dp->Dirty(1);
    $this->Processor()->Flush();
    }
  $tournament->TellUser('idone');
  }

=back

=cut

1;
