#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Player;

use strict;
use warnings;

use Carp;
use TSH::Utility qw(Debug DebugDumpPairings);
use JavaScript::Serializable;
use threads::shared;

our (@ISA);
@ISA = qw(JavaScript::Serializable);
# sub EXPORT_JAVASCRIPT () { return map { $_ => $_ } qw(etc name pairings photo rating rrank rwins rlosses rspread scores); }
sub EXPORT_JAVASCRIPT () { return map { $_ => $_ } qw(etc name pairings photo rating scores); }

=pod

=head1 NAME

TSH::Player - abstraction of a Scrabble player within C<tsh>

=head1 SYNOPSIS

  $p = new TSH::Player();
  $boolean = CanBePaired $repeats, \@players, $must_catchup, $setupp
  $s = $p->CountOpponents();
  $s = $p->CountScores();
  $n = $p->CountRepeats();
  $s = $p->DeleteLastScore();
  $s = $p->Division();
  $p12 = $p->First($r0);
  $oldp12 = $p->First($r0, $p12);
  $n = $p->Firsts();
  $oldn = $p->Firsts($n);
  $oldp12p = $p->FirstVector($newp12p);
  $p12p = $p->FirstVector();
  $s = $p->FullID();
  $s = $p->GamesPlayed();
  $s = $p->ID();
  $s = $p->Initials();
  $r = $p->MaxRank();
  $p->MaxRank($r);
  $s = $p->Name();
  $r = $p->NewRating();
  $p->NewRating($r);
  $s = $p->Opponent($round0);
  $s = $p->OpponentID($round0);
  $s = $p->OpponentScore($round0);
  $success = PairGRT($grtsub, $psp, $grtargs, $round0);
  $s = $p->PrettyName();
  $r = $p->Random();
  $p->Random($r);
  $r = $p->Rating();
  $p->Rating($r);
  $r = $p->Repeats($oppid);
  $p->Repeats($oppid, $r);
  $boolean = ResolvePairings $unpairedp[, \%options]
  $s = $p->RoundLosses($round0);
  $s = $p->RoundRank($round0);
  $p->RoundRank($round0, $rank);
  $s = $p->RoundSpread($round0);
  $s = $p->RoundWins($round0);
  $s = $p->Score($round0);
  $old = $p->Score($round0, $score);
  $n = $p->Seconds();
  $oldn = $p->Seconds($n);
  $n = $p->Time();
  $oldn = $p->Time($n);
  @p = TSH::Player::SortByCurrentStanding(@p);
  @p = TSH::Player::SortByInitialStanding(@p);
  @p = TSH::Player::SortByStanding($sr0, @p);
  TSH::Player::SpliceInactive(@ps, $nrounds, $round0);
  $s = $p->TaggedName();
  $s = $p->Team();
  $j = $p->ToJavaScript();
  $success = $p->UnpairRound($round0);
  $n = $p->UnscoredGames();
  $n = $p->Wins();

=head1 ABSTRACT

This Perl module is used to manipulate players within C<tsh>.

=head1 DESCRIPTION

A Player has (at least) the following member fields, none of which
ought to be accessed directly from outside the class.

  byes        number of byes, last time we counted
  class       class for prizes
  cspread     cumulative spread, capped (see rcspread)
  division    pointer to division
  etc         supplementary player data (see below)
  ewins1      rated wins earned in the first half of a split tournament
  ewins2      rated wins earned in the second half of a split tournament
  id          player ID (1-based) # not sure this is still here
  initials    two letters to help identify a player
  losses      number of losses
  maxrank     highest rank still attainable by player in this tournament
  name        player name
  noscores    # of undefined values in scores, up to $config::max_rounds
  nscores     number of defined values in scores
  opp         provisional opponent used within pairing routines
  p1          number of firsts (starts)
  p2          number of seconds (replies)
  p3          number of indeterminates (starts/replies)
  pairings    list of 1-based opponent IDs by 0-based round
  photo       URL for player photo
  prettyname  player name formatted for human eyes
  ratedgames  number of rated games
  ratedwins   number of rated wins
  rating      pre-tournament rating
  rcrank      1-based capped rank by 0-based round (round 0 = preevent)
  rcspread    cumulative spread by round, capped by standings_spread_cap
  repeats     data structure tracking repeat pairings
  rnd         pseudorandom value used to break ties in standings
  rrank       1-based rank by 0-based round (round 0 = preevent)
  rlosses     cumulative losses by round
  rspread     cumulative spread by round
  rwins       cumulative wins by round
  scores      list of this player's scores by round (0-based)
  spread      cumulative spread
  tspread     temporary spread variable used by some routines
  twins       temporary wins variable used by some routines
  wins        number of wins
  x*          keys beginning x are for scratch data

Supplementary player data is currently as follows:

  board       0-based list indicating board at which player played each 
              round. A value of 0 indicates that no board is assigned.
  off         exists if player is inactive, single value indicates
              type of byes (-50/0/50) to be assigned
  penalty     0-based list indicating spread penalty applied to player 
              in each round. A value of 0 indicates no penalty,
	      a negative value indicates spread deducted.
  p12         0-based list, 1 if went first, 2 if second, 0 if neither
              (bye), 3 if must draw, 4 if indeterminate
  team        Team name
  time        Seconds since Unix epoch when player data was last changed

The following member functions are currently defined.

=over 4

=cut

sub Active ($);
sub Board ($$;$);
sub Byes ($;$);
sub CanBePaired ($$$$);
sub CountOpponents ($);
sub CountScores ($);
sub DeleteLastScore ($);
sub Division ($);
sub First ($$;$);
sub Firsts ($;$);
sub FirstVector ($;$);
sub FullID ($);
sub GamesPlayed ($);
sub GetOrSetEtcScalar ($$;$);
sub GetOrSetEtcVector ($$;$);
sub ID ($);
sub Initials ($);
sub initialise ($);
sub IsThai ($);
sub LifeGames ($;$);
sub Losses ($);
sub MaxRank ($;$);
sub new ($);
sub Name ($);
sub NewRating ($;$);
sub OffSpread ($);
sub Opponent ($$);
sub OpponentID ($$);
sub OpponentScore ($$);
sub PairGRT ($$$$$;$);
sub Password ($;$);
sub PrettyName ($);
sub Rating ($;$);
sub Repeats ($$;$);
sub ResolvePairings ($$);
sub RoundCappedRank ($$;$);
sub RoundRank ($$;$);
sub RoundSpread ($$);
sub RoundWins ($$);
sub Score ($$;$);
sub Seconds ($;$);
sub SortByCappedStanding ($@);
sub SortByCurrentStanding (@);
sub SortByHandicap ($@);
sub SortByInitialStanding (@);
sub SortByStanding ($@);
sub SpliceInactive (\@$$);
sub Spread ($);
sub TaggedName ($);
sub Time ($;$);
sub UnpairRound ($$);
sub Wins ($);

=item $n = $p->Active();

Return true if the player is active for pairings.

=cut

sub Active ($) { 
  my $this = shift;
  return !exists $this->{'etc'}{'off'};
  }

=item $n = $p->Byes();
=item $newn = $p->Byes($n);

Set or get the number of byes a player has had.

=cut

sub Byes ($;$) { TSH::Utility::GetOrSet('byes', @_); }

=item $n = $p->Board($r0);

Set/get the board number at which the player played in zero-based round $r0.

=cut

sub Board ($$;$) { 
  my $this = shift;
  my $r0 = shift;
  my $newboard = shift;
  my $boardp = $this->{'etc'}{'board'};
  unless (defined $boardp) { $this->{'etc'}{'board'} = $boardp = &share([]); }
  my $oldboard = $boardp->[$r0] || 0;
  if (defined $newboard) {
    if ($r0 > $#$boardp + 1) {
      push(@$boardp, (0) x ($r0 - $#$boardp - 1));
      }
    $boardp->[$r0] = $newboard;
    $this->Division()->Dirty(1);
    }
  return $oldboard;
  }

=item $boolean = CanBePaired $repeats, \@players, $ketchup, $setupp

Returns true iff @players can be paired without exceeding C<$repeats>.

If $ketchup is true, the top players can only be paired with those who
can catch up to them.

If $setupp->{'exagony'} is true, players from the same team cannot
play each other.

This routine is a bottleneck, and we've tried doing the following

OPT-A: Rearranging (1,2,3,4,...,n) to (1,n,2,n-1,3,n-2,4,n-3,...)
improves runtime by two orders of magnitude in bad cases, as looking
first at the top and bottom players typically works on the toughest
cases first.  We don't use this anymore because of the reduction
in pairings quality.  (Huh? This routine is just testing for pairability.)

OPT-B: Randomly rolling player opponent preferences will somewhat
improve mean runtime, and substantially improve runtime when 
low-numbered players are in high demand.  The worst-case runtime
is still terrible, though.  We don't use this anymore because of
the reduction in pairings quality, and the poor worst-case behaviour.

We currently do the optimization in ResolvePairings, where
we sort players by how difficult they are to pair, so that for
example players who have only one possible opponent are all paired
first.  

=cut

sub CanBePaired ($$$$) { 
  my $repeats = shift;
  my $psp = shift;
  my $must_catchup = shift;
  my $setupp = shift;

  my (@ps) = (@$psp);
  my @shuffled;
# OPT-A while (@ps) {
# OPT-A   push(@shuffled, shift @ps);
# OPT-A   push(@shuffled, pop @ps) if @ps;
# OPT-A   }
  @shuffled = @ps;
  for my $i (0..$#shuffled) {
    my @prefs : shared
      = grep { 
	# must not exceed repeats criterion
	$shuffled[$i]->Repeats($_->ID()) <= $repeats 
	# must be able to catch up
        && ($must_catchup ? do {
	  my $p1 = $shuffled[$i];
	  my $p2 = $_;
	  if ($p1->RoundRank(-2) > $p2->RoundRank(-2)) {
	    ($p2, $p1) = ($p1, $p2);
	    }
	  Debug 'CBP', '%s %d ?>= %d %s', $p1->Name(), $p1->RoundRank(-2), $p2->MaxRank(), $p2->Name();
	  $p1->RoundRank(-2) >= $p2->MaxRank();
	  } : 1) 
	&& ($setupp->{'exagony'}
	  ? $shuffled[$i]->Team() ne $_->Team() 
	  || $shuffled[$i]->Team() eq ''
	  : 1)
        }
      (@shuffled[0..$i-1,$i+1..$#shuffled]);
    # randomly roll preferences
# OPT-B  unshift(@prefs, splice(@prefs, rand(@prefs))); # not thread-safe
#   Debug 'CBP', "prefs[%d]: %s", $i, join(',',map { $_->Name() } @prefs);
    $shuffled[$i]{'pref'} = \@prefs;
    }
# Debug 'CBP', "Resolving pairings";
  return ResolvePairings \@shuffled, {'target'=>undef};
  }

=item $c = $p->Class();

=item $p->Class($c);

Set or get the player's class.

=cut

# sub Class ($;$) { TSH::Utility::GetOrSet('class', @_); }

sub Class ($;$) { 
  my $this = shift;
  my $new = shift;
  return $this->GetOrSetEtcScalar('class', $new);
  }


=item $n = $p->CountOpponents();

Returns the number of the last round in which this player has a
scheduled pairing (including byes).

=cut

sub CountOpponents ($) { 
  my $this = shift;
  return scalar(@{$this->{'pairings'}});
  }

=item $n = $p->CountRepeats($opp);

Returns the correct number of repeat pairings between two players,
even before Division::Synch is called.

=cut

sub CountRepeats ($$) { 
  my $this = shift;
  my $opp = shift;
  my $oid = $opp->{'id'};
  my $repeats = 0;
  for my $aid (@{$this->{'pairings'}}) {
    $repeats++ if (defined $aid) && $aid == $oid;
    }
  return $repeats;
  }

=item $n = $p->CountRoundRepeats($opp, $r0);

Returns the number of repeat pairings between two players,
up to and including zero-based round C<$r0>.

=cut

sub CountRoundRepeats ($$) { 
  my $this = shift;
  my $opp = shift;
  my $r0 = shift;
  my $oid = $opp->{'id'};
  my $repeats = 0;
  my $pairingsp = $this->{'pairings'};
  for my $i (0..$#$pairingsp) {
    my $aid = $pairingsp->[$i];
    $repeats++ if (defined $aid) && $aid == $oid;
    last if $i >= $r0;
    }
  return $repeats;
  }

=item $n = $p->CountScores();

Returns the number of the last round in which this player has a
recorded score.

=cut

sub CountScores ($) { 
  my $this = shift;
  return scalar(@{$this->{'scores'}});
  }

=item $n = $p->DeleteLastScore();

Deletes the player's last score. 
Should be used with extreme caution, and only after checking with
the player's opponent if any.

=cut

sub DeleteLastScore ($) { 
  my $this = shift;
  return pop @{$this->{'scores'}};
  }

=item $n = $p->Division();

Return the player's division.

=cut

sub Division ($) { 
  my $this = shift;
  return $this->{'division'};
  }

=item $e = $p->Expiry();
=item $p->Expiry($e);

Set or get the player's expiry, a rating system-dependent value.

=cut

sub Expiry ($;$) { TSH::Utility::GetOrSet('expiry', @_); }

=item $n = $p->First($round0[, $newvalue]);

Returns a number indicating whether the player did or will go
first in a zero-based round.

  0: did not play this round
  1: went first (started)
  2: went second (replied)
  3: must draw
  4: unknown pending earlier firsts/seconds or other information

=cut

sub First ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newp12 = shift;
  my $p12sp = $this->{'etc'}{'p12'};
  if (!defined $p12sp) {
    $p12sp = $this->{'etc'}{'p12'} = &share([]);
    }
  my $oldp12 = $p12sp->[$round0];
  $oldp12 = 4 unless defined $oldp12;
  if (defined $newp12) {
    if ($round0 < 0 || $round0 > $#$p12sp+1) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'p12', $this->TaggedName(), $round0+1, $newp12);
      }
    elsif ($newp12 !~ /^[0-4]$/) {
      $this->Division()->Tournament->TellUser('eplyrbv',
	'p12', $this->TaggedName(), $round0+1, $newp12);
      }
    else {
      $p12sp->[$round0] = $newp12;
#     print "set p12 for $this->{'id'} in r0 $round0 to $newp12\n";
      }
    }
  return $oldp12;
  }

=item $n = $p->Firsts();

=item $newn = $p->Firsts($n);

Set or get the number of firsts a player has had.
Is called to set value in TSH::Division::SynchFirsts.

=cut

sub Firsts ($;$) { TSH::Utility::GetOrSet('p1', @_); }

=item $n = $p->FullID();

Return the player's 1-based ID number, formatted by
prepending either the division name (if there is more than
one division) or a number sign.

=cut

sub FullID ($) { 
  my $this = shift;
  my $dname = '';
  my $tournament = $this->Division()->Tournament();
  if ($tournament->CountDivisions() > 1) {
    $dname = $this->Division()->Name();
    $dname .= '-' if $dname =~ /\d$/;
    }
  my $fmt = $tournament->Config()->Value('player_number_format');
  return sprintf("%s$fmt", uc($dname), $this->{'id'});
  }

=item $p12p = $p->FirstVector();

=item $oldp12p = $p->Firsts($newp12p);

Returns a reference to a list of values (0-indexed by round)
as described in First(), or stores the list.

=cut

sub FirstVector ($;$) { 
  my $this = shift;
  my $newp12sp = shift;
  my $p12sp = $this->{'etc'}{'p12'};
  if (!defined $p12sp) {
    $p12sp = $this->{'etc'}{'p12'} = [];
    }
  if (defined $newp12sp) {
    $this->{'etc'}{'p12'} = $newp12sp;
    }
  return $p12sp;
  }

=item $n = $p->GamesPlayed();

Return the number of games a player has played, not including byes,
forfeits or unscored games.

=cut

sub GamesPlayed ($) { 
  my $this = shift;
  my $n = 0;
  my $pairingsp = $this->{'pairings'};
  my $scoresp = $this->{'scores'};
  for my $r0 (0..$#$scoresp) {
#   if ((defined $scoresp->[$r0]) && (defined $pairingsp->[$r0])) {
    if ((defined $scoresp->[$r0]) && $pairingsp->[$r0]) {
      $n++;
      }
    }
  return $n;
  }

=item $n = $p->GetOrSetEtcScalar($key[, $value]);

Get or set a miscellaneous scalar data field.

=cut

sub GetOrSetEtcScalar ($$;$) { 
  my $this = shift;
  my $key = shift;
  my $new = shift;
  my $old = $this->{'etc'}{$key};
  if (ref($old)) {
    $old = $old->[0];
    if (defined $new) {
      $this->{'etc'}{$key}[0] = $new;
      $this->Division()->Dirty(1);
      }
    }
  else {
    if (defined $new) {
      my @new : shared = ($new);
      $this->{'etc'}{$key} = \@new;
      $this->Division()->Dirty(1);
      }
    $old = undef;
    }
  return $old;
  }

=item $l = $p->GetOrSetEtcVector($key[, $value]);

Get or set a miscellaneous vector data field.

=cut

sub GetOrSetEtcVector ($$;$) { 
  my $this = shift;
  my $key = shift;
  my $new = shift;
  my $old = $this->{'etc'}{$key};
  if (ref($old)) {
    return $old;
    if (defined $new) {
      @{$this->{'etc'}{$key}} = @$new;
      }
    }
  else {
    if (defined $new) {
      my @new : shared = @$new;
      $this->{'etc'}{$key} = \@new;
      }
    $old = undef;
    }
  return $old;
  }

=item $n = $p->ID();

Return the player's 1-based ID number.

=cut

sub ID ($) { 
  my $this = shift;
  return $this->{'id'};
  }

=item $s = $p->Initials();

Return the player's initials

=cut

sub Initials ($) { 
  my $this = shift;
  unless ($this->{'initials'}) {
    my ($last, $first);
    my $name = uc $this->{'name'};
    if ($name =~ /,/) { ($last, $first) = split(/,\s*/, $name, 2); }
    else { ($first, $last) = $name =~ /^(.*\S)(?:\s+)(.*)$/; }
    $first =~ s/^.*?([A-Z]).*/$1/;
    $first = ' ' unless $first;
    $last =~ s/^.*?([A-Z]).*/$1/;
    $last = ' ' unless $last;
    $this->{'initials'} = "$first$last";
    }
  return $this->{'initials'};
  }

=item $d->initialise();

(Re)initialise a Player object, for internal use.

=cut

sub initialise ($) {
  my $this = shift;
  $this->{'name'} = '';
  }

=item $old = $p->LifeGames([$games]);

Get or set the player's life (career) games.

=cut

sub LifeGames ($;$) { 
  my $this = shift;
  my $new = shift;
  return $this->GetOrSetEtcScalar('lifeg', $new);
  }

=item $n = $p->Losses();

Return the player's total losses so far this tournament.

=cut

sub Losses ($) { 
  my $this = shift;
  return ($this->{'losses'} || 0);
  }

=item $r = $p->MaxRank();

=item $p->MaxRank($r);

Set or get the player's maximum attainable ranking.

=cut

sub MaxRank ($;$) { TSH::Utility::GetOrSet('maxrank', @_); }

=item $d = new Player;

Create a new Player object.  

=cut

sub new ($) { return TSH::Utility::newshared(@_); }

=item $n = $p->Name();

Return the player's name.

=cut

sub Name ($) { 
  my $this = shift;
  my $name = ($this->{'name'} || '?');
  $name =~ s/,\s*$//; # kludge to allow names ending in digits
  return $name;
  }

=item $r = $p->NewRating();

=item $p->NewRating($r);

Set or get the player's newly calculated rating estimate

=cut

sub NewRating ($;$) { my $this = shift; $this->GetOrSetEtcScalar('newr', @_); }

=item $n = $p->OffSpread();

Return the spread being assigned to an inactive player's games.

=cut

sub OffSpread ($) {
  my $this = shift;
  return $this->{'etc'}{'off'}[0];
  }

=item $n = $p->Opponent($round0);

Return the player's opponent in 0-based
round $round0.

=cut

sub Opponent ($$) {
  my $this = shift;
  my $round0 = shift;
  my $oppid = $this->OpponentID($round0);
  if ($oppid) {
    return $this->Division()->Player($oppid);
    }
  else {
    return undef;
    }
  }

=item $n = $p->OpponentID($round0);

Return the 1-based ID number of the player's opponent in 0-based
round $round0, or undef if the pairing has yet been assigned.

=cut

sub OpponentID ($$) { 
  my $this = shift;
  my $round0 = shift;
# confess unless defined $round0;
  return $this->{'pairings'}[$round0];
  }

=item $n = $p->OpponentScore($round0);

Return the player's opponent's score in 0-based
round $round0.

=cut

sub OpponentScore ($$) {
  my $this = shift;
  my $round0 = shift;
  my $opp = $this->Opponent($round0);
  if ($opp) {
    return $opp->Score($round0);
    }
  else {
    return undef;
    }
  }

=item $success = PairGRT($psp, $grtsub, $filter, \@grtargs, $round0[, \%rpopts]);

Pair the players listed in @$psp allowing each player to be paired
only with the opponents that pass &$filter, ranking them in preference
order according to the Guttman-Rosler Transform sub $grtsub, which
must pack player index in $psp as an 'N' at the end of its output
string.  $grtsub and $filter are passed the following arguments:

  - $psp 
  - index of current player in $psp
  - index of opponent in $psp
  - @grtargs

If $round0 is defined, pairings are stored in that zero-based round.
If $round0 is undefined, pairings are not stored, but status is still
returned based on whether or not pairings could be computed.

Arguments in %rpopts are passed to ResolvePairings.

=cut

sub PairGRT ($$$$$;$) { 
  my $psp = shift;
  my $grt = shift;
  my $filter = shift;
  my $argsp = shift;
  my $r0 = shift;
  my $rpopts = shift;
  return 1 unless @$psp;
# Debug 'GRT', 'PGRT psp: %s', join(',',map { $_->ID() } @$psp);
  for my $i (0..$#$psp) {
    my $p = $psp->[$i];
    my @opps : shared =
      # Guttman-Rosler Transform
      map { $psp->[unpack('N', substr($_, -4))] }
      sort 
      map { &$grt($psp, $i, $_, @$argsp) }
      grep { &$filter($psp, $i, $_, @$argsp) }
      (0..$i-1,$i+1..$#$psp);
    Debug 'GRT', 'pref %d: %s', $p->ID(), join(',',map { 
	defined $_ ? $_->ID() : '???';
#	$_->ID() 
      } @opps);
    $psp->[$i]{'pref'} = \@opps;
    }
  my (%rpopts) = $rpopts ? %$rpopts : ();
  $rpopts{'target'} = $r0 unless exists $rpopts{'target'};
  if (ResolvePairings($psp, \%rpopts)) {
    DebugDumpPairings 'GRT', $psp->[0]->CountOpponents()-1, $psp
      if defined $r0;
    return 1;
    }
  else {
    return 0;
    }
  }

=item $old = $p->Password([$password]);

Get or set the player's data entry password.

=cut

sub Password ($;$) { 
  my $this = shift;
  my $new = shift;
  return $this->GetOrSetEtcScalar('password', $new);
  }

=item $n = $p->Penalty($r0[, -$x]);

Set/get the spread penalty applied to the player played in zero-based round $r0.

=cut

sub Penalty ($$;$) { 
  my $this = shift;
  my $r0 = shift;
  my $newpenalty = shift;
  my $penaltyp = $this->{'etc'}{'penalty'};
  unless (defined $penaltyp) { $this->{'etc'}{'penalty'} = $penaltyp = &share([]); }
  my $oldpenalty = $penaltyp->[$r0] || 0;
  if (defined $newpenalty) {
    if ($r0 > $#$penaltyp + 1) {
      push(@$penaltyp, (0) x ($r0 - $#$penaltyp - 1));
      }
    $penaltyp->[$r0] = $newpenalty;
    $this->Division()->Dirty(1);
    }
  return $oldpenalty;
  }

=item $p = $p->PrettyName();

Return the player's name formatted for human eyes.  The raw name
as stored in the .t file should be in "surnames, given names" format
for easy conversion to NSA ratings list format (which is "surnames
given_names", with no indication of how the names should be separated.
The "pretty name" will be "given_names surnames" if there is a comma
in the raw name and "config surname_last" is set,
else the raw name unprocessed.

=cut

sub PrettyName ($) {
  my $this = shift;
  if (!exists $this->{'prettyname'}) {
    my $name = $this->Name();
    $name =~ s/^Zxqkj, Winter$/Winter/;
    if ($this->Division()->Tournament->Config()->Value('surname_last')) {
      $name =~ s/^([^,]+), (.*)$/$2 $1/;
      }
    $this->{'prettyname'} = $name;
    }
  return $this->{'prettyname'};
  }

=item $u = $p->PhotoURL();
=item $p->PhotoURL($u);

Set or get the player's photos URL

=cut

sub PhotoURL ($;$) { TSH::Utility::GetOrSet('photo', @_); }

=item $r = $p->Random();
=item $p->Random($r);

Set or get the player's random value.

=cut

sub Random ($;$) { TSH::Utility::GetOrSet('rnd', @_); }

=item $r = $p->Rating();
=item $p->Rating($r);

Set or get the player's rating

=cut

sub Rating ($;$) { TSH::Utility::GetOrSet('rating', @_); }

=item $n = $p->Repeats($oppid);

=item $n = $p->Repeats($oppid, $n);

Set or get the number of repeat pairings of this player with $oppid.

=cut

sub Repeats ($$;$) { 
  my $this = shift;
  my $oppid = shift;
  my $newrepeats = shift;
  my $repeatsp = $this->{'repeats'};
  unless (defined $repeatsp) { $this->{'repeats'} = $repeatsp = []; }
  my $oldrepeats = $repeatsp->[$oppid];
  if (defined $newrepeats) {
    if ($oppid < 0) {
      TSH::Utility::Error("Can't set repeats for " 
	. $this->TaggedName() 
	. " vs #$oppid to $newrepeats: round number is out of range."
        );
      }
    else {
      $repeatsp->[$oppid] = $newrepeats;
      }
    }
  if ((!defined $oldrepeats) && !defined $newrepeats) {
    TSH::Utility::Error("Can't get repeats for " 
      . $this->TaggedName() 
      . " vs #$oppid: repeats are not yet computed."
      );
    $oldrepeats = 0;
    }
  return $oldrepeats;
  }

=item $boolean = ResolvePairings $unpairedp[, \%options]

Given a division and a list of unpaired players who have their
'pref' field set to a list of opponent preferences, find a reasonable
pairing of all the players.  Return success.  
Sets the 'opp' field of each player paired to the
opponent's ID.  The following options are currently used:

optimize: sort players to optimize for runtime at the cost of
pairings quality (by default, do so if more than 12 players are
involved).

target: zero-based round in which to store pairings, if not
defined, set 'opp' and return status but do not store pairings in
'pairings'.

=cut

sub ResolvePairings ($$) {
  my $unpairedp = shift;
  my $optionsp = shift;
  my $just_checking = !defined $optionsp->{'target'};
  my $optimization_threshold = (defined $optionsp->{'optimize'})
    ? $optionsp->{'optimize'} : 12;
# Debug 'RP', "--- begin run ---";

  for my $p (@$unpairedp) {
#   Debug ('RP', ("$p->{'name'} ($p->{'id'}): " . join(' ', map { $_->{'id'} } @{$p->{'pref'}})));
    }
# print "# finding optimal pairing\n";
  # pruning dead branches saves us two orders of magnitude or so
  my %dead;

  my @sorted;
  # another (slight) speed optimization
  # don't use the following line: when doing Swiss pairings, we may explore
  # pairings and then expect to extract good values from 'opp'
# if ($just_checking || @$unpairedp > $optimization_threshold) {
  if (@$unpairedp > $optimization_threshold) {
#   Debug 'RP', "above optimization threshold $optimization_threshold, sorting players";
    @sorted = @$unpairedp[sort {
      # prefer players with fewer choices
      @{$unpairedp->[$a]{'pref'}} <=> @{$unpairedp->[$b]{'pref'}} ||
      # ties broken according to input ordering
      $a <=> $b;
      } 0..$#$unpairedp
      ];
    for my $p (@sorted) { Debug 'RP', $p->Name().': '.scalar(@{$p->{'pref'}}); }
    }
  else { @sorted = @$unpairedp; }

  { # block for scope isolation only
    my(@choice, $opp, $oppid);

    # mark all players as initially unpaired
    # 'opp' points to the provisional opponent
    for my $p (@sorted) { 
      $p->{'opp'} = -1; 
      # check quickly to see if pairings are impossible
      unless (@{$p->{'pref'}}) {
#	TSH::Utility::Error "No candidate opponents for " . $p->{'name'};
	return 0;
        }
      }

    # find best opp for each player, favoring top of field
    for (my $i=0; $i<=$#sorted; ) {
      my $p = $sorted[$i];
      if ($p->{'opp'} >= 0)
        { $i++; next; } # player has already been paired - skip
      my $key = join('', grep { $_->{'opp'} < 0 } 
	@sorted[$i..$#sorted]);
      if ($dead{$key}) {
	# this code is duplicated below and should be merged 
	# when fully debugged
        for ($choice[$i]=undef; $i>=0 && !defined $choice[$i]; $i--) { }
# print "$i.\n";
        if ($i < 0) {
	  TSH::Utility::Error "Walked entire tree, couldn't find acceptable pairing.\n"
	    unless $just_checking;
          return 0;
          }

        # find last paired player's opponent, which now has to be unpaired
        my $opp = $sorted[$i]{'pref'}[$choice[$i]];
        # unpair opponent from that player
        $opp->{'opp'} = -1;
        # unpair that player from the opponent
        $sorted[$i]{'opp'} = -1;
        next;
        }

      # go to head of preference list if visiting player for first time
      $choice[$i] = -1 unless defined $choice[$i];

      # try the next preferred opp for this player.
      $opp = $p->{'pref'}[++$choice[$i]];

      if (!defined $opp) {
#       Debug 'RP', "$p->{'name'} ($p->{'id'})can't be paired, backtracking";
	$dead{$key}++;
        for ($choice[$i]=undef; $i>=0 && !defined $choice[$i]; $i--) { }
# print "$i.\n";
        if ($i < 0) {
	  TSH::Utility::Error "Walked entire tree, couldn't find acceptable pairing.\n"
	    unless $just_checking;
          return 0;
          }

        # find last paired player's opponent, which now has to be unpaired
        my $opp = $sorted[$i]{'pref'}[$choice[$i]];
        # unpair opponent from that player
        $opp->{'opp'} = -1;
        # unpair that player from the opponent
        $sorted[$i]{'opp'} = -1;
        next;
        } # if (!defined $opp) - we've run out of opps, back up

#      Debug ('RP', ("$p->{'name'} ($p->{'id'}) has pairing vector: ".join(',', map { $_->ID() } @{$p->{'pref'}})));
#      Debug 'RP', (" trying to pair $p->{'name'}, choice $choice[$i] is " . (defined $opp ? "$opp->{'name'} ($opp->{'id'})" : 'undef'));

      if ($opp->{'opp'} >= 0) {
#	Debug 'RP', " but $opp->{'name'} has already been paired.\n";
        next;
        }

      # looks good so far, let's try to keep going
      $p->{'opp'} = $opp->{'id'};
      $opp->{'opp'} = $p->{'id'};
      $i++;
      } # for $i
    }
  # copy provisional opponents to pairings
  unless ($just_checking) {
    my $r0 = $optionsp->{'target'};
    for my $i (0..$#sorted) {
      my $p = $sorted[$i];
      my $old = $p->{'pairings'}[$r0];
      if (defined $old) {
	$p->Division()->Tournament->TellUser('erpowp', $p->TaggedName(), $old,
	  $r0+1, $p->{'opp'});
        }
      $p->{'pairings'}[$r0] = $p->{'opp'};
#     push(@{$p->{'pairings'}}, $p->{'opp'});
      }
    }
  1;
  } # sub ResolvePairings

=item $n = $p->RoundCappedSpread($round0);

Return the player's cumulative spread as of 0-based round $round0,
capped per round according to config standings_spread_cap.

=cut

sub RoundCappedSpread ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rcspread'}[$round0] 
      ? $this->{'rcspread'}[$round0]
      : $this->{'cspread'};
  }

=item $n = $p->RoundLosses($round0);

Return the player's cumulative losses as of 0-based round $round0.

=cut

sub RoundLosses ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rlosses'}[$round0] 
      ? $this->{'rlosses'}[$round0]
      : ($this->{'losses'} || 0);
  }

=item $n = $p->RoundCappedRank($round0, $rank);

Set or get the player's rank in 0-based round $round0.
TODO: Should combine code shared with RoundRank.

=cut

sub RoundCappedRank ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newrank = shift;
  my $ranksp = $this->{'rcrank'};
  unless (defined $ranksp) { $this->{'rcrank'} = $ranksp = &share([]); }
  # Normally we work internally with zero-based indexing for round numbers.
  # Here we use one-based indexing and reserve the 'round 0 ranking' for
  # preevent ranking.
  my $round = $round0 + 1;
  my $oldrank = $ranksp->[$round];
# printf STDERR "r=%d new=%d old=%d\n", $round, ($newrank||-1), ($oldrank||-1);
  if (defined $newrank) {
    if ($round < 0) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'rank', $this->TaggedName(), $round, $newrank);
      }
    else {
      $ranksp->[$round] = $newrank;
      }
    }
  if ((!defined $newrank) && !defined $oldrank) {
    $this->Division()->Tournament->TellUser('enorank', 
      $this->TaggedName(), $round);
    $oldrank = 0;
    }
  return $oldrank;
  }

=item $n = $p->RoundRank($round0, $rank);

Set or get the player's rank in 0-based
round $round0.

=cut

sub RoundRank ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newrank = shift;
  my $ranksp = $this->{'rrank'};
  unless (defined $ranksp) { $this->{'rrank'} = $ranksp = &share([]); }
  # Normally we work internally with zero-based indexing for round numbers.
  # Here we use one-based indexing and reserve the 'round 0 ranking' for
  # preevent ranking.
  my $round = $round0 + 1;
  my $oldrank = $ranksp->[$round];
# printf STDERR "r=%d new=%d old=%d\n", $round, ($newrank||-1), ($oldrank||-1);
  if (defined $newrank) {
    if ($round < 0) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'rank', $this->TaggedName(), $round, $newrank);
      }
    else {
      $ranksp->[$round] = $newrank;
      }
    }
  if ((!defined $newrank) && !defined $oldrank) {
    $this->Division()->Tournament->TellUser('enorank', 
      $this->TaggedName(), $round);
    $oldrank = 0;
    }
  return $oldrank;
  }

=item $n = $p->RoundSpread($round0);

Return the player's cumulative spread as of 0-based round $round0.

=cut

sub RoundSpread ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rspread'}[$round0] 
      ? $this->{'rspread'}[$round0]
      : $this->{'spread'};
  }

=item $n = $p->RoundWins($round0);

Return the player's cumulative wins as of 0-based round $round0.

=cut

sub RoundWins ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rwins'}[$round0] 
      ? $this->{'rwins'}[$round0]
      : ($this->{'wins'} || 0);
  }

=item $n = $p->Score($round0[, $score]);

Return or set the player's score in 0-based
round $round0.

=cut

sub Score ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newscore = shift;
  # warn "Set $this->{'name'} score in round $round0+1 to $newscore\n";
  my $scoresp = $this->{'scores'};
  my $oldscore = $scoresp->[$round0];
Debug 'SCORE', 'r=%d ns=%d s=%d ds1=%d ds2=%d n=%s', $round0+1, scalar(@$scoresp), $oldscore, (defined $oldscore), (defined $scoresp->[$round0]), $this->{'id'};
  if (defined $newscore) {
    if ($round0 < 0 
      || ($config::allow_gaps 
	? $round0 >= ($this->Division->MaxRound0()||0)+1
	: $round0 > @$scoresp)) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'score', $this->TaggedName(), $round0+1, $newscore);
      }
    else {
      $scoresp->[$round0] = $newscore;
      }
    }
  return $oldscore && ($oldscore == 9999 ? undef : $oldscore);
  }

=item $n = $p->Seconds();

=item $newn = $p->Seconds($n);

Set or get the number of seconds a player has had.
Is called to set value in TSH::Division::SynchFirsts.

=cut

sub Seconds ($;$) { TSH::Utility::GetOrSet('p2', @_); }

=item SortByCappedStanding($round0, @players)

Sorts players according to their standings as of zero-based round
$round0, capped according to standings_spread_cap and returns the sorted list.

=cut

sub SortByCappedStanding ($@) {
  my $sr0 = shift;

# local($^W) = 0;
  return sort { 
#  die ("Incomplete player $a->{'name'} ($a):\n  ".join(', ',keys %$a)."\n") unless defined $a->{'wins'} && defined $a->{'spread'} && defined $a->{'rating'} && defined $a->{'rnd'};
    $sr0 >= 0 ? 
      ((defined ($b->{'rwins'}[$sr0]) ? $b->{'rwins'}[$sr0] : $b->{'wins'})<=>(defined ($a->{'rwins'}[$sr0]) ? $a->{'rwins'}[$sr0] : $a->{'wins'}) ||
      ((defined ($a->{'rlosses'}[$sr0]) ? $a->{'rlosses'}[$sr0] : $a->{'losses'})<=>(defined ($b->{'rlosses'}[$sr0]) ? $b->{'rlosses'}[$sr0] : $b->{'losses'})) ||
      ((defined ($b->{'rcspread'}[$sr0]) ? $b->{'rcspread'}[$sr0] : $b->{'cspread'})<=>(defined $a->{'rcspread'}[$sr0] ? $a->{'rcspread'}[$sr0] : $a->{'cspread'})) || 
      $b->{'rating'}<=>$a->{'rating'} ||
      $b->{'rnd'}<=>$a->{'rnd'})
    : ($b->{rating}<=>$a->{rating} || $b->{rnd} <=> $a->{rnd})
    ; } @_;
  }

=item @sorted = SortByCurrentStanding(@players)

Sorts players according to their current standings and returns the sorted list.

=cut

sub SortByCurrentStanding (@) {
  return sort {
    $b->{wins} <=> $a->{wins} ||
    $a->{losses} <=> $b->{losses} ||
    $b->{spread} <=> $a->{spread} ||
    $b->{rating} <=> $a->{rating} ||
    $b->{rnd} <=> $a->{rnd};
    } @_;
  }

=item SortByHandicap($round0, @players)

Sorts players according to their handicap points as of zero-based round
$round0 and returns the sorted list.

=cut

sub SortByHandicap ($@) {
  my $sr0 = shift;

# local($^W) = 0;
  # TODO: this crashes with invalid value for shared scalar if someone doesn't have a handicap
  return sort { 
#  die ("Incomplete player $a->{'name'} ($a):\n  ".join(', ',keys %$a)."\n") unless defined $a->{'wins'} && defined $a->{'spread'} && defined $a->{'rating'} && defined $a->{'rnd'};
#   warn $b->{'name'} . ' ' . $b->{'etc'}{'handicap'}+2*$b->{'rwins'}[$sr0];
    $sr0 >= 0 ? 
      ((defined ($b->{'rwins'}[$sr0]) ? ($b->{'etc'}{'handicap'}[0]||0)+2*$b->{'rwins'}[$sr0] : ($b->{'etc'}{'handicap'}[0]||0)+2*$b->{'wins'})<=>(defined ($a->{'rwins'}[$sr0]) ? ($a->{'etc'}{'handicap'}[0]||0)+2*$a->{'rwins'}[$sr0] : ($a->{'etc'}{'handicap'}[0]||0)+2*$a->{'wins'}) ||
      ((defined ($b->{'rspread'}[$sr0]) ? $b->{'rspread'}[$sr0] : $b->{'spread'})<=>(defined $a->{'rspread'}[$sr0] ? $a->{'rspread'}[$sr0] : $a->{'spread'})) || 
      (defined ($b->{'rwins'}[$sr0]) ? $b->{'rwins'}[$sr0] : $b->{'wins'})<=>(defined ($a->{'rwins'}[$sr0]) ? $a->{'rwins'}[$sr0] : $a->{'wins'}) ||
      ((defined ($a->{'rlosses'}[$sr0]) ? $a->{'rlosses'}[$sr0] : $a->{'losses'})<=>(defined ($b->{'rlosses'}[$sr0]) ? $b->{'rlosses'}[$sr0] : $b->{'losses'})) ||
      $b->{'rating'}<=>$a->{'rating'} ||
      $b->{'rnd'}<=>$a->{'rnd'})
    : (($b->{'etc'}{'handicap'}[0]||0)<=> ($a->{'etc'}{'handicap'}[0]||0) || $b->{rating}<=>$a->{rating} || $b->{rnd} <=> $a->{rnd})
    ; } @_;
  }

=item SortByInitialStanding(@players)

Sorts players according to their initial standings and returns the sorted list.

=cut

sub SortByInitialStanding (@) {
  return sort {
    $b->{rating} <=> $a->{rating} ||
    $b->{rnd} <=> $a->{rnd}
    } @_;
  }

=item SortByStanding($round0, @players)

Sorts players according to their standings as of zero-based round
$round0 and returns the sorted list.

=cut

sub SortByStanding ($@) {
  my $sr0 = shift;

# local($^W) = 0;
  return sort { 
#  die ("Incomplete player $a->{'name'} ($a):\n  ".join(', ',keys %$a)."\n") unless defined $a->{'wins'} && defined $a->{'spread'} && defined $a->{'rating'} && defined $a->{'rnd'};
    $sr0 >= 0 ? 
      ((defined ($b->{'rwins'}[$sr0]) ? $b->{'rwins'}[$sr0] : $b->{'wins'})<=>(defined ($a->{'rwins'}[$sr0]) ? $a->{'rwins'}[$sr0] : $a->{'wins'}) ||
      ((defined ($a->{'rlosses'}[$sr0]) ? $a->{'rlosses'}[$sr0] : $a->{'losses'})<=>(defined ($b->{'rlosses'}[$sr0]) ? $b->{'rlosses'}[$sr0] : $b->{'losses'})) ||
      ((defined ($b->{'rspread'}[$sr0]) ? $b->{'rspread'}[$sr0] : $b->{'spread'})<=>(defined $a->{'rspread'}[$sr0] ? $a->{'rspread'}[$sr0] : $a->{'spread'})) || 
      $b->{'rating'}<=>$a->{'rating'} ||
      $b->{'rnd'}<=>$a->{'rnd'})
    : ($b->{rating}<=>$a->{rating} || $b->{rnd} <=> $a->{rnd})
    ; } @_;
  }

=item SpliceInactive(@ps, $nrounds, $round0)

Removes all inactive players from C<@ps>
and assigns them byes for the next $nrounds$ rounds
beginning with round $round0
unless they already have pairings/scores for those rounds.

=cut

sub SpliceInactive (\@$$) {
  my $psp = shift;
  my $count = shift;
  my $round0 = shift;

  return if $round0 < 0;
  for (my $i=0; $i<=$#$psp; $i++) {
    my $p = $psp->[$i];
    my $off = $p->{'etc'}{'off'};
    next unless defined $off;
    TSH::Utility::SpliceSafely(@$psp, $i--, 1);
    
    my $pairingsp = $p->{'pairings'};
    next if $#$pairingsp < $round0-1;
    my $scoresp = $p->{'scores'};
    for my $j ($round0..$round0+$count-1) {
      $pairingsp->[$j] = 0 unless defined $pairingsp->[$j];
      $scoresp->[$j] = $off->[0] unless defined $scoresp->[$j];
      }
    }
  }

sub CappedSpread ($) { 
  my $this = shift;
  return $this->{'cspread'};
  }

=item $n = $p->Spread();

Return the player's cumulative spread so far this tournament.

=cut

sub Spread ($) { 
  my $this = shift;
  my $cume;
  $cume = $this->{'etc'}{'cume'}[0] if exists $this->{'etc'}{'cume'};
  unless (defined $cume) {
    $cume = $this->{'spread'} || 0;
    }
  return $cume;
  }

=item $n = $p->SupplementaryRatingsData($system_type, $subfield[, $value]);

Get or set supplementary ratings data. 
C<$system_type> should be a valid rating system name.
C<$subfield> should be a zero-based index:
0 for the primary rating value,
1 for the number of career games,
2 for the newly updated primary rating value,
3 for the most recent performance rating value,
4 for the number of additional split segments if > 0,
5 for the first mid-tournament rating,
6 for the first segment performance rating,
7 for the second mid-tournament rating,
8 for the second segment performance rating,
higher values for system-specific values.

For code clarity, C<$subfield> should be one of
the following values: old games new perf nseg mid1 perf1 mids perf2.

=cut

{
my %subfield_types = (
  'old' => 0,
  'games' => 1,
  'new' => 2,
  'perf' => 3,
  'nseg' => 4,
  'mid1' => 5,
  'perf1' => 6,
  'mid2' => 7,
  'perf2' => 8,
  );
sub SupplementaryRatingsData($$$;$) {
  my $this = shift;
  my $type = shift;
  my $subfield = shift;
  my $new = shift;
# warn "$this->{'name'} $type $subfield $new";
  unless ($subfield =~ /^\d+$/) {
    my $n = $subfield_types{$subfield};
    die "bad subfield id: $subfield" unless defined $n;
    $subfield = $n;
    }
  $type = lc $type;
  $type =~ s/\W/_/g;
  my $key = "rating_$type";
  my $ref = $this->GetOrSetEtcVector($key);
  my $old = $ref ? $ref->[$subfield] : undef;
  if (defined $new) {
    if (defined $ref) {
      $ref->[$subfield] = $new;
      }
    else {
      $ref = &share([]);
      $ref->[$subfield] = $new;
      $this->GetOrSetEtcVector($key, $ref);
      }
    $this->Division()->Dirty(1);
    }
# warn "$this ".($old||'undef');
  return $old;
  }
}

=item $n = $p->TaggedHTMLName();

As TaggedName, but <span>-labels parts of name.

=cut

sub TaggedHTMLName ($) { 
  my $this = shift;
  my $clean_name = $this->PrettyName();
  my $fullid = $this->FullID();
  my $team = '';
  my $config = $this->Division()->Tournament()->Config();
  if ($config->Value('show_teams')) {
    $team = '<span class=team>/' . $this->Team() . '</span>';
    }
  defined $this && length($clean_name)
    ? "<span class=name>$clean_name</span> <span class=lp>(</span><span class=id>$fullid</span>$team<span class=rp>)</span>"
    : 'nobody';
  }

=item $n = $p->TaggedName();

Return a formatted version of the player's name, including
their player number.  You should call the wrapper TSH::Utility::TaggedName
unless you are 100% sure that the player pointer is valid.

=cut

sub TaggedName ($) { 
  my $this = shift;
  my $clean_name = $this->PrettyName();
  my $fullid = $this->FullID();
  my $team = '';
  my $config = $this->Division()->Tournament()->Config();
  if ($config->Value('show_teams')) {
    $team = '/' . $this->Team();
    }
  defined $this && length($clean_name)
    ? "$clean_name ($fullid$team)"
    : 'nobody';
  }

=item $s = $p->Team();

Get the player's team name.

=cut

sub Team ($) { 
  my $this = shift;
  $this->{'etc'}{'team'} ? join(' ', @{$this->{'etc'}{'team'}}) : '';
  }

=item $t = $p->Time([$time]);

Get or set the player's update time.

=cut

sub Time ($;$) { 
  my $this = shift;
  my $new = shift;
  return $this->GetOrSetEtcScalar('time', $new);
  }

=item $changed = $p->Truncate($r0);

Remove all player data after round $r0.

=cut

sub Truncate ($$) {
  my $this = shift;
  my $r0 = shift;
  my $changed = 0;
  for my $key (qw(board p12 penalty)) {
    next if exists $this->{'etc'}{$key};
    $this->{'etc'}{$key} = &share([]);
    }
  for my $ip (
    $this->{'scores'},
    $this->{'pairings'},
    $this->{'etc'}{'board'},
    $this->{'etc'}{'penalty'},
    $this->{'etc'}{'p12'},
    ) {
    if ($#$ip > $r0) { 
#     $#$ip = $r0; # splices are not thread-safe
      my @truncated : shared = @$ip[0..$r0];
      $ip = \@truncated;
      $changed = 1;
      }
    }
  if ($r0 < 0) {
    delete $this->{'etc'}{'bracketseed'};
    }
  return $changed;
  }

=item $success = $p->UnpairRound($r0);

Remove pairings for player in round $r0, return success.

=cut

sub UnpairRound ($$) { 
  my $this = shift;
  my $r0 = shift;
  my $tourney = $this->{'division'}->Tournament();

  my $opp;
  if ($config::allow_gaps) {
    if (defined $this->{'scores'}[$r0]) {
      $tourney->TellUser('euprhas1s', $this->{'id'}, $r0+1);
      return 0;
      }
    $this->{'etc'}{'board'}[$r0] = 0 if $this->{'etc'}{'board'}[$r0];
    $this->{'etc'}{'p12'}[$r0] = 4 if $config::track_firsts;
    $this->{'scores'}[$r0] = undef;
    $opp = $this->{'pairings'}[$r0];
    $this->{'pairings'}[$r0] = undef;
    }
  else {
    my $pairingsp = $this->{'pairings'};
    if ($#$pairingsp != $r0) {
      $tourney->TellUser('iupr1bad', $this->{'name'}, $r0+1);
      return 0;
      }
    $opp = pop @$pairingsp;
    if ($#{$this->{'etc'}{'board'}} >= $r0) {
#     $#{$this->{'etc'}{'board'}} = $r0 - 1; # splices are not thread-safe
      my @board : shared = @{$this->{'etc'}{'board'}}[0..$r0-1];
      $this->{'etc'}{'board'} = \@board;
      }
    if ($#{$this->{'etc'}{'p12'}} >= $r0) {
#     $#{$this->{'etc'}{'p12'}} = $r0 - 1; # splices are not thread-safe
      my @p12 : shared = @{$this->{'etc'}{'p12'}}[0..$r0-1];
      $this->{'etc'}{'p12'} = \@p12;
      }
    return 0 if $#{$this->{'pairings'}} != $r0 - 1;
    if ($#{$this->{'scores'}} >= $r0) {
#     $#{$this->{'scores'}} = $r0 - 1; # splices are not thread-safe
      my @scores : shared = @{$this->{'scores'}}[0..$r0-1];
      $this->{'scores'} = \@scores;
      }
    }
  $tourney->TellUser('iupr1ok', $this->{'id'}, $opp) if defined $opp;
  return 1;
  }

=item $n = $p->UnscoredGames();

Return the number of games that the player has not yet recorded a score in.

=cut

sub UnscoredGames ($) { 
  my $this = shift;
  return $this->{'noscores'};
  }

=item $n = $p->Wins();

Return the player's total wins so far this tournament.

=cut

sub Wins ($) { 
  my $this = shift;
  return ($this->{'wins'} || 0);
  }

=back

=cut

=head1 BUGS

Rather than calling C<Division::Synch()> when the C<.t> file is
loaded (which substantially delays the loading of large files),
the relevant statistics that it computes should be computed only
as needed.

Not all routines call TSH::Division::Dirty(1) when needed yet.

Team() should look more like Board().

First() should check consistency with opponent.

ResolvePairings is currently the critical sub, and should be optimized,
or rewritten in C.

Should replace wins with rwins[-1], passim losses and spread.

=cut

1;

