#!/usr/bin/perl

# Copyright (C) 2009 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::EXPORTRATINGS;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::EXPORTRATINGS - implement the C<tsh> EXPORTRATINGS command

=head1 SYNOPSIS

  my $command = new TSH::Command::EXPORTRATINGS;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::EXPORTRATINGS is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to export a new rating list including the results of this event.
EOF
  $this->{'names'} = [qw(exportratings)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $processor = $this->Processor();
  my $config = $tournament->Config();
  my (@divisions) = $tournament->Divisions();

  my $ratings_list = $config->Value('rating_list') || 'nsa';
  $tournament->LoadRatings() or return 0;
  $this->Processor()->Flush();
  my %p;
  for my $dp (@divisions) { 
    $dp->ComputeRatings($config->Value('max_rounds')-1);
    for my $p ($dp->Players()) { 
      my $key = uc $p->Name();
      $key =~ s/,//;
      $p{$key} = $p;
      }
    }
  my $ofh;
  my $ofn = $config->MakeLibPath("ratings/$ratings_list/current.txt");
  my $nfh;
  my $nfn = $config->MakeLibPath("ratings/$ratings_list/new.txt");
  if (!open $ofh, '<', $ofn) {
    $tournament->TellUser('euseropen', $nfn, $!);
    return 0;
    }
  if (!open $nfh, '>', $nfn) {
    $tournament->TellUser('euseropen', $nfn, $!);
    return 0;
    }
  binmode $nfh, ':encoding(isolatin1)';
  binmode $ofh, ':encoding(isolatin1)';
  while (<$ofh>) {
    chomp;
    my ($name, $rating, $rank, $expiry, $games) = split(/\t/, $_, 5);
    $name =~ s/,//;
    if (my $p = $p{uc $name}) {
      $rating = $p->NewRating(-1);
      $games += $p->GamesPlayed();
      print $nfh join("\t", $name, $rating, $rank, $expiry, $games), "\n";
      delete $p{uc $name};
      }
    else {
      print $nfh "$_\n";
      }
    }
  for my $key (sort keys %p) {
    my $p = $p{$key};
    warn $key;
    print $nfh join("\t", $p->Name(), $p->NewRating(-1), 0, 0, $p->GamesPlayed()), "\n";
    }
  close $nfh;
  close $ofh;
  $tournament->TellUser('iexratok', $nfn);
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

