#!/usr/bin/perl

# Copyright (C) 2009 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::EnhancedScoreBoard;

use strict;
use warnings;

use TSH::Log;
use File::Copy;
# use TSH::Utility qw(Debug DebugOn FormatHTMLHalfInteger FormatHTMLSignedInteger Ordinal);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::EnhancedScoreBoard - implement the C<tsh> EnhancedScoreBoard command

=head1 SYNOPSIS

  my $command = new TSH::Command::EnhancedScoreBoard;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::EnhancedScoreBoard is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to add an enhanced scoreboard to your web coverage.
Enhanced scoreboards use AJAX to dynamically update information in
the user browser whenever you trigger the saveJSON command in tsh.
EOF
  $this->{'names'} = [qw(esb enhancedscoreboard)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args);

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;
  my $dname = uc $dp->Name();
  my $config = $tournament->Config();

  $this->Processor()->Process("savejson");
  my $sourcefn = $config->MakeLibPath('js/esb.js');
  my $destfn = $config->MakeHTMLPath("esb.js");
  copy($sourcefn, $destfn) or die $!;

  my $logp = new TSH::Log($tournament, $dp, 
    'enhanced-scoreboard', undef,
    {
      'body_class' => 'enhanced scoreboard',
      'noconsole' => 1,
      'notext' => 1,
      'notitle' => 1,
      'javascript' => 'esb.js',
    });

  $logp->Write('', <<"EOF");
</table>
<div id='sb'></div>
<script type="text/javascript"><!--
  var url = window.location.href.replace(/[^\//]+\$/, 'tourney.js');
  KeepLoadingScoreBoard({
    'id' : 'sb',
    'url' : url,
    'dname' : '$dname',
    'columns' : 5,
    'rows' : 4,
    'offset' : 0
    });
--></script>
<table>
EOF
  $logp->Close();
  return 0;
  }

=back

=cut

=head2 BUGS

None known.

=cut

1;
