#!/usr/bin/perl

# Copyright (C) 2007-2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundRATings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn FormatHTMLHalfInteger);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundRATings - implement the C<tsh> RoundRATings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundRATings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundRATings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($@);
sub new ($);
sub RenderPlayer ($$$);
sub RenderTable ($$$$);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($@) {
  my $this = shift;

  $this->SUPER::initialise(@_);
  $this->{'help'} = <<'EOF';
Use this command to display ratings estimates based on standings
in a given round or rounds within a division.  If you specify
just one round, the results will be displayed on the screen.
If you specify a range of rounds (e.g., 4-7), the results will
not be shown on your screen but report files will be updated.
EOF
  $this->{'names'} = [qw(rrat roundratings)];
  $this->{'argtypes'} = [qw(RoundRange Division)];

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item RenderPlayer($logp, $pp, \%myconfig);

Render a row of a ratings table.

=cut

sub RenderPlayer ($$$) {
  my $logp = shift;
  my $p = shift;
  my $myconfigp = shift;

  my $r0 = $myconfigp->{'r0'};
  my $config = $myconfigp->{'config'};

#   if ($config::pair_page_break && $i && ($i) % $config::pair_page_break == 0) { $logp->Write('',qq(</table><table class=ratings align=center cellspacing=0 style="page-break-before:always"><tr>$headings</tr>)); }
  my $rating = $p->Rating();
  my $newr = $r0 >= 0 ? $p->NewRating($r0) : $rating;
  my $rank = $myconfigp->{'is_capped'} 
    ? $p->RoundCappedRank($r0) : $p->RoundRank($r0);
  my @fields;
  my @text_fields;

  push(@fields, $rank);
  push(@text_fields, $rank);
  {
    my $w = $p->RoundWins($r0);
    my $l = $p->RoundLosses($r0);
    push(@text_fields, sprintf("%.1f-%.1f", $w, $l)),
    push(@fields,
      FormatHTMLHalfInteger($w) . '&ndash;' . FormatHTMLHalfInteger($l));
  }
  {
    my $spread = sprintf("%+d", $myconfigp->{'is_capped'}
      ? $p->RoundCappedSpread($r0) : $p->RoundSpread($r0));
    push(@fields, $spread);
    push(@text_fields, $spread);
  }
  if ($myconfigp->{'thaipoints'}) {
    my $points = 2*$p->RoundWins($r0)+($p->GetOrSetEtcScalar('handicap')||0);
    push(@fields, $points);
    push(@text_fields, $points);
    }
  if ($myconfigp->{'oldrh'}) {
    push(@fields, $rating);
    push(@text_fields, $rating);
    }
  if ($myconfigp->{'newrh'}) {
    push(@fields, $newr);
    push(@text_fields, $newr);
    }
  if ($myconfigp->{'difrh'}) {
    my $diff = $rating ? sprintf("%+d", $newr-$rating) : '';
    push(@fields, $diff);
    push(@text_fields, $diff);
    }
  if ($myconfigp->{'has_classes'}) {
    push(@fields, $p->Class());
    push(@text_fields, $p->Class());
    }
  push(@fields, $p->TaggedName('print'));
  push(@text_fields, $p->TaggedName());
  unless ($myconfigp->{'noshowlast'}) {
    my $s = '';
    my $opp = $p->Opponent($r0);
    if ($opp) {
      my $oname = $myconfigp->{'showlastplayername'}
        ? $opp->TaggedName() : $opp->FullID();
      my $ms = $p->Score($r0);
      if (defined $ms) {
	my $os = $p->OpponentScore($r0);
	$os = 0 unless defined $os; # possible if pairing data bad
	my $p12 = $p->First($r0);
	$s = 
	  ($myconfigp->{'trackfirsts'} ? substr('B12??', $p12,1) : '')
	  .($myconfigp->{'spreadentry'} 
	    ? '' : ($ms > $os ? 'W' : $ms < $os ? 'L' : 'T'));
	$s .= ':' if $s;
	$s .=($myconfigp->{'spreadentry'} 
	  ? sprintf("%+d", $ms-$os) : "$ms-$os")
	  . ":$oname";
	}
      else {
	$s = $config->Terminology('pending') . ":$oname";
	}
      }
    else {
      if (defined $p->OpponentID($r0)) {
	$s = $config->Terminology('bye');
	}
      else {
	$s = $config->Terminology('unpaired');
	}
      }
    push(@fields, $s);
    push(@text_fields, $s);
    }
  # next round
  if (my $opp = $p->Opponent($r0+1)) {
    push(@fields, $opp ? $opp->FullID() : $config->Terminology('bye'));
    }
  $logp->WriteRow(\@text_fields, \@fields);
  }

=item RenderTable($dp, $r0, $optionsp, $noconsole)

Render the rows of the ratings table. This code is independent of
rating system.

=cut

sub RenderTable ($$$$) {
  my $dp = shift;
  my $r0 = shift;
  my $optionsp = shift;
  my $noconsole = shift;
  my $tournament = $dp->Tournament();
  my $config = $tournament->Config();
  my (%myconfig) = (%$optionsp);
  $myconfig{'config'} = $config;
  $myconfig{'has_classes'} = $dp->Classes() || defined $dp->Player(1)->Class();
  $myconfig{'is_capped'} = $config->Value('standings_spread_cap');
  $myconfig{'noshowlast'} = $config->Value('no_show_last') || $r0 < 0;
  $myconfig{'r0'} = $r0;
  $myconfig{'showlastplayername'} = $config->Value('show_last_player_name');
  $myconfig{'spreadentry'} = $config->Value('entry') eq 'spread';
  $myconfig{'thaipoints'} = $config->Value('thai_points');
  $myconfig{'trackfirsts'} = $config->Value('track_firsts');
  $optionsp->{'order'} ||= 'rating';
  if ($optionsp->{'no_ratings'}) {
    $optionsp->{'oldrh'} = $optionsp->{'newrh'} = $optionsp->{'difrh'} = undef;
    }

  my $logp = new TSH::Log($tournament, $dp, 
    ($optionsp->{'filename'} || 'ratings'),
    $r0+1,
    {
      'noconsole' => $noconsole,
      %$optionsp,
    },
    );
  my (@classes) = qw(rank wl spread);
  push(@classes, 'thai') if $myconfig{'thaipoints'};
  my @html_titles;
  my @text_titles;
  for my $code (qw(Rank Won_Lost Spread)) {
    push(@html_titles, $config->Terminology($code));
    }
  for my $code (qw(Rnk W_L Sprd)) {
    push(@text_titles, $config->Terminology($code));
    }
  if ($myconfig{'thaipoints'}) {
    push(@html_titles, $config->Terminology('thaipoints'));
    push(@text_titles, $config->Terminology('thaipts'));
    }
  for my $key (qw(oldr newr difr)) {
    if ($optionsp->{"${key}h"}) {
      push(@html_titles, $config->Terminology($optionsp->{"${key}h"}));
      push(@text_titles, $config->Terminology($optionsp->{"${key}t"}));
      push(@classes, "rating $key");
      }
    };
  if ($myconfig{'has_classes'}) {
    push(@classes, 'pclass');
    push(@html_titles, $config->Terminology('Class'));
    push(@text_titles, $config->Terminology('Cl'));
    }
  push(@classes, 'name');
  push(@html_titles, $config->Terminology('Player'));
  push(@text_titles, $config->Terminology('Player'));
  unless ($myconfig{'noshowlast'}) {
    push(@classes, qw(last));
    push(@html_titles, $config->Terminology('Last_Game'));
    push(@text_titles, $config->Terminology('Last_G'));
    }
  if ($dp->LastPairedRound0() > $r0) {
    push(@classes, 'next');
    push(@html_titles, $config->Terminology('Next_Game'));
#   push(@text_titles, 'Next Opponent'); # too wide
    }
  $logp->ColumnClasses(\@classes);
  $logp->ColumnTitles(
    {
    'text' => \@text_titles,
    'html' => \@html_titles,
    }
    );
  my @ps;
  if ($myconfig{'is_capped'}) {
    $dp->ComputeCappedRanks($r0);
    if ($optionsp->{'order'} eq 'handicap') {
      @ps = TSH::Player::SortByHandicap($r0, $dp->Players());
      }
    else {
      @ps = TSH::Player::SortByCappedStanding($r0, $dp->Players());
      }
    }
  else {
    $dp->ComputeRanks($r0);
    if ($optionsp->{'order'} eq 'handicap') {
      @ps = TSH::Player::SortByHandicap($r0, $dp->Players());
      }
    else {
      @ps = TSH::Player::SortByStanding($r0, $dp->Players());
      }
    }
  for my $p (@ps) {
    next unless $p->Active();
    RenderPlayer $logp, $p, \%myconfig;
    }
  $logp->Close();
  }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($firstr1, $lastr1, $dp) = @_;
  my $firstr0 = $firstr1 - 1;
  my $lastr0 = $lastr1 - 1;
  my $rating_system = $dp->RatingSystem();

  $dp->CountPlayers() or return 0;
  $dp->CheckRoundHasResults($lastr0) or return 0;

  my (%options_by_rating_system) = (
    'nsa' => {
      'titlename' => 'Ratings',
      'oldrh'=>'Old_Rating',
      'oldrt'=>'Old_R',
      'newrh'=>'New_Rating',
      'newrt'=>'New_R',
      'difrh'=>'Rating_Change',
      'difrt'=>'Rating_Ch',
      },
    'none' => {
      'titlename' => 'Rankings',
      },
    );
  
  my $optionsp = $options_by_rating_system{lc $rating_system}
    || $options_by_rating_system{'nsa'};
  for my $r1 ($firstr1..$lastr1-1) {
    my $r0 = $r1 - 1;
    $tournament->TellUser('irratok', $r1);
#   warn $r0;
    $dp->ComputeRatings($r0);
    RenderTable $dp, $r0, $optionsp, 1;
    }
  $dp->ComputeRatings($lastr0, $this->{'noconsole'}) if $lastr0 >= 0;

  RenderTable $dp, $lastr0, $optionsp, $this->{'noconsole'};
  return 0;
  }

=back

=cut

1;
