#!/usr/bin/perl

# Copyright (C) 2009 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::THAICASH;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(FormatHTMLHalfInteger);

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::THAICASH - implement the C<tsh> THAICASH command

=head1 SYNOPSIS

  my $command = new TSH::Command::THAICASH;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::THAICASH is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list winner's bonus cash payments at a Thai tournament.
EOF
  $this->{'names'} = [qw(thaicash)];
  $this->{'argtypes'} = [qw(Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $r1 = shift;
  my $dp = shift;
  my $config = $tournament->Config();
  my $r0 = $r1 - 1;

  my $has_classes = $dp->Classes() || defined $dp->Player(1)->Class();

  for my $key (qw(thai_cash_cutoff thai_cash_payout)) {
    next if defined $config->Value($key);
    warn "Missing required configuration key: $key\n";
    return 0;
    }

  my $logp = new TSH::Log( $tournament, $dp, 'thaicash', undef,
    {'titlename' => 'Winner\'s Bonuses'});
  my @classes;
  my @html_titles;
  my @text_titles;
  @classes = qw(rank wl spread name cash rounds);
  for my $code (qw(Rank Won_Lost Spread Player Cash Rounds)) {
    push(@html_titles, $config->Terminology($code));
    }
  for my $code (qw(Rnk W_L Sprd Player Cash Rounds)) {
    push(@text_titles, $config->Terminology($code));
    }
  $logp->ColumnClasses(\@classes);
  $logp->ColumnTitles(
    {
    'text' => \@text_titles,
    'html' => \@html_titles,
    }
    );
  $dp->ComputeRanks($r0);
  my @ps = TSH::Player::SortByStanding($r0, $dp->Players());

  my $total = 0;
  my $payouts = $config->Value('thai_cash_payout');
  for my $p (@ps) {
    my $rounds = '';
    my (@results);
    my $cash = 0;
    my $maxr0 = $p->CountScores() - 1;
    $maxr0 = $r0 if $maxr0 > $r0;
    for my $ar0 (0..$maxr0) {
      next unless $p->Opponent($ar0);
      next unless $p->Score($ar0) > $p->OpponentScore($ar0);
      $results[$ar0+1] = 1;
      }
    for my $payout (@$payouts) {
      my ($first1, $last1, $pergame, $sweep) = @$payout;
      my $swept = 1;
      my $nwins = 0;
      my (@wins);
      for my $r1 ($first1..$last1) {
	if ($results[$r1]) {
	  $nwins++;
	  push(@wins, $r1);
	  }
	else {
	  $swept = 0;
	  }
        }
      if ($swept) {
	$cash += $sweep;
	$rounds .= "$first1-$last1 ";
	}
      elsif ($nwins) {
	$cash += $nwins * $pergame;
	$rounds .= join(' ', @wins, '');
        }
      }

    my (@fields, @text_fields);
    {
      my $rank = $p->RoundRank($r0);
      if ($rank <= $config->Value('thai_cash_cutoff')) {
	$cash = 0;
	}
      push(@fields, $rank);
      push(@text_fields, $rank);
    }
    $total += $cash;
    {
      my $w = $p->RoundWins($r0);
      my $l = $p->RoundLosses($r0);
      push(@text_fields, sprintf("%.1f-%.1f", $w, $l)),
      push(@fields,
	FormatHTMLHalfInteger($w) . '&ndash;' . FormatHTMLHalfInteger($l));
    }
    {
      my $spread = sprintf("%+d", $p->RoundSpread($r0));
      push(@fields, $spread);
      push(@text_fields, $spread);
    }
    {
      my $s = $p->TaggedName();
      push(@fields, $s);
      push(@text_fields, $s);
    }
    push(@fields, $cash, $rounds);
    push(@text_fields, $cash, $rounds);
    $logp->WriteRow(\@text_fields, \@fields);
    }
  $logp->Close();
  print "Total is $total.\n";
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;
