#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::TOTalScore;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::TOTalScore - implement the C<tsh> TOTalScore command

=head1 SYNOPSIS

  my $command = new TSH::Command::TOTalScore;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::TOTalScore is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list the players whose total scores are the
highest in a division.
If you specify an integer, it indicates the number of scores you
would like to see.  If not, 20 scores are shown.
EOF
  $this->{'names'} = [qw(tots totalscore)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $nscores = @_ ? shift @_ : 20;

  my $logp = new TSH::Log( $tournament, $dp, 'totalscore', undef,
    {'title' => 'Total Scores'});
  $logp->ColumnClasses([ qw(rank score name) ]);
  $logp->ColumnTitles(
    {
    'text' => [ qw(Rnk Total Player) ],
    'html' => [ qw(Rank Total<br>Score Player) ],
    }
    );
  my (@ps) = $dp->Players();
  for my $pp (@ps) {
    $pp->{'xsum'} = 0;
    for my $r0 (0..$pp->CountScores()-1) {
#     warn "$pp->{'name'} $r0" unless defined $pp->Score($r0);
      $pp->{'xsum'} += $pp->Score($r0);
      }
    }
  my (@sorted) = (sort {
    $b->{'xsum'} <=> $a->{'xsum'} ||
    $a->{'name'} cmp $b->{'name'}
    } @ps);
  my $lastrank = -1;
  my $lastsum = -1;
  for my $i (0..$#sorted) {
    my $pp = $sorted[$i];
    if ($lastsum != $pp->{'xsum'}) {
      $lastsum = $pp->{'xsum'};
      $lastrank = $i+1;
      }
    last if $lastrank > $nscores;
    $logp->WriteRow([
      $lastrank,
      $pp->{'xsum'},
      $pp->TaggedName(),
      ]);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

