#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::TUFFluck;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;
use TSH::Player::TuffLuck;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::TUFFluck - implement the C<tsh> TUFFluck command

=head1 SYNOPSIS

  my $command = new TSH::Command::TUFFluck;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::TUFFluck is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings for a "Tuff Luck" prize, 
typically awarded to a player who loses six games by the smallest
combined spread.  The optional second argument, if given, specifies
a value other than "six".
EOF
  $this->{'names'} = [qw(tuff tuffluck)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $nscores = @_ ? shift @_ : 6;

  my $logp = new TSH::Log( $tournament, $dp, 'tuffluck', undef,
    {'title' => 'Division ' . $dp->Name() . ' Tuff Luck'});
  $logp->ColumnClasses([ qw(round name scores) ]);
  $logp->ColumnTitles(
    {
    'text' => [ qw(Sum Player Losses) ],
    'html' => [ 'Sum', 'Player', 'Losing Spreads'],
    }
    );
  my @data;
  for my $pp ($dp->Players()) {
    my ($tuffness, $lossesp) = $pp->TuffLuck($nscores);
    push(@data, [$tuffness, $pp->TaggedName(), $lossesp])
      if $tuffness;
    }
  for my $data (sort {
    $a->[0] <=> $b->[0] || do {
      for (my $i=0; $i<$nscores; $i++) {
	my $cmp = $a->[2][$i] <=> $b->[2][$i];
	return $cmp if $cmp;
	}
      0;
      }
    } @data) {
    $data->[2] = join(' ', @{$data->[2]});
    $logp->WriteRow($data, $data);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

