
# Copyright (C) 2007-2008 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Config;

use strict;
use warnings;

# $::SIG{__WARN__} = sub { eval 'use Carp'; &confess($_[0]); };
# $::SIG{__DIE__} = sub { eval 'use Carp'; &confess($_[0]); };

use Encode;
use FileHandle;
use File::Copy;
use File::Path;
use File::Spec;
use File::Temp qw(tempfile);
use TSH::Division;
use TSH::Utility;
use UNIVERSAL qw(isa);
use JavaScript::Serializable;
use TFile; 
use UserMessage;
use threads::shared;

our (@ISA);
@ISA = qw(JavaScript::Serializable);
sub EXPORT_JAVASCRIPT () { return map { $_ => $_ } qw(event_date event_name max_rounds no_boards player_photos tables photo_database prize_bands scoreboard_teams); }

=pod

=head1 NAME

TSH::Config - Manipulate tsh's configuration settings

=head1 SYNOPSIS

  ## newstyle usage

  my $config = new TSH::Config($tournament, $config_filename) 
    or die "no such file";
  my $config = new TSH::Config($tournament); # use default configuration file
  # in Tournament.pm
  $config->Read(); # reads from designated file
  $config->Setup(); # checks that directories exist, etc.
  # deprecated access methods
  $config->Export(); # exports into 'config' namespace
  $value = $config::key;
  $config::key = $value;
  # preferred access methods
  $old_value = $config->Value($key);
  $config->Value($key, $value);
  $config->Write(); # updates configuration file

  ## oldstyle usage (remove documentation after testing newstyle)

  my $config = new TSH::Config($config_filename) or die "no such file";
  my $config = new TSH::Config(); # select default configuration file
  $config->Lock() or die; # prevents other users from changing tournament data
  $config->Load($tournament); # populates package 'config' with loaded values
  $config->Unlock(); # releases Lock()
  $config::key = 'new value';
  $config->Save(); # updates configuration file from package 'config'

  ## noninteractive support
  
  @option_names = TSH::Config::UserOptions();
  $help = TSH::Config::UserOptionHelp($option_name);
  $type = TSH::Config::UserOptionType($option_name);
  $time = $config->LastModified(); 
  $js = $config->ToJavaScript();

=head1 ABSTRACT

This class manages tsh's configuration settings.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($;$$);
sub new ($;$);
sub Append ($@);
sub AssignClasses ($);
sub AssignPasswords ($);
sub AssignTables ($);
sub AttachedContent ($$);
sub CheckDirectory ($$);
sub ChooseFile ($$);
sub ComputedValues ($);
sub Export ($);
sub GetPlayerByPassword ($$);
sub GetToken ($);
sub InstallPhoto ($$);
sub LastModified ($);
sub LoadPhotoIndex ($);
sub LastPrizeRank ($$);
sub MakeBackupPath ($$);
sub MakeHTMLPath ($$);
sub MakeRootPath ($$;$);
sub Normalise ($);
sub PhotoPath ($$);
sub Read ($);
sub RealmDefaults ($);
sub Render ($;$);
sub RootDirectory ($;$);
sub Save ($);
sub SetPassword ($$$);
sub Setup ($);
sub SetupTerminology ($);
sub SetupWindows ($);
sub Terminology ($$@);
sub UninstallPhoto ($$);
sub UserOptions ();
sub UserOptionHelp ($);
sub UserOptionType ($);
sub ValidateLocalDirectory ($$$);
sub Value ($$@);
sub Write ($$;$);

# information about individual configuration options
my (%user_option_data) = (
  'allow_gaps' => { 'type' => 'boolean', 'help' => 'If checked, unpaired gaps may appear in a player\'s schedule.  Should almost never be checked.' },
  'alpha_pair_page_break' => { 'type' => 'integer', 'help' => 'Gives the maximum number of rows that will be printed on one page of alpha pairings before a column or page break.' },
  'assign_firsts' => { 'type' => 'boolean', 'help' => 'If checked, tsh decides who plays first (starts) or second (replies) in each game. Should be checked in the U.K. and unchecked in Canada and the United States.' },
  'autopair' => { 'type' => 'internal', 'help' => 'Autopair instructions specify which round\'s results trigger which round\'s pairings and how.' },
  'backup_directory' => { 'type' => 'string', 'help' => 'Specifies where journalled ".t" and ".tsh" files are kept.', 'validate' => \&ValidateLocalDirectory, },
  'board_stability' => { 'type' => 'boolean', 'help' => 'If checked, tsh will try to keep one player from each game at the same board during a session. If not checked, tsh will try to assign players to boards according to their current rank.' },
  'bye_spread' => { 'type' => 'integer', 'help' => 'Specifies the spread assigned to a player who is automatically assigned a bye. Usually not adjusted directly, but by changing the value of \'realm\'.' },
  'director_name' => { 'type' => 'string', 'help' => 'Gives the name of the director of this event, for use in ratings submission.' },
  'event_date' => { 'type' => 'string', 'help' => 'Gives the date(s) of this event, for use in report headers and ratings submission.' },
  'event_name' => { 'type' => 'string', 'help' => 'Gives the name of this event, for use in report headers and ratings submission.' },
  'html_directory' => { 'type' => 'string', 'help' => 'Specifies where generated web files are kept.', 'validate' => \&ValidateLocalDirectory, },
  'max_rounds' => { 'type' => 'integer', 'help' => 'Gives the number of rounds in this tournament. This parameter is mandatory when using tsh in server mode.' },
  'no_text_files' => { 'type' => 'boolean', 'help' => 'If checked, tsh creates only web HTML files. Leave this checked unless you prefer the retro look.' },
  'port' => { 'type' => 'integer', 'help' => 'If set to a nonzero value, enables the web interface that you are using and specifies its TCP/IP port number.  Do not change unless you know what this is.' },
  'track_firsts' => { 'type' => 'boolean', 'help' => 'If checked, keeps track of who played first (started) or second (replied) in each game. Should be checked in most parts of the world now.' },
  );

=item $config->initialise($tournament, $argv)

Initialise the config object.
$tournament should be of type TSH::Tournament.
$argv should contain whatever the user specified on the command line,
typically undef or the name of an event directory.

=cut

sub initialise ($;$$) {
  my $this = shift;
  my $tournament = shift;
  my $argv = shift;
  my $readonly = shift;
  my $search = shift;
  # default values (don't include realm-specific)
  $this->{'attachments'} = &share({});
  $this->{'attachments_lc'} = &share({});
  $this->{'autopair'} = &share({});
  $this->{'backup_directory'} = File::Spec->catdir(File::Spec->curdir(),'old');
  eval { $this->{'director_name'} = (getpwuid($<))[6] };
  $this->{'division_label'} = &share({});
  $this->{'division_rating_system'} = &share({});
  {
    my (@now) = localtime;
    $this->{'event_date'} = sprintf("%04d-%02d-%02d", $now[5]+1900, $now[4]+1, $now[3]);
  }
  $this->{'event_name'} = 'Unnamed Event'; 
  $this->{'external_path'} = &share([]);
  $this->{'external_path'}[0] = q(./bin);
  $this->{'filename'} = undef;
  $this->{'flight_cap'} = 'TSH::PairingCommand::FlightCapDefault';
  $this->{'gibson_equivalent'} = &share({});
  $this->{'gibson_groups'} = &share({});
  $this->{'hook_division_complete'} = &share({});
  $this->{'html_directory'} = File::Spec->catdir(File::Spec->curdir(),'html');
  $this->{'max_div_rounds'} = &share({});
  $this->{'max_name_length'} = 22;
  $this->{'mirror_directories'} = &share([]);
  $this->{'name_format'} = '%-22s';
  $this->{'no_boards'} = undef;
  $this->{'pairing_system'} = 'chew'; # auto bracket chew manual nast basd guelph green
  $this->{'passwords'} = &share({});
  $this->{'pix'} = &share({});
  $this->{'prize_bands'} = &share({});
  $this->{'prizes'} = &share([]);
  $this->{'random'} = undef; # fixed random value for this run
  $this->{'realm'} = 'nsa';
  $this->{'reserved'} = &share({});
  $this->{'show_last_player_name'} = undef;
  $this->{'spitfile'} = &share({});
  $this->{'split1'} = 1000000;
  $this->{'table_format'} = '%3s';
  $this->{'tables'} = &share({});
  $this->{'terminology'} = undef;
  $this->{'tournament'} = $tournament;
  $this->{'_readonly'} = $readonly;
  $this->{'_saved'} = &share([]);
  $this->{'_termdict'} = undef;

  if ($search || defined $argv) {
    return undef unless $this->ChooseFile($argv);
    }
  elsif (!defined $argv) {
    return $this;
    }
  my $time = time; # can't use utime undef, undef without AIX warnings
  utime $time, $time, $this->MakeRootPath($this->{'filename'});
  return $this;
  }

=item $d = new TSH::Config($tourney, $filename);

Create a new TSH::Config object.  If the optional parameter 
C<$filename> is omitted, a default value is chosen.

=cut

sub new ($;$) { return TSH::Utility::newshared(@_); }

sub Append ($@) {
  my $this = shift;
  my (@lines) = @_;
  for my $line (@lines) {
    $line =~ s/[^\n]$/$&\n/;
    }
  push(@{$this->{'_saved'}}, @lines);
  }

=item $config->AssignClasses();

Assign players to classes.

=cut

sub AssignClasses ($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  # assign players to classes

  for my $dp ($tournament->Divisions()) {
    if (my $classes = $dp->Classes()) {
      my (@psp) = $dp->Players();
      TSH::Player::SpliceInactive @psp, 0, 0;
      @psp = TSH::Player::SortByInitialStanding @psp;
      my $phase = 0;
      my $last_rating;
      my $last_class;
      my $current_class = sprintf("%c", ord('A') + $classes - 1);
      for (my $i = $#psp; $i >= 0; $i--) {
	my $pp = $psp[$i];
	my $class = $current_class;
	$phase += $classes;
	my $current_rating = $pp->Rating();
	if ((defined $last_rating) && $last_rating == $current_rating) {
	  $class = $last_class;
#	  warn "1 $current_rating $last_rating $current_class $last_class\n";
	  }
	else {
	  $last_rating = $current_rating;
	  $last_class = $current_class;
#	  warn "2 $current_rating $last_rating $current_class $last_class\n";
	  }
	$pp->Class($class);
	if ($phase >= @psp) {
	  $phase -= @psp;
	  $current_class = sprintf("%c", ord($current_class)-1);
	  }
        }
      }
    }
  }

=item $config->AssignPasswords();

Assign data entry passwords to any players who do not yet have them.

=cut

sub AssignPasswords ($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  my %passwords : shared;
  my $bag = 'AAAAAAAAABBCCDDDDEEEEEEEEEEEEFFGGGHHIIIIIIIIIJKLLLLMMNNNNNNOOOOOOOOPPQRRRRRRSSSSTTTTTTUUUUVVWWXYYZ';
  my @missing;
  # check already assigned passwords
  for my $dp ($tournament->Divisions()) {
    my $dname = $dp->Name();
    for my $p ($dp->Players()) {
      my $password = uc $p->Password();
      if ((!defined $password) || $password eq '') { # no password yet
	push(@missing, $p);
        }
      elsif (exists $passwords{$password}) { # has someone else's password
	my ($adname, $pid) = @{$passwords{$password}};
	$tournament->TellUser('eduppass', $tournament->GetDivisionByName($adname)->Player($pid)->TaggedName(), $p->TaggedName(), $p->TaggedName());
        }
      else { # has unique password already
	my @data : shared;
	@data = ($dname, $p->ID());
	$passwords{$password} = \@data;
        }
      }
    }
  $this->{'passwords'} = \%passwords;

  # assign needed passwords
  for my $p (@missing) {
    my $password;
    for (my $tries = 0; $tries<10; $tries++) {
      my $apassword = join('', sort map { substr($bag, rand(length($bag)), 1) } 1..7);
      unless (exists $passwords{$apassword}) {
	$password = $apassword;
	last;
        }
      }
    if ($password) {
      $this->SetPassword($p, $password);
      }
    }

  $tournament->UpdateDivisions();
  }

=item $config->AssignTables();

Assign boards to tables (after the division data has been loaded, 
in case of consecutive numbering).

=cut

sub AssignTables ($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  if (defined $this->{'table_method'}) {
    # number tables consecutively across divisions
    if ($this->{'table_method'} eq 'consecutive') {
      my $base = 1;
      for my $dp (sort { $a->Name() cmp $b->Name() } $tournament->Divisions()) {
	my $np = $dp->CountPlayers();
	my $nb = int($np/2);
	my @tables : shared = ($base..$base+$nb);
	$this->{'tables'}{$dp->Name()} = \@tables;
#	warn "$dp->{'name'} $base..$base+$nb";
	$base += $nb;
        }
      }
    elsif ($this->{'table_method'} eq 'none') {
      }
    else {
      $tournament->TellUser('ebtabmet', $this->{'table_method'});
      }
    }
  }

=item $content = $c->AttachedContent($filename);

Returns content that was attached to the configuration file, or
undef if the requested content is not present.

=cut

sub AttachedContent ($$) {
  my $this = shift;
  my $filename = shift;
  if (exists $this->{'attachments'}{$filename}) {
    return $this->{'attachments'}{$filename}{'content'};
    }
  else {
    return $this->{'attachments_lc'}{lc $filename}{'content'};
    }
  }

=item CheckDirectory($path, $message);

Checks to see if $path exists, creates it if necessary and possible,
reports an error message on failure.

=cut

sub CheckDirectory ($$) {
  my ($dir, $what) = @_;
  return if -d $dir;
  eval { mkpath $dir, 0, 0755; };
  return if -d $dir;
  $_[0] = File::Spec->curdir();
  warn "Cannot create $dir, so $what will have to go in the main tsh directory.\n" if $what;
  }

=item $success = $config->ChooseFile($argv);

Used internally to choose a default configuration file.
$argv should contain whatever the user specified on the command line,
typically undef or the name of an event directory.
Sets $this->{'filename'}.
Returns true on success; dies on failure.

=cut

sub ChooseFile ($$) {
  my $this = shift;
  my $argv = shift;
  $this->{'root_directory'} = '.';
  # first choice: something specified on the command line
  if (defined $argv) {
    # if it's a directory
    if (-d $argv) {
      for my $try (qw(config.tsh tsh.config)) {
	if (-e "$argv/$try") {
	  $this->{'root_directory'} = $argv;
	  $this->{'filename'} = $try;
	  return 1;
	  }
        }
      die "$argv is a directory but has neither config.tsh nor tsh.config\n";
      }
    # else it's a configuration file
    elsif (-f $argv) {
      if ($argv =~ /\//) {
        @$this{qw(root_directory filename)} = $argv =~ /^(.*)\/(.*)/;
	return 1;
        }
      else { 
	# using the main tsh directory is a bad idea, but still allowed
	$this->{'root_directory'} = File::Spec->curdir();
	$this->{'filename'} = $argv;
	return 1; 
        }
      }
    else { die "No such file or directory: $argv\n"; }
    }
  # second choice: the directory containing the newest of */config.tsh
  elsif (my (@glob) = glob('*/config.tsh')) {
    # if more than one, choose newest
    if (@glob > 1) {
      my $newest_age = -M $glob[0];
      while (@glob > 1) {
	my $this_age = -M $glob[1];
	if ($this_age < $newest_age) {
	  $newest_age = $this_age;
	  shift @glob;
	  }
	else {
	  splice(@glob, 1, 1);
	  }
        }
      }
    @$this{qw(root_directory filename)} = $glob[0] =~ /^(.*)\/(.*)/;
#   print "Directory: $this->{'root_directory'}\n";
    if ($this->{'root_directory'} =~ /^sample\d$/) {
      print <<"EOF";
You are about to run tsh using its sample event "$this->{'root_directory'}".
This is fine for practice, but might not be want you meant.  If you
want to continue with the sample event, please press the Return key
now.  To choose a different event, enter its name and then press
the Return key.
EOF
      print "[$this->{'root_directory'}] ";
      my $choice = scalar(<STDIN>);
      $choice =~ s/^\s+//;
      $choice =~ s/\s+$//;
      $this->{'root_directory'} = $choice if $choice =~ /\S/;
      unless (-f "$this->{'root_directory'}/$this->{'filename'}") {
	die "Cannot find config.tsh in $this->{'root_directory'}/.\n";
        }
      }
    else {
      print "Using most recent config.tsh in $this->{'root_directory'}.\n";
      }
    return 1;
    }
  # third choice: ./tsh.config
  if (-f 'tsh.config') {
    # don't use colour here, as colour configuration hasn't been found
    print "Warning: use of tsh.config is deprecated.\n";
    print "Please place your event files in a subdirectory, and save your\n";
    print "configuration in that subdirectory under the name config.tsh.\n";
    $this->{'filename'} = 'tsh.config';
    return 1;
    }
  die "Cannot find a configuration file.\n";
  }

sub DecodeConsoleInput ($$) {
  my $this = shift;
  my $line = shift;
  return $line unless (defined $line) && length($line) && (defined &Win32::Console::new) && -t STDOUT && $this->Value('use_windows_code_page');
  my $cp = TSH::Utility::GetWindowsCP();
  return Encode::decode("cp$cp", $line, 1);
  }

=item $config->DivisionsLoaded();

Informs the configuration object that division data has been loaded,
so that it can perform any late initialisation.

=cut

sub DivisionsLoaded($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  $this->AssignTables();
  $this->AssignClasses();
  if ($this->{'port'}) {
    $this->AssignPasswords();
    }
  }

=item $config->ComputedValues();

Computes configuration parameter values that depend on other values.

=cut

sub ComputedValues ($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  my $pairing_system = $this->{'pairing_system'} || '';
  if ($pairing_system eq 'guelph') {
    $this->{'max_rounds'} = 6 unless defined $this->{'max_rounds'};
    $this->{'avoid_sr_runs'} = 1 unless defined $this->{'avoid_sr_runs'};
    }
  elsif ($pairing_system eq 'bracket') {
    $this->{'bracket_order'} = 'id' unless defined $this->{'bracket_order'};
    }
  elsif ($pairing_system eq 'green') {
    $this->{'max_rounds'} = 6 unless defined $this->{'max_rounds'};
    }
  elsif ($pairing_system eq 'auto') {
    if ($this->{'force_koth'}) {
      $tournament->TellUser('eautokoth');
      }
    }
  unless ($this->{'max_rounds'}) {
    # set to maximum of max_div_rounds values
    my $maxr1 = 0;
    for my $value (values %{$this->{'max_div_rounds'}}) {
      $maxr1 = $value if $maxr1 < $value;
      }
    $this->{'max_rounds'} = $maxr1 if $maxr1;
    }
  unless ($this->{'player_number_format'}) {
    $this->{'player_number_format'} = 
      $tournament->CountDivisions() == 1 ? '#%s' : '%s';
    }
  for my $div (keys %{$this->{'gibson_groups'}}) {
    my $divp = $this->{'gibson_groups'}{$div};
    $this->{'gibson_equivalent'}{$div} = &share([])
      unless exists $this->{'gibson_equivalent'}{$div};
    for my $gibson_group (@$divp) {
      my $first = $gibson_group->[0];
      for my $i (1..$#$gibson_group) {
	$this->{'gibson_equivalent'}{$div}[$gibson_group->[$i]] = $first;
#	print "$gibson_group->[$i] equiv $first in $div.\n";
	}
      }
    }
  if ($this->{'assign_firsts'}) { # can't have one without the other
    $this->{'track_firsts'} = 1;
    }
  if ($this->{'manual_pairings'}) {
    $this->{'pairing_system'} = 'manual';
    $tournament->TellUser('wmanpair');
    }
  if (defined $this->{'entry'}) {
    if ($this->{'entry'} =~ /^absp$/i) { 
      $this->{'entry'} = 'spread';
      $tournament->TellUser('wentryabsp');
      }
    elsif ($this->{'entry'} =~ /^nsa$/i) { 
      $this->{'entry'} = 'scores';
      $tournament->TellUser('wentrynsa');
      }
    elsif ($this->{'entry'} =~ /^(?:both|scores|spread)$/i) {
      $this->{'entry'} = lc $this->{'entry'};
      }
    else {
      $tournament->TellUser('ebadconfigentry', $this->{'entry'});
      $this->{'entry'} = 'scores';
      }
    if ($this->{'entry'} eq 'spread') {
      if (defined $this->{'assign_firsts'}) {
	if (!$this->{'assign_firsts'}) {
	  $tournament->TellUser('wsetassignfirsts');
	  }
	}
      $this->{'assign_firsts'} = 1;
      if (defined $this->{'track_firsts'}) {
	if (!$this->{'track_firsts'}) {
	  $tournament->TellUser('wsettrackfirsts');
	  }
	}
      $this->{'track_firsts'} = 1;
      }
    }
  else {
    $this->{'entry'} = 'scores';
    }
  $this->SetupTerminology();
  if ($this->{'player_photos'}) {
    if ($this->{'html_in_event_directory'}) {
      $tournament->TellUser('warnhied');
      }
    $this->LoadPhotoIndex();
    }
  # Don't set this earlier in case someone wants to use srand
  $this->{'random'} = rand() unless defined $this->{'random'};
  # tell divisions how many rounds they have before they are loaded
  for my $dp ($tournament->Divisions()) {
    $dp->RatingSystem($this->{'division_rating_system'}{$dp->Name()}||$this->{'rating_system'});
    my $maxr = $this->{'max_div_rounds'}{$dp->Name()} 
      || $this->{'max_rounds'};
    next unless defined $maxr;
    $dp->MaxRound0($maxr-1);
    }
  }

=item $config->Export();

Exports $config->{'key'} to $config::key, @config::key or %config::key
depending on the type of its value.
Severely deprecated.

=cut

sub Export ($) {
  my $this = shift;
  while (my ($key, $value) = each %$this) {
    my $ref = ref($value);
    if ($ref eq '') 
      { eval "\$config::$key=\$value"; }
    elsif ($ref eq 'ARRAY')
      { eval "\@config::$key=\@\$value"; }
    elsif ($ref eq 'HASH')
      { eval "\%config::$key=\%\$value"; }
    elsif ($ref eq 'CODE')
      { eval "\$config::$key=\$value"; }
    }
  }

=item $p = $config->GetPlayerByPassword($password);

If there is a player whose data entry password is (ignoring case)
equal to C<$password>, return the player; else return undefined.

=cut

sub GetPlayerByPassword ($$) {
  my $this = shift;
  my $password = shift;
  my $data = $this->{'passwords'}{uc $password} or return undef;
  return $this->{'tournament'}->GetDivisionByName($data->[0])->Player($data->[1]);
  }

=item $token = GetToken($string);

Delete a token from the beginning of the string and return it. 
Leading whitespace is skipped. 
Tokens not containing whitespace do not need to be delimited.
Tokens with whitespace should be delimited with single or double
quotes.  Backslashes and delimiters can be escaped with backslashes.
Return undef if nothing was found, or if a delimiter wasn't matched.

=cut

sub GetToken ($) {
  $_[0] =~ s/^\s+//;
  return undef unless $_[0] =~ /\S/;
  if ($_[0] =~ s/^(['"])//) {
    my $delimiter = $1;
    my $token = '';
    while (1) {
      if (s/^[^\\$delimiter]+//) {
	$token .= $&;
        }
      if (s/^\\([\\$delimiter])//) {
	$token .= $1;
        }
      elsif (s/^\\//) {
	$token .= $&;
        }
      elsif (s/^$delimiter\s*//) {
	return $token;
        }
      else {
	# unmatched delimiter
	return undef;
        }
      }
    }
  else {
    s/^(\S+)\s*// or die "assertion failed";
    return $1;
    }
  }

=item $config->InstallPhoto($pp);

If a player has a photo available, copy it to the appropriate location in the web directory
and store that location in the player object.

=cut

sub InstallPhoto ($$) {
  my $this = shift;
  my $pp = shift;
  my $name = uc $pp->Name();
  my $path = $this->{'pix'}{$name};
# return unless $path;
  $path = 'u/unknown_player.gif' unless $path;
  my $sourcefn = $this->MakeLibPixPath($path);
  my $destfn = $this->MakeHTMLPath("pix/$path");
  my (@sstat) = stat $sourcefn;
  if (!@sstat) {
#   warn $sourcefn;
    $this->{'tournament'}->TellUser('enopic', $name);
    return;
    }
  else {
#   warn "$sourcefn ok";
    }
  unless ($this->{'_readonly'}) {
    my (@dstat) = stat $destfn;
    # see if we need to copy the file
    if ((!@dstat)  # no destination file
      || $sstat[7] != $dstat[7] # different file size
      || $sstat[9] > $dstat[9] # stale mod time
      ) {
      # create the directory if necessary
      my ($destvol, $destdir, $destfile) = File::Spec->splitpath($destfn);
      my $destpath = File::Spec->catpath($destvol, $destdir, '');
      unless (-d $destpath) {
  #     warn "Creating $destpath for $destfn\n";
	eval { mkpath $destpath, 0, 0755; };
	}
      # copy the file
      copy($sourcefn, $destfn) 
	or $this->{'tournament'}->TellUser('ecopypic', $sourcefn, $!);
      }
    }
  $pp->PhotoURL("pix/$path");
  }

=item $config->LastModified();

Returns the time (in seconds since the Unix epoch) when the most
recent associated file (config.tsh or *.t) was modified.

=cut

sub LastModified ($) {
  my $this = shift;
  my $modtime;
  my (@stat) = stat $this->MakeRootPath($this->{'filename'});
  $modtime = $stat[9] if defined $stat[9];
  my (@files);
  if (defined $this->{'tournament'}) {
    @files = map { $this->MakeRootPath($_->File()) } 
      $this->{'tournament'}->Divisions();
    }
  else {
    my $divsp = $this->Load(undef, 1);
    @files = map { $this->MakeRootPath($_) } values %$divsp;
    }
  for my $fn (@files) {
    (@stat) = stat $fn;
    $modtime = $stat[9] if (defined $stat[9]) && $stat[9] > $modtime;
    }
  return $modtime;
  }


=item $rank1 = $c->LastPrizeRank($divname);

Return the one-based rank of the last place to which prizes are paid out
in division C<$divname>.

=cut

sub LastPrizeRank ($$) {
  my $this = shift;
  my $divname = shift;
  my $last_prize_rank = $this->{'prize_bands'};
  $last_prize_rank = $last_prize_rank->{$divname} if defined $last_prize_rank;
  $last_prize_rank = $last_prize_rank->[-1] if defined $last_prize_rank;
  if (!defined $last_prize_rank) {
    $this->{'tournament'}->TellUser('wwant_prize_bands');
    $last_prize_rank 
      = int($this->{'tournament'}->GetDivisionByName($divname)->CountPlayers()/4) || 1;
    }
  return $last_prize_rank;
  }

=item $c->LoadPhotoIndex();

Load the photo index (or complain about it being missing)

=cut

sub LoadPhotoIndex ($) {
  my $this = shift;
  my $fn = $this->MakeLibPixPath('photos.txt');
  my $fh = new FileHandle($fn, "<");
  unless ($fh) {
    $this->{'tournament'}->TellUser('enopixind', $fn, $!);
    return;
    }
  binmode $fh, ':encoding(isolatin1)';
  while (<$fh>) {
    chomp;
    my ($given, $surname, $path) = split(/\t/);
    unless (defined $path) {
      $this->{'tournament'}->TellUser('ebadpixind', $.);
      return;
      }
    my $name = uc(length($surname) ? "$surname, $given" : $given);
    $this->{'pix'}{$name} = $path;
    }
  close($fh);
  }

=item $path = $c->MakeBackupPath($relpath);

Return a path to a file in the configured backup directory.

=cut

sub MakeBackupPath ($$) {
  my $this = shift;
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($this->{'backup_directory'})
    ? File::Spec->join($this->{'backup_directory'}, $relpath)
    : File::Spec->join($this->{'root_directory'},
      $this->{'backup_directory'}, $relpath);
  }

=item $path = $c->MakeHTMLPath($relpath);

Return a path to a file in the configured HTML directory.

=cut

sub MakeHTMLPath ($$) {
  my $this = shift;
  my $relpath = shift;
  my (@path_components) = $this->{'html_directory'};
  push(@path_components, $relpath) if defined $relpath;
  return File::Spec->file_name_is_absolute($this->{'html_directory'})
    ? File::Spec->join(@path_components)
    : $this->MakeRootPath(File::Spec->join(@path_components));
  }

=item $path = $c->MakeLibPath($relpath);

Return a path to a file in the lib directory.

=cut

sub MakeLibPath ($$) {
  my $this = shift;
  my $relpath = shift;
  # first we find where lib is
  if (!$this->{'lib_directory'}) {
    my $lib_perl_tfile_path = $main::INC{'TFile.pm'};
    my ($libvol, $libdir, $file) = File::Spec->splitpath($lib_perl_tfile_path);
    my (@libdir) = File::Spec->splitdir($libdir);
# warn "1libdir: ".join(',',@libdir)."\n";
    pop @libdir if $libdir[-1] eq '';
    pop @libdir; # remove the 'perl';
# warn "2libdir: ".join(',',@libdir)."\n";
    $this->{'lib_directory'} = File::Spec->catpath($libvol, File::Spec->catdir(@libdir), '');
    }
  return File::Spec->join($this->{'lib_directory'}, $relpath)
  }

=item $path = $c->MakeLibPixPath($relpath);

Return a path to a file in the lib/pix directory.

=cut

sub MakeLibPixPath ($$) {
  my $this = shift;
  my $relpath = shift;
  if ($this->{'photo_database'} && $this->{'photo_database'} ne 'nsa') {
    $relpath = File::Spec->join($this->{'photo_database'}, $relpath);
    }
  $relpath = File::Spec->join('pix', $relpath);
  return $this->MakeLibPath($relpath);
  }

=item $path = $c->MakeRootPath($relpath);

Return a path to a file in the root directory for the event, where
its config.tsh file is.
Arguably not the best-named function ever.

=cut

sub MakeRootPath ($$;$) {
  my $this = shift;
  my $relpath = shift;
  my $separator = shift;
  return File::Spec->file_name_is_absolute($relpath)
    ? $relpath
    : (defined $separator) 
      ? "$this->{'root_directory'}$separator$relpath"
      : File::Spec->join($this->{'root_directory'}, $relpath)
  }

=item $config->Normalise();

Called internally after the configuration is read to normalise 
parameter values.

=cut

sub Normalise ($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  for my $key (qw(pairing_system realm)) {
    $this->{$key} = lc $this->{$key};
    }
  unless (($this->{'bracket_order'}||'') =~ /^(?:|id|rating)$/) {
    $tournament->TellUser('ebadconfigbracket_order', $this->{'bracket_order'});
    $this->{'bracket_order'} = 'id';
    warn $this->{'bracket_order'};
    }
  unless ($this->{'realm'} =~ /^(?:absp|caspa|deu|nsa|sgp|thai|wespa)$/) {
    $tournament->TellUser('ebadconfigrealm', $this->{'realm'});
    $this->{'realm'} = 'nsa';
    }
  unless($this->{'pairing_system'} =~ /^(?:auto|basd|bracket|chew|guelph|green|manual|nast)$/) {
    $tournament->TellUser('ebadconfigpairing', $this->{'pairing_system'});
    $this->{'pairing_system'} = 'auto';
    }
  }

=item $config->Normalise2();

Called internally after the configuration is read to normalise 
parameter values that might or might not have been set by a realm
default.

=cut

sub Normalise2 ($) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  # post-realm normalisation
  $this->{'rating_system'} = lc($this->{'rating_system'}||'nsa');
  unless($this->{'rating_system'} =~ /^(?:absp|aus|deu|elo|nsa|nsa lct|nsa2008|nsa2008 lct|none|thai|wespa)$/) {
    $tournament->TellUser('ebadconfigrating', $this->{'rating_system'});
    $this->{'entry'} = 'auto';
    }
  }

=item $path = $c->PhotoPath($name);

Return a relative path for a photo corresponding to a named player.
Most of the time, you'll want to make sure that a photo is actually
installed at that path relative say to your event web directory,
and use C<TSH::Config::InstallPhoto> and C<TSH::Player::PhotoURL>.

=cut

sub PhotoPath ($$) {
  my $this = shift;
  my $name = shift;
  $name =~ s/ /, / unless $name =~ /,/;
  return $this->{'pix'}{$name} || 'u/unknown_player.gif';
  }

=item $config->Read();

Read the associated configuration file and set configuration values.

=cut

sub Read ($) {
  my $this = shift;
  my $fn = $this->{'filename'};
  my $tournament = $this->{'tournament'};
  my $fqfn = $this->MakeRootPath($fn);
  my $fh = new FileHandle($fqfn, "<")
    or die "Can't open $fn: $!\n";
  binmode $fh, ':encoding(isolatin1)';
  local($_);
  $tournament->TellUser('iloadcfg', $fqfn) if $tournament->can('TellUser');
  while (<$fh>) { 
    if (/^#begin_file\s+(.+)/) {
      my $options = $1;
      my $end = $_; $end =~ s/begin/end/;
      my %data : shared;
      while (length($options)) {
	$options =~ s/^(\w+)=// or die "Can't parse: $options";
	my $key = $1;
	if ($options =~ s/^'([^']*?)'//
	  || $options =~ s/^"([^"]*?)"//
	  || $options =~ s/^(\S+)//) 
	  { $data{$key} = $1; }
	else {
	  die "Can't parse: $options";
	  }
	$options =~ s/^\s+//;
        }
      die "#begin_file is missing required option 'name'" 
        unless defined $data{'name'};
      $data{'content'} = '';
      while (<$fh>) {
	last if $_ eq $end;
	$data{'content'} .= $_;
        }
      $this->{'attachments_lc'}{lc $data{'name'}} =
      $this->{'attachments'}{$data{'name'}} = \%data;
      next;
      }
    push(@{$this->{'_saved'}}, $_)
      unless /^\s*config\s+\w+_password\s+/;;
    s/^\s*(#.*)//; s/^\s*//; s/\s*$//; next unless /\S/;
    if (/^division\s+(\S+)\s+(.*)/i) {
      my $dname = $1;
      my $dfile = $2;
      my $dp = new TSH::Division;
      $dp->Name($dname);
      $dp->File($dfile);
      $tournament->AddDivision($dp);
      next;
      }
    if (s/^perl\s+//i) { 
      local(*main::gTournament) = \%$tournament; # backward compatibility
      eval $_;
      print "eval: $@\n" if length($@);
      next;
      }
    if (/^config\s+(\w+)\s*((?:(?:\[[^]]+\])|(?:\{[^}]+\}))*)\s*=\s*(.*)/i) { 
      my $key = $1;
      my $subkey = $2;
      my $rhs = $3;
      $rhs =~ s/\s*;\s*$//;
      my $s = "\$this->{'$key'}\U$subkey\E=TSH::Utility::ShareSafely($rhs)";
      eval $s;
#     print "\n***$s\n";
      if ($@) {
	$tournament->TellUser('ebadcfg', $_);
	warn "$@\n";
        }
      next;
      }
    # classes divname nclasses: specify number of prize classes in a division
    if (/^classes\s+(\S+)\s+(\S+)\s*$/i) {
      my $dname = $1;
      my $classes = $2;
      my $dp = $tournament->GetDivisionByName($1);
      unless ($dp) {
	$tournament->TellUser('eclassdiv', $dname);
	die "\n";
        }
      unless ($classes =~ /^\d+$/ && $classes > 1 && $classes <= 26) {
	$tournament->TellUser('eclassnum', $classes);
	die "\n";
        }
      $dp->Classes($classes);
      next;
      }
    if (s/^autopair\s+//i) { 
      $this->{'pairing_system'} = 'auto';
      if (/^(\w+)\s+(\d+)\s+(\d+)\s+(\w+)\s+(.*)/) {
	my ($div, $sr, $round, $command, $args) = ($1, $2, $3, $4, $5);
	if ($sr >= $round) {
	  $tournament->TellUser('eapbr', $div, $sr, $round);
	  die "\n";
	  }
	my (@args) = split(/\s+/, $args);
	my $ref = $this->{'autopair'};
	if (exists $ref->{uc $div}) {
	  $ref = $ref->{uc $div};
	  }
	else {
	  $ref = $this->{'autopair'}{uc $div} = &share([]);
	  }
	$ref = $ref->[$round] = &share([]);
	push(@$ref, $sr, $command, @args);
        }
      else {
	chomp;
        $tournament->TellUser('ebadap', $_, $fn);
	die "\n";
        }
      next;
      }
    if (s/^prize\s+//i) {
      my %pdata : shared;
      my $save = $_;
      for my $key (qw(type subtype division value)) {
	unless (defined($pdata{$key} = GetToken($_))) {
          $tournament->TellUser('ebadprzf', $key, $save);
	  }
	last if $pdata{'type'} eq 'separator';
        }
      while (s/^\s*(\w+)=//) {
	my $key = $1;
	my $value = GetToken($_);
	$value = '' unless defined $value;
	$pdata{$key} = $value;
        }
      if (/\S/) {
	$tournament->TellUser('ebadprzt', $save, $_);
        }
      else {
	push(@{$this->{'prizes'}}, \%pdata);
#	warn "saved $pdata{type} $pdata{subtype} =".scalar(@{$this->{'prizes'}});
        }
      next;
      }
    $tournament->TellUser('ebadcfg', $_);
    die "\n";
    }
  close($fh); 
# print "Configuration file loaded.\n";
  $this->Normalise();
  $this->RealmDefaults();
  $this->ComputedValues();
  $this->Normalise2();
  }

=item $old_value = $c->ReadOnly();
=item $c->ReadOnly($new);

Get/set the read-only flag.

=cut

sub ReadOnly ($;$) { 
  my $this = shift; 
  my $new = shift; 
  TSH::Utility::GetOrSet('_readonly', $this, $new); 
  }

=item $c->RealmDefaults();

Assign realm-based defaults.

=cut

{
  my (%realm_data) = (
    'absp' => {
      'alpha_pair_page_break' => 10000,
      'assign_firsts' => 1,
      'avoid_sr_runs' => 1,
      'bye_firsts' => 'ignore',
      'bye_spread' => 75,
#     'entry' => 'spread',
#     'gibson_class' => 'A', # causes problems if classes unspecified
      'html_in_event_directory' => 1,
      'photo_database' => 'centrestar',
      'rating_list' => 'absp',
      'rating_system' => 'absp',
      'surname_last' => 1,
      'table_method' => 'consecutive',
      'terminology' => 'absp',
      'track_firsts' => 1,
      },
    'deu' => {
      'assign_firsts' => 0,
      'bye_firsts' => 'ignore',
      'message_file' => 'lib/messages/deu.txt',
      'rating_list' => 'deu',
      'rating_system' => 'deu',
      'terminology' => 'deu',
      'use_windows_code_page' => 1,
      },
    'nsa' => {
      'alpha_pair_page_break' => 10,
      'bye_firsts' => 'alternate',
      'bye_spread' => 50,
      'entry' => 'scores',
#     'rating_system' => 'nsa',
      'rating_system' => 'nsa2008',
      },
    'sgp' => {
      'bye_firsts' => 'ignore',
      'bye_spread' => 50,
      'entry' => 'scores',
      'rating_system' => 'nsa',
      'surname_last' => 1,
      },
    'thai' => {
      'bye_firsts' => 'alternate',
      'bye_spread' => 100,
      'entry' => 'scores',
      'rating_list' => 'thai',
      'rating_system' => 'thai',
      'spread_cap' => 350,
      },
    );
sub RealmDefaults ($) { 
  my $this = shift; 
  my $defp = $realm_data{$this->{'realm'}} || $realm_data{'nsa'};
  while (my ($key, $value) = each %$defp) {
    $this->{$key} = $value unless defined $this->{$key};
    }
  }
}

=item $s = $c->Render($division_filter);

Render and return a copy of the current configuration information.
If C<&$division_filter($division)> is true for a division, 
include the public information in the division's  C<.t> file.

=cut

sub Render ($;$) {
  my $this = shift;
  my $division_filter = shift;

  my $tournament = $this->{'tournament'};
  my $version = $::gkVersion ? " version $::gkVersion" : '';
  my $s = '';
  if ($division_filter) {
    $s .= "# tsh archive file\n";
    }
  else {
    $s .= "# tsh configuration file\n";
    }
  $s .= "# automatically generated by tsh$version.\n";
  $s .= join('', @{$this->{'_saved'}});
  $s =~ s/\n?$/\n/;
  if ($division_filter) {
    for my $dp ($tournament->Divisions()) {
      next unless &$division_filter($dp);
      my $dfile = $dp->File();
      $s .= "#begin_file name=$dfile\n";
      for my $pp ($dp->Players()) {
	$s .= TFile::FormatLine($pp, 'public');
	}
      $s .= "#end_file name=$dfile\n";
      }
    }
  return $s;
  }

=item $old_value = $c->RootDirectory();
=item $c->RootDirectory($new);

Get/set the root directory.  Use with extreme caution.

=cut

sub RootDirectory ($;$) { 
  my $this = shift; 
  my $new = shift; 
  TSH::Utility::GetOrSet('root_directory', $this, $new); 
  }

=item $config->Save();

Save the current configuration in its configuration file.

=cut

sub Save ($) {
  my $this = shift;
  my $fn = $this->MakeRootPath($this->{'filename'});
  my $fh = new FileHandle($fn, "<");
  unless ($fh) {
    return "<div class=failure>Could not open '$fn': $!</div>";
    }
  local($/) = undef;
  my $s = <$fh>;
  close($fh);
  # update user options
  while (my ($key, $datap) = each %user_option_data) {
    my $value = ${$config::{$key}};
    my $type = $datap->{'type'};
    unless (defined $value) {
      if ($type eq 'boolean') { $value = 0; }
      elsif ($type eq 'integer') { $value = 0; }
      elsif ($type eq 'string') { $value = ''; }
      else { die "oops - unknown type $datap->{'type'} for $key\n"; }
      }
    if ($type eq 'string') { $value = '"' . quotemeta($value) . '"'; }
    $s =~ s/^\s*config\s+$key\s*=.*$/config $key = $value/m
      or $s .= "config $key = $value\n";
    }
  my $error = '';
  copy($fn, $this->{'backup_directory'} . time . '.tsh')
    or ($error = "A backup copy of the configuration could not be saved: $!");
  TSH::Utility::ReplaceFile($fn, $s)
    or ($error = "A new copy of the configuration could not be written: $!");
  return $error 
    ? "<div class=failure>Your changes were not saved. $error</div>"
    : "<div class=success>Your changes were saved.</div>";
  }

=item $c->SetPassword($p, $password);

Sets a players password and updates the internal password map.

=cut

sub SetPassword ($$$) {
  my $this = shift;
  my $p = shift;
  my $password = shift;
  my @data : shared;
  my $dp = $p->Division();
  @data = ($dp->Name(), $p->ID());
  $this->{'passwords'}{$password} = \@data;
  $p->Password($password);
  $dp->Dirty(1);
  }

=item $c->Setup();

Perform any initialisation required after the configuration has been
read: create necessary directories.

=cut

sub Setup ($) {
  my $this = shift;
  # do nothing if we are not supposed to change the filesystem
  return if $this->{'_readonly'};
  # Check for backup and HTML directories
  CheckDirectory ($this->MakeRootPath($this->{'backup_directory'}), "backups");
  CheckDirectory ($this->MakeRootPath($this->{'html_directory'}), "web files");
  my $realm = $this->Value('realm');
  # install CSS stylesheet if missing
  my $destfn = $this->MakeHTMLPath('tsh.css');
  my $sourcefn = $this->MakeLibPath(
    $realm eq 'nsa' ? 'tsh.css' : "tsh-$realm.css");
  copy($sourcefn, $destfn) unless -f $destfn;
  if ($this->Value('html_in_event_directory')) {
    $destfn = $this->MakeRootPath('tsh.css');
    copy($sourcefn, $destfn) unless -f $destfn;
    }
  $this->SetupWindows();
  }

=item $c->SetupTerminology();

Set up tournament terminology database.

=cut

sub SetupTerminology ($) {
  my $this = shift;
  if (defined $this->{'terminology'}) {
    $this->{'terminology'} = lc $this->{'terminology'};
    }
  else {
    $this->{'terminology'} = 'nsa';
    }
  my (@filenames) = ('nsa');
  unshift(@filenames, $this->{'terminology'}) unless $this->{'terminology'} eq 'nsa';
  @filenames = grep { -f $_ } map { 
    $this->MakeLibPath(
      File::Spec->catfile('terms', "$_.txt"))
    } @filenames;

  unless ($this->{'_termdict'} = new UserMessage(
    'file' => \@filenames,
    'nodetail' => 1,
    'hidecode' => 1,
    )) {
    my $tournament = $this->{'tournament'};
    $tournament->TellUser('ebadconfigterms', $this->{'terminology'}, $!);
    }
  }

sub SetupWindows ($) {
  my $this = shift;
  return unless defined &Win32::Console::new;
  if ($this->{'use_windows_code_page'}) {
    my $cp = TSH::Utility::GetWindowsCP();
#   $cp = 850; system 'chcp 850'; # for testing Western Europe settings from elsewhere
    # following doesn't work with TERM::ReadLine
#   binmode STDIN, ":encoding(cp$cp)" or die "binmode STDIN failed: $!";
    binmode STDERR, ":encoding(cp$cp)" or die "binmode STDERR failed: $!";
    binmode STDOUT, ":encoding(cp$cp)" or die "binmode STDOUT failed: $!";
    print "Using Windows code page $cp.\n";
    }
  if ($this->{'windows_console_unicode'}) {
    TSH::Utility::SetupWindowsUnicodeConsole();
    }
  }

=item my $s = $c->Terminology($code, @argv);

Return a term from the currently selected international terminology database.

=cut

sub Terminology ($$@) {
  my $this = shift;
  my $code = shift;
  if ($code =~ /^(?:Sprd|Spread)$/ && $this->Value('oppless_spread')) {
    $code = 'SOS';
    }
  return $this->{'_termdict'}->Render(
    'code' => $code,
    'argv' => \@_,
    );
  }

sub UninstallPhoto ($$) {
  my $this = shift;
  my $pp = shift;
  # for now, don't actually remove file, which might be needed by someone
  # else, if say it's withheld.gif
  $pp->PhotoURL('');
  }

=item @option_names = UserOptions();

Returns a list of user-configurable option names.

=cut

sub UserOptions () {
  return keys %user_option_data;
  }

=item $help = UserOptionHelp($option_name);

Returns the help text associated with a user-configurable option name.

=cut

sub UserOptionHelp ($) {
  my $key = shift;
  return $user_option_data{$key}{'help'};
  }

=item $type = UserOptionType($option_name);

Returns the type of a user-configurable option name:
boolean, integer or string.

=cut

sub UserOptionType ($) {
  my $key = shift;
  return $user_option_data{$key}{'type'};
  }

=item $error = UserOptionValidate($option_name, $option_value);

Checks to see if $option_value is a valid value for $option_name.
If it isn't, an error message is returned.  If it is, $option_value
may be cleaned slightly before being returned.

=cut

sub UserOptionValidate ($$) {
  my $key = shift;
  my $sub = $user_option_data{$key}{'validate'};
  return '' unless $sub;
  return &$sub($key, $_[0]);
  }

=item $error = $config->ValidateLocalDirectory($option_name, $option_value);

Returns an error message if $option_value is not a valid value for $option_name.
Returns the null string if it is valid.
Cleans up $option_value if necessary.
Checks to see if the value designates a local directory that exists
or can be created.
Paths may be relative to C<$config->{'root_directory'}>.

=cut

sub ValidateLocalDirectory ($$$) {
  my $this = shift;
  my $key = shift;
  my $path = $_[0];
  $path =~ s/^\s+//;
  $path =~ s/\s+$//;
  $path = '/' unless $_[0] =~ /\/$/;
  $path = File::Spec->canonpath($path);
  my $fqpath = $path;
  if (!File::Spec->file_name_is_absolute($path)) {
    $fqpath = File::Spec->join($this->{'root_directory'}, $path);
    }
  if (-e $fqpath) { 
    unless (-d $fqpath) {
      return "$path exists but is not a directory.";
      }
    }
  else {
    mkpath $fqpath, 0, 0755
      or return "$fqpath does not exist and cannot be created.";
    }
  $_[0] = $path;
  return 1;
  }

=item $old_value = $c->Value($key);
=item $c->Value($key, $value);

Get/set a configuration value.  If setting, return the old value, if any.

=cut

sub Value ($$@) { 
  my $this = shift; 
  my $key = shift; 
  TSH::Utility::GetOrSet($key, $this, @_); 
  }

=item $c->Write($fh, $division_filter);

Write a copy of the current configuration information to file handle
C<$fh>.  See C<Render()> for C<$division_filter> usage.

=cut

sub Write ($$;$) {
  my $this = shift;
  my $fh = shift;
  my $division_filter = shift;

  print $fh $this->Render($division_filter);
  }

=head1 BUGS

See master issues list.

Should think about combining Save() and Write().

=cut

1;
