#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Processor;

use strict;
use warnings;
use TSH::ParseArgs;
use TSH::Utility;
use TSH::VCommand;

=pod

=head1 NAME

TSH::Processor - interprets text commands related to a tournament

=head1 SYNOPSIS

This class interprets text commands received from a user and applies
them with reference to a TSH::Tournament object.

This is the easiest way to use this class:

  my $p = new TSH::Processor($tournament)
  $p->RunInteractive();

User commands are typically dispatched to a call to the Run method
of a subclass of TSH::Command.

  sub TSH::Command::Whatever::Run ($$@) {
    my $this = shift;
    my $tournament = shift;
    my (@argv) = @_;
    }

A "quit" command should inform the Processor that it is time to quit:

  $this->Processor()->QuittingTime(1);

If more than one user is using the same tournament, they should each
have their own Processor.  Note that at present users are not 
automatically notified if data that they are interested in has been
changed by another user.

A modal command may transfer focus to a subclass of Processor:

  sub TSH::Command::Whatever::Run ($$@) {
    my $this = shift;
    my $tournament = shift;
    my (@argv) = @_;

    my $subprocessor = new TSH::Processor::Whatever($this, @argv);
    $this->Processor()->Push($subprocessor);
    }

The subprocessor must define its own user prompt using either of
the following techniques:

  sub Prompt ($) { my $this = shift; return time . '> '; }
  $this->{'prompt'} = 'tsh> ';

The subprocessor will eventually relinquish control:

  $this->Command()->Processor->Pop();

There are several methods used to interact with members of
TSH::Command.

  $p->AddCommand($command); # add command to the list that we handle
  @c = $p->CommandNames(); # list of commands we know
  $p->GetCommandByName($); # return reference to command object, given name
  $p->ReplaceCommand($command); # replace command (typically a VCommand stub)


=head1 ABSTRACT

This Perl module is a class for objects that receive text commands
from users, interprets them and applies them with reference to
a TSH::Tournament structure.

=cut

=head1 DESCRIPTION

=head2 Variables

=over 4

=item $p->{'child'}

Reference to TSH::Processor object that has been pushed on top of us.

=item $p->{'cmdhash'}

Reference to hash mapping command name to TSH::Command object.

=item $p->{'cmdlist'}

Reference to list of known TSH::Command objects.

=item $p->{'parent'}

Reference to the TSH::Processor object that is below us on the stack.

=item $p->{'parser'}

Reference to TSH::ParseArgs parser object.

=item $p->{'prompt'}

Prompt displayed to elicit response from user.

=item $p->{'quit'}

Boolean, true if we want to stop receiving input.

=item $p->{'tournament'}

Reference to TSH::Tournament object associated with this Processor.

=back

=head2 Methods

The following methods are defined in this module.

=over 4

=cut

sub AddCommand ($$);
sub CommandNames ($);
sub GetCommandByName ($$);
sub initialise ($$);
sub new ($$);
sub Parser ($;$);
sub Pop ($);
sub Process ($$;$);
sub Push ($$);
sub QuittingTime ($;$);
sub ReplaceCommand ($$);
sub RunInteractive ($);
sub RunServer ($);

=item $t->AddCommand($c)

Add a command to the processor.

=cut

sub AddCommand ($$) { 
  my $this = shift;
  my $command = shift;
  for my $name ($command->Names()) {
    if (exists $this->{'cmdhash'}{$name}) {
      $this->{'tournament'}->TellUser('econfcmd', $name);
      }
    $this->{'cmdhash'}{$name} = $command;
#   print "AddCommand: $name.\n";
    }
  # clear the command list so that CommandNames will rebuild it
  $this->{'cmdlist'} = [];
  }

=item @d = $p->CommandNames()

Return a list of a processor's command names.

=cut

sub CommandNames ($) { 
  my $this = shift;
  my $csp = $this->{'cmdlist'};
  return @$csp if $csp && @$csp;
  $csp = $this->{'cmdlist'} = [sort keys %{$this->{'cmdhash'}}];
  return @$csp;
  }

=item $p->Flush();

Write out all volatile data to disk.

=cut

sub Flush ($$) {
  my $this = shift;
  my $tournament = $this->{'tournament'};
  if (my $count = $tournament->UpdateDivisions()) {
    if (my $cmds = $tournament->Config()->Value('hook_division_update')) {
      $this->Process($cmds, 'nohistory');
      }
    }
  }

=item $c = $p->GetCommandByName($cn);

Obtain a Command pointer given a command name.

=cut

sub GetCommandByName ($$) {
  my $this = shift;
  my $cname = shift;
  my $c = $this->{'cmdhash'}{lc $cname};
  return $c;
  }

=item $c->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$) {
  my $this = shift;
  $this->{'tournament'} = shift;
  $this->{'child'} = undef;
  $this->{'cmdhash'} = undef;
  $this->{'cmdlist'} = undef;
  $this->{'parent'} = undef;
  $this->{'parser'} = new TSH::ParseArgs($this);
  $this->{'prompt'} = 'tsh> ';
  $this->{'quit'} = undef;
  TSH::VCommand::LoadAll($this);
  my $xpath = $this->{'tournament'}->Config()->Value('external_path');
  if ($xpath) {
    TSH::XCommand::LoadAll($this, $xpath);
    }
  return $this;
  }

=item $c = new TSH::Processor($tournament);

Creator method.

=cut

sub new ($$) { return TSH::Utility::new(@_); }

=item $parser = $p->Parser();

=item $p->Parser($parser);

Get/set the parser object associated with us.

=cut

sub Parser ($;$) { TSH::Utility::GetOrSet('parser', @_); }

=item $command->Pop();

Pops our child off the processor stack.
Dies if the child is not at the top of the stack.

=cut

sub Pop ($) {
  my $this = shift;
  my $child = $this->{'child'};
  die "Stack empty" unless defined $child;
  die "Can't pop from middle of stack" if $child->{'child'};
  $child->{'parent'} = undef;
  $this->{'child'} = undef;
  }

=item $p->Process($text) or print "Huh?\n";

Process a command line.

=cut

sub Process ($$;$) {
  my $this = shift;
  my $text = shift;
  my $nohistory = shift;
  my $config = $this->{'tournament'}->Config();
  my $save_nohistory = $config->Value('nohistory') || 0;

  if ($this->{'child'}) { 
    $config->Value('nohistory', $nohistory);
    my $status = $this->{'child'}->Process($text); 
    $config->Value('nohistory', $save_nohistory);
    return $status;
    }
  $text =~ s/^\s+//; 
  return 1 unless $text =~ /\S/;
  my (@subcommands) = split(/\s*;\s*/, $text);
  my $ok = 1;
  for my $subcommand (@subcommands) {
    my(@argv) = split(/\s+/, $subcommand);
    my $arg0 = lc $argv[0];
    my $commandsp = $this->{'cmdhash'};
    my $command = $commandsp->{$arg0};
    unless ($command) {
      $this->{'tournament'}->TellUser('ebadcmd', $arg0);
      $ok = 0;
      next;
      }
    my (@parsed_argv)= $this->{'parser'}->Parse(\@argv, $command->ArgumentTypes());
    if ($this->{'parser'}->Error()) {
      print $command->Usage();
      $ok = 0;
      next;
      }
    $command = $commandsp->{$arg0}; # ArgumentTypes() might have called VCommand::Load()
    $config->Value('nohistory', $nohistory);
    $command->Run($this->{'tournament'}, @parsed_argv);
    $config->Value('nohistory', $save_nohistory);
    }
  return $ok;
  }

=item $p->Prompt()

Return the current text prompt.

=cut

sub Prompt ($) {
  my $this = shift;

  if ($this->{'child'}) { return $this->{'child'}->Prompt(); }
  return $this->{'prompt'};
  }

=item $p->Push($subprocessor)

Push a processor onto the stack.
Dies if we are not at the top of the stack.

=cut

sub Push ($$) {
  my $this = shift;
  my $child = shift;

  die "Can't push into middle of stack" if $this->{'child'};
  $this->{'child'} = $child;
  $this->{'child'}->{'parent'} = $this;
  }

=item $boolean = $p->QuittingTime();

=item $p->QuittingTime($boolean);

Get/set whether or not it's time to quit.

=cut

sub QuittingTime ($;$) { TSH::Utility::GetOrSet('quit', @_); }

=item $t->ReplaceCommand($d)

Replace a command in the tournament's user interface,
either because a new version has been downloaded or
because the command has been invoked for the first
time and a stub command object is being replaced.

=cut

sub ReplaceCommand ($$) { 
  my $this = shift;
  my $command = shift;
  for my $name ($command->Names()) {
    unless (exists $this->{'cmdhash'}{$name}) {
      TSH::Utility::Error "Adding '$name' to the command list, replacing nothing?!\n";
      }
    $this->{'cmdhash'}{$name} = $command;
#   print "ReplaceCommand: $name.\n";
    }
  # 'cmdlist' will be rebuilt by the 'help' command.
  # Could insert here, but probably faster to rebuild once when help is
  # asked for rather than insert each time a command is added
  $this->{'cmdlist'} = []; 
  }

=item $p->RunInteractive();

Run an interactive tsh session with the user, return when the
user exits.

=cut

sub RunInteractive ($) {
  my $this = shift;
  if (-t STDIN && -t STDOUT) {
    my $config = $this->Tournament()->Config();
    eval "use Term::ReadLine";
    my $term = Term::ReadLine->new("tsh $::gkVersion");
  # my $ofh = $term->OUT || \*STDOUT;
    while (1) {
      $_ = $config->DecodeConsoleInput($term->readline($this->Prompt()));
      last unless defined $_;
      $this->Process($_) or $this->Tournament()->TellUser('ienterhelp');
      last if $this->QuittingTime();
      }
    }
  else {
    while (<>) {
      $this->Process($_) or $this->Tournament()->TellUser('ienterhelp');
      last if $this->QuittingTime();
      }
    }
  }

=item $p->RunServer();

Run a tsh HTML server session.  If one can't be launched, fall back to
an interactive tsh session.  If threads are enabled, run both an HTML
server session and an interactive session in separate threads until
the interactive session exits.

=cut

sub RunServer ($) {
  my $this = shift;
  my $tournament = $this->Tournament();
  &::Use("TSH::Server");
  my $server 
    = TSH::Server->new($tournament->Config()->Value('port'), $tournament);
  unless ($server->Start()) {
    $tournament->TellUser('etshtpon', $server->Error());
    $this->RunInteractive();
    return;
    }
  $this->Process('GUI');
  eval { threads->list(); };
  my $has_threads = $@ eq '';
  if ($has_threads) {
    my $serverthread = threads->new(sub {
      while ($server->Run()) { }
      $server->Stop();
      });
    $this->RunInteractive();
    print "\nShutting down server.\n";
    $server->Stop();
    $serverthread->join; # would block until server completed execution, should instead signal exit
    }
  else {
    while ($server->Run()) { }
    $server->Stop();
    }
  }

=item $t = $p->Tournament();

=item $p->Tournament($t);

Get/set the TSH::Tournament associated with the processor.

=cut

sub Tournament ($;$) { TSH::Utility::GetOrSet('tournament', @_); }

=back

=cut

=head1 BUGS

Commands should arguably not receive $tournament as their second
argument, as this can be retrieved from the command object.
On the other hand, they always need the tournament object,
so it saves a little coding and execution time, and it would be
a lot of work to recode all the command code.

=cut

1;
