#!/usr/bin/perl -w

## tsh - tournament shell
## Copyright (C) 1998-2007 by John J. Chew, III.

=head1 NAME

B<tsh> - Scrabble tournament management shell

=head1 SYNOPSIS

B<tsh> [directory|configuration-file]

=head1 DESCRIPTION

For user information, see the accompanying HTML documentation.
This builtin (pod) documentation is intended for program maintenance 
and development.

=cut

## Version history: moved to doc/news.html

## public libraries

BEGIN { unshift(@::INC, "$::ENV{'HOME'}/lib/perl") if defined $::ENV{'HOME'}; }
use strict;
use lib './lib/perl';
use FileHandle;
BEGIN {
  if (-f 'lib/threads.txt') {
    require threads;
    import threads;
    }
  }

# use warnings FATAL => 'uninitialized';
# $::SIG{__WARN__} = sub { eval 'use Carp'; &confess($_[0]); };
# $::SIG{__DIE__} = sub { eval 'use Carp'; &confess($_[0]); };

## private libraries

use TSH::Command;
use TSH::Config;
use TSH::PairingCommand;
use TSH::Processor;
use TSH::Player;
use TSH::Tournament;
use TSH::XCommand;
use TSH::Utility qw(Debug DebugOn);

## global constants
our $gkVersion = '3.300';

## prototypes

# sub CheckGroupRepeats ($$);
# sub CmdFactorPairQuads ($$);
sub DefineExternal ($$$;$);
# sub DoFactor ($$$$);
# sub DoFactorGroup ($$$);
sub lint ();
sub LockFailed ($);
sub Main ();
sub Prompt ();
sub ReopenConsole ();
sub Use ($);

sub lint () {
  lint;
  }

=head1 SUBROUTINES

=over 4

=cut

=item DefineExternal $command_name, $script_path, $template, $default_args;

Add an external command (plug-in) to the user interface.

=cut

sub DefineExternal ($$$;$) {
  my $name = lc shift;
  my $script = shift;
  my $template = shift;
  my $default_args = shift;

  my $command = new TSH::XCommand($main::processor, "$main::path/$script",
    [$name], $template, $default_args);
  $main::processor->AddCommand($command);
# print " $name";
  return 1;
  }

=item LockFailed $reason;

Report that tsh could not open its lock file and abort.

=cut

sub LockFailed ($) {
  my $reason = shift;
  print <<"EOF";
System call failed: $reason

You should not run more than one copy of tsh using the same 
configuration file at the same time.  tsh uses a "lock file" called
tsh.lock to keep track of when it is running.  This copy of tsh
was unable to get access to the lock file.  The most likely reason
for this is that tsh is already in use.
EOF
  exit 1;
  }

=item Main;

Main entry point for tsh script.

=cut

sub Main () {
  srand; # harmless pre-5.004 legacy support

  DebugOn('CBPF');
  DebugOn('CP');
  DebugOn('GIB');
  ReopenConsole if $^O eq 'MacOS'; # harmless legacy support
  my $dir = @::ARGV ? shift @::ARGV : undef;
  my $tournament = new TSH::Tournament($dir);
  {
    my $error = $tournament->Lock();
    LockFailed $error if $error;
  }
  eval { $tournament->LoadConfiguration(); };
  if ($@) {
    unless ($tournament->TellUser('econfigerr', $@)) {
      print "Configuration error: $@\n";
      print "Press enter (return) to exit.\n";
      }
    scalar(<>);
    exit 1;
    }
  # must load processor after configuration
  my $processor = new TSH::Processor($tournament);
  my $config = $tournament->Config();
  $tournament->TellUser('iwelcome', $gkVersion);
  $tournament->LoadDivisions();
  if ($tournament->CountDivisions() == 0) {
    $tournament->TellUser('enodivs');
    exit(1);
    }
  $config->DivisionsLoaded();
  $config->Export(); # deprecated, but must come after DivisionsLoaded
  my $event_name = $config->Value('event_name');
  if (defined $event_name) {
    my $date = $config->Value('event_date');
    my $namedate = $event_name;
    $namedate .= ", " . $date if defined $date;
    $tournament->TellUser('ievtname', $namedate);
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/ && -t STDOUT) {
      print "\e]0;tsh $gkVersion - $event_name\a";
      }
    }
  if ($config->Value('port')) { $processor->RunServer(); }
  else { $processor->RunInteractive(); }
  $tournament->Unlock();
  if (defined $event_name) {
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/ && -t STDOUT) {
      print "\e]0;\a";
      }
    }
  }

sub Prompt () { 
  TSH::Utility::PrintColour 'yellow on blue', 'tsh>';
  print ' ';
  }

sub ReopenConsole () {
  close(STDOUT);
  close(STDERR);
  close(STDIN);
  open(STDOUT, "+>dev:console:tsh console") || die;
  open(STDERR, "+>dev:console:tsh console") || die;
  open(STDIN, "<dev:console:tsh console") || die;
  $| = 1;
  }

=item Use $module or die

Carefully evaluate "use $module" and try to explain to a naive user
the meaning of failure.

=cut

sub Use ($) {
  my $module = shift;
  eval "use $module";
  die "Couldn't load $module, most likely because your tsh software distribution\nis incomplete.  Please download it again.  Here is what Perl says caused\nthe problem:\n$@" if $@;
  return 1;
  }

=back

=cut

END { 
  sleep 10 if $^O eq 'MSWin32'; # to prevent error messages from disappearing
  }

## main code
Main;

