#!/usr/bin/perl

# Copyright (C) 2008-2009 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package Ratings;

# use Ratings::Event;

use strict;
use warnings;
use Carp;

=head1 SYNOPSIS

  use Ratings;

  my $rs = new Ratings('homan_state_filename' => $filename) or die;

  my $rs = new Ratings('logistic_state_filename' => $filename) or die;

  Ratings::EventIterator({
    'start' => [$year1, $month1, $event1],
    'end' => [$year2, $month2, $event2],
    'handlers' => {
      'monthly' => $sub1,
      'delete' => $sub2,
      'rename' => $sub3,
      'ort' => $sub4,  
      'lct' => $sub5,
      },
     });
  
=head1 ABSTRACT

This Perl module represents the state of a (Scrabble, so far) tournament rating system. 

=cut

=head1 DESCRIPTION

=head2 Methods

=cut

BEGIN {
  use Exporter ();
  use vars qw(
    $VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS
    );
  $VERSION = do { my @r = (q$Revision: 1.1 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r }; 
  @ISA = qw(Exporter);
  @EXPORT = qw();
  @EXPORT_OK = qw(
    );
  %EXPORT_TAGS = ();
  }

sub AddPlayer ($$);
sub BaseSystemName ($);
sub DeleteAllPlayers ($);
sub EventIterator ($);
sub LoadHomanStateFile ($);
sub LoadLogisticStateFile ($);
sub LoadNSAStateFile ($);
sub new ($@);
sub ProcessSpecialEvent ($$);
sub ProcessTournament ($$;$);
sub SaveHomanStateFile ($$);
sub SaveLogisticStateFile ($$);

=over 4

=cut

sub AddPlayer ($$) {
  my $this = shift;
# my $pname = shift;
  my $pp = shift;
  Carp::confess if @_;
# Carp::cluck join(',',%$pp) if $pname =~ /RIDOUT MAT/;
  my $key = $this->PlayerKey($pp);
  if (exists $this->{'_players'}{$key}) {
    confess "Duplicate player: '$key'";
    }
  else {
#   warn "Added $pp->{'name'} as $key.\n";
    $this->{'_players'}{$key} = $pp;
    }
  }

=item $s = BaseSystemName($sf);

Some rating system names contain variant information that needs
to be stripped off to leave the key under which ratings data
is stored.  This function does that stripping.

=cut

sub BaseSystemName ($) {
  my $name = shift;
  $name =~ s/ .*//;
  return $name;
  }

sub DeleteAllPlayers ($) {
  my $this = shift;
  $this->{'_players'} = {};
  }

=item EventIterator(\%options);

Iterate over all events in the specified range, calling callbacks depending
on the type of event.  The range is given by the options

  - start => [ $year, $month, $event_id]
  - end => [ $year, $month, $event_id]

Events are numbered consecutively beginning with zero in each month,
and the zero event is the monthly ratings list issued at the beginning
of the month.

Other required options are

  - caller => calling object, to pass back on callbacks
  - handlers => reference to hash giving handlers to various events

  Handlers are passed the caller and the event.
    - monthly => \&sub (monthly ratings list)
    - rename => \&sub (rename player)
    - delete => \&sub (delete player)
    - ort => \&sub (open rated tourney cross-table)
    - lct => \&sub (local club tourney cross-table)
    - final => \&sub (final standings from multi-segment tourney))

=cut

sub EventIterator ($) {
  my $optionsp = shift;
  my ($current_year, $current_month, $current_event_id)
    = @{$optionsp->{'start'}};
  my ($final_year, $final_month, $final_event_id)
    = @{$optionsp->{'end'}};
  eval "use Ratings::Event";
  confess $@ if $@;
  while (
    (($current_year <=> $final_year) ||
    ($current_month <=> $final_month) ||
    ($current_event_id <=> $final_event_id)) <= 0) {
    if (my $event = new Ratings::Event({
      'type' => 'xtable',
      'year' => $current_year,
      'month' => $current_month,
      'id' => $current_event_id,
      })) {
      if (my $sub = $optionsp->{'handlers'}{$event->HandlerType()}) {
	&$sub($optionsp->{'caller'}, $event);
        }
      else {
#       warn "Ignoring $current_year-$current_month-$current_event_id";
        }
      $current_event_id++;
      }
    else {
#     warn "Did not find $current_year-$current_month-$current_event_id";
      $current_event_id = 0;
      if (12 < ++$current_month) {
	$current_month = 1;
	$current_year++;
	}
      }
    }
  }

=item $g = $rs->Games($player_key, $games);

Set or get a player's current game total.

=cut

sub Games($$;$) {
  my $this = shift;
  my $key = shift;
  my $new = shift;
  if (ref($key)) { $key = $this->PlayerKey($key); }
  Carp::confess $key unless exists $this->{'_players'}{$key};
  if (defined $new) {
    $this->{'_players'}{$key}{'games'} = $new;
    }
  return $this->{'_players'}{$key}{'games'};
  }

=item $rs->LoadStateFile();

Load the state of the rating system from the designated file.

=cut

sub LoadStateFile ($) {
  my $this = shift;
  my $type = $this->{'_type'};
  if ($type eq 'homan') { $this->LoadHomanStateFile(); }
  elsif ($type eq 'logistic') { $this->LoadLogisticStateFile(); }
  elsif ($type eq 'nsa') { $this->LoadNSAStateFile(); }
  else { die "Unknown type: $type"; }
  }

=item $rs->LoadHomanStateFile();

Load the state of the rating system from a Homan file, or confess failure.
Internal use only.

=cut

sub LoadHomanStateFile ($) {
  my $this = shift;
  my $filename = $this->{'homan_state_filename'};
  eval "use Ratings::Player";
  confess $@ if $@;

  if (open my $fh, '<', $filename) {
    binmode $fh, ':encoding(isolatin1)';
    while (<$fh>) {
      chomp;
      confess "Can't parse (1): '$_'" unless length($_) == 51;
      my %param;
      @param{qw(name rating games peak_rating)} = unpack('A36A5A5A5', $_);
      for my $key (qw(rating games peak_rating)) { 
	$param{$key} =~ s/^(?:\s+)(\d+)$/$1/
	  or confess "Can't parse (2: $key): '$param{$key}'";
        }
      if ($param{'name'} =~ s/=(.*)//) {
	$param{'membership_number'} = $1;
        }
      $this->AddPlayer($param{'name'}, Ratings::Player->new(%param));
      }
    }
  else {
    confess "Error opening $filename: $!";
    }
  }

=item $rs->LoadLogisticStateFile();

Load the state of the rating system from a logistic file, or confess failure.
Internal use only.

=cut

sub LoadLogisticStateFile ($) {
  my $this = shift;
  my $filename = $this->{'logistic_state_filename'};
  eval "use Ratings::Player";
  confess $@ if $@;

  if (open my $fh, '<:encoding(isolatin1)', $filename) {
#   warn $filename;
    while (<$fh>) {
      chomp;
#     warn $_;
      my %param;
      @param{qw(name rating games)} = split(/\t/);
      die "Can't parse: '$_'" unless (defined $param{'games'}) && $param{'games'} =~ /^\d+$/;
      die "Can't parse: '$_'" unless $param{'rating'} =~ /^\d+$/;
      $this->AddPlayer(Ratings::Player->new(%param));
      }
    }
  else {
    confess "Error opening $filename: $!";
    }
  }

=item $rs->LoadNSAStateFile();

Load the state of the rating system from an NSA file, or confess failure.
Internal use only.

=cut

sub LoadNSAStateFile ($) {
  my $this = shift;
  my $filename = $this->{'nsa_state_filename'};
  eval "use Ratings::Player";
  confess $@ if $@;

  if (open my $fh, '<', $filename) {
    binmode $fh, ':encoding(isolatin1)';
#   warn $filename;
    while (<$fh>) {
      chomp;
#     warn $_;
      my %param;
      @param{qw(name rating rank expiry games)} = split(/\t/);
      die "Can't parse: '$_'" unless $param{'games'} && $param{'games'} =~ /^\d+$/;
      die "Can't parse: '$_'" unless $param{'rating'} =~ /^\d+$/;
      $this->AddPlayer(Ratings::Player->new(%param));
      }
    }
  else {
    confess "Error opening $filename: $!";
    }
  }

=item $key = $rs->PlayerKey($pp);

Compute the internal key corresponding to a TSH player object.

=cut

sub PlayerKey ($$) {
  my $this = shift;
  my $pp = shift;
  my $key = uc $pp->Name();
  Carp::confess join(';', %$pp) unless defined $key;
  $key =~ s/, / /g;
  if (defined $pp->{'membership_number'}) {
    $key .= "\n$pp->{'membership_number'}";
    }
  else {
    # thought about doing the following at first
#   $key =~ s/:([A-Z][A-Z]\d{6})$/\n$1/;
    # we currently use the following, because otherwise a player might have two keys, one with and without the membership number
    $key =~ s/:([A-Z][A-Z]\d{6})$//;
    # eventually need to make the key be the membership number, looked up if necessary
    }
  return $key;
  }

=item $rs->ProcessSpecialEvent($xml);

Update the rating system to include the results of a special event,
such as a player renaming or deletion.

=cut

sub ProcessSpecialEvent ($$) {
  my $this = shift;
  my $xml = shift;
  eval 'use RatingEvent';
  die $@ if $@;
# warn "PSE $xml";
  &RatingEvent::ProcessEvents($xml, $this->{'_players'});
# warn join("\n", grep { /EDWARDS/ } keys %{$this->{'_players'}});
  }

=item $rs->ProcessTournament($tournament[, $type]);

Update the rating system to include the results of a tournament.

If a rating system type is specified, it overrides the default type configured
for the tournament.

=cut

sub ProcessTournament ($$;$) {
  my $this = shift;
  my $tourney = shift;
  my $type = shift;

  unless ($tourney) { eval 'use Carp'; &confess("no tourney"); }
  my $config = $tourney->Config();
  $type ||= $config->Value('rating_system');
  my $basetype = BaseSystemName($type);
  for my $dp ($tourney->Divisions()) {
    $dp->LoadSupplementaryRatings($this, $type);
    $dp->ComputeSupplementaryRatings($type);
    for my $pp ($dp->Players()) {
      $this->Rating($pp, $pp->SupplementaryRatingsData($basetype, 'new'), $pp->Name());
      if (my $games = $pp->GamesPlayed()) {
	$this->Games($pp, ($this->Games($pp)||0)+$games);
	}
      }
    }
  }

=item $rs = new Ratings(%argv);

Create a new object representing a ratings system,
confess if the object cannot be created.
Required arguments: type.
Optional arguments: homan_state_filename logistic_state_filename nsa_state_filename.

=cut

sub new ($@) { 
  my $proto = shift;
  my (%argv) = @_;
  my $class = ref($proto) || $proto;
  my $this = { 
    '_type' => 'none',
    };
  bless($this, $class);
  for my $required (qw()) {
    unless (exists $argv{$required}) {
      confess "Missing required argument: $required";
      }
    $this->{$required} = $argv{$required};
    }
  for my $optional (qw(homan_state_filename logistic_state_filename nsa_state_filename)) {
    $this->{$optional} = $argv{$optional};
    }
  if ($this->{'homan_state_filename'}) {
    $this->{'_type'} = 'homan';
    $this->LoadHomanStateFile();
    }
  if ($this->{'logistic_state_filename'}) {
    $this->{'_type'} = 'logistic';
    $this->LoadLogisticStateFile();
    }
  if ($this->{'nsa_state_filename'}) {
    $this->{'_type'} = 'nsa';
    $this->LoadNSAStateFile();
    }
  return $this;
  }

=item $r = $rs->Rating($player_key[, $rating[, $name]]);

Set or get a player's current rating.

=cut

sub Rating($$;$$) {
  my $this = shift;
  my $key = shift;
  my $new = shift;
  my $name = shift || $key;
  if (ref($key)) { $key = $this->PlayerKey($key); }
  my $p = $this->{'_players'}{$key};
  if (!defined $p) {
    $this->AddPlayer(
      ($p = Ratings::Player->new(
      'name' => $name,
      'peak_rating' => ($new||0),
      'rating' => ($new||0),
      'games' => 0,
      )));
    }
  elsif (defined $new) {
    $p->{'rating'} = $new;
#   confess "$p->{'name'}: ".join(',',%$p)  unless defined $p->{'peak_rating'};
    $p->{'peak_rating'} = $p->{'rating'} 
      if (!defined $p->{'peak_rating'}) || $p->{'peak_rating'} < $p->{'rating'};
    }
  return $p->{'rating'};
  }

=item $rs->SaveStateFile($fn);

Save the state of the rating system from to the designated file.

=cut

sub SaveStateFile ($$) {
  my $this = shift;
  my $fn = shift;
  my $type = $this->{'_type'};
  if ($type eq 'homan') { $this->SaveHomanStateFile($fn); }
  elsif ($type eq 'logistic') { $this->SaveLogisticStateFile($fn); }
  else { die "Unknown type: $type"; }
  }

=item $rs->SaveHomanStateFile($filename);

Save the state of the rating system from to a Homan file, or confess failure.
Internal use only.

=cut

sub SaveHomanStateFile ($$) {
  my $this = shift;
  my $filename = shift;
  eval "use Ratings::Player";
  confess $@ if $@;

  if (open my $fh, '>', $filename) {
    binmode $fh, ':encoding(isolatin1)';
    my $psp = $this->{'_players'};
    for my $key (sort keys %$psp) {
      my $p = $psp->{$key};
      my $name = $p->Get('name');
      my $memnum = $p->Get('membership_number');
      $name .= "=$memnum" if defined $memnum;
      my $rating = $p->Get('rating');
      my $games = $p->Get('games');
      my $peak_rating = $p->Get('peak_rating');
#     confess unless defined $p->{'rating'};
#     confess unless defined $p->{'games'};
#     confess unless defined $p->{'peak_rating'};
#     confess unless defined $filename;
#     confess $key unless defined $name;
      printf $fh "%-36s%5s%5s%5s\n", $name, $rating, $games, $peak_rating
        or confess "Error writing to $filename: $!";
      }
    close $fh;
    }
  else {
    confess "Error creating $filename: $!";
    }
  }

=item $rs->SaveLogisticStateFile($filename);

Save the state of the rating system from to a logistic file, or confess failure.
Internal use only.

=cut

sub SaveLogisticStateFile ($$) {
  my $this = shift;
  my $filename = shift;
  eval "use Ratings::Player";
  confess $@ if $@;

  if (open my $fh, '>', $filename) {
    binmode $fh, ':encoding(isolatin1)';
    my $psp = $this->{'_players'};
    for my $key (sort keys %$psp) {
      my $p = $psp->{$key};
#     my $name = $p->Get('name');
      my $name = $key;
      my $memnum = $p->Get('membership_number');
      my $rating = $p->Get('rating');
      my $games = $p->Get('games');
      Carp::confess $name if $name =~ /\n/;
#     warn $name if $name =~ /EDWARDS G/;
      if ($rating || $games) {
	print $fh join("\t", $name, $rating, $games), "\n"
	  or confess "Error writing to $filename: $!";
        }
      else {
	warn "Not writing null user $name to $filename.";
        }
      }
    close $fh;
    }
  else {
    confess "Error creating $filename: $!";
    }
  }

=item $type = $rs->Type();

Return a string identifying the rating system type. 

=cut

sub Type ($) {
  my $this = shift;
  return $this->{'type'};
  }

=back

=cut

1;
