#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::HighCombined;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::HighCombined - implement the C<tsh> HighCombined command

=head1 SYNOPSIS

  my $command = new TSH::Command::HighCombined;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::HighCombined is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list the high combined scores in a division.
If you specify an integer, it indicates the number of scores you
would like to see.  If not, 20 scores are shown.
EOF
  $this->{'names'} = [qw(hc highcombined)];
  $this->{'argtypes'} = [qw(OptionalInteger Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = pop @_;
  my $nscores = @_ ? shift @_ : 20;
  my $config = $tournament->Config();

  my $has_classes = $dp->Classes() || defined $dp->Player(1)->Class();

  my $logp = new TSH::Log( $tournament, $dp, 'highcomb', undef,
    {'titlename' => $config->Terminology('highcomb')});
  my $entriesp = TSH::Division::FindExtremeGames::Search($dp, $nscores, 
    sub ($) { (defined $_[0][0]) && (defined $_[0][1]) && (($_[0][0] <=> $_[0][1]) || $_[0][2]->ID() > $_[0][3]->ID()) > 0 }, # count each game once
    sub ($$) { $_[1][0]+$_[1][1] <=> $_[0][0]+$_[0][1] }, # big totals first
    );
  my @classes;
  my @html_titles;
  my @text_titles;
  @classes = qw(round score score score);
  push(@html_titles, 
    $config->Terminology('Round'),
    $config->Terminology('Combined_Score'),
    $config->Terminology('Winning_Score'),
    $config->Terminology('Losing_Score'),
    );
  push(@text_titles, 
    $config->Terminology('Rnd'),
    $config->Terminology('CombinedS'),
    $config->Terminology('WinningS'),
    $config->Terminology('LosingS'),
    );
  my $class_term;
  if ($has_classes) {
    push(@classes, 'class');
    $class_term = $config->Terminology('Class');
    push(@html_titles, $class_term);
    push(@text_titles, $class_term);
    }
  push(@classes, qw(name));
  my $s = $config->Terminology('Winner');
  push(@html_titles, $s);
  push(@text_titles, $s);
  if ($has_classes) {
    push(@classes, 'class');
    push(@html_titles, $class_term);
    push(@text_titles, $class_term);
    }
  $s = $config->Terminology('Loser');
  push(@classes, qw(name));
  push(@html_titles, $s);
  push(@text_titles, $s);

  $logp->ColumnClasses(\@classes);
  $logp->ColumnTitles(
    {
    'text' => \@text_titles,
    'html' => \@html_titles,
    }
    );
  for my $entry (@$entriesp) {
    my (@data) = ($entry->[4]+1, $entry->[0]+$entry->[1],
      $entry->[0], $entry->[1]);
    push(@data, $entry->[2]->Class()) if $has_classes;
    push(@data, $entry->[2]->TaggedName());
    push(@data, $entry->[3]->Class()) if $has_classes;
    push(@data, $entry->[3]->TaggedName());
    $logp->WriteRow(\@data);
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;

