#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::HighRoundWins;

use strict;
use warnings;

use TSH::Division::FindExtremeGames;
use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::HighRoundWins - implement the C<tsh> HighRoundWins command

=head1 SYNOPSIS

  my $command = new TSH::Command::HighRoundWins;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::HighRoundWins is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to list the high winning scores in a division
in one or more rounds.
EOF
  $this->{'names'} = [qw(hrw highroundwins)];
  $this->{'argtypes'} = [qw(RoundRange Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($firstr1, $lastr1, $dp) = @_;
  return if $firstr1 > $lastr1;
  my $nscores = 1;

  my $logp = new TSH::Log( $tournament, $dp, 'highwin', "$firstr1-$lastr1",
    {'title' => 'Division ' . $dp->Name() . ' High Round Wins'});

  my @classes = qw(round score score);
  my @html_titles = qw(Round Winning<br>Score Losing<br>Score Winner Loser);
  my @text_titles = qw(Rd WS LS Winner Loser);

  $logp->ColumnClasses(\@classes);
  $logp->ColumnTitles( { 'text' => \@text_titles, 'html' => \@html_titles, }
    );
  my $sortsub = sub ($$) { $_[1][0] <=> $_[0][0] }; # big wins first
  for my $r1 ($firstr1..$lastr1) {
    my $r0 = $r1 - 1;
    #  games are represented as lists: [$score1, $score2, $p1, $p2, $r0, $rat1, $rat2].
    my $entriesp = TSH::Division::FindExtremeGames::Search($dp, $nscores+10, 
      sub ($) { 
	my $e = $_[0];
	$e->[4] == $r0 # round0 == $r0
	  && (defined $e->[0]) 
	  && (defined $e->[1]) 
	  && $e->[0] > $e->[1] # wins only
	},
      \&$sortsub,
      );
    TSH::Utility::DoOneRank(
      $entriesp, 
      $sortsub, 
      sub {
	my $e = shift;
	my $rank0 = shift;
	my $count = shift;
	$logp->WriteRow([$e->[4]+1, $e->[0], $e->[1], 
	  $e->[2]->TaggedName(), $e->[3]->TaggedName()]);
        },
      0,
      );
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known.

=cut


1;
