#!/usr/bin/perl

# Copyright (C) 2005-2010 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RANDomscores;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RANDomscores - implement the C<tsh> RANDomscores command

=head1 SYNOPSIS

  my $command = new TSH::Command::RANDomscores;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RANDomscores is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to add a round of random scores to a division
for testing purposes.
EOF
  $this->{'names'} = [qw(rand randomscores)];
  $this->{'argtypes'} = [qw(Divisions)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();
  my $track_firsts = $config->Value('track_firsts');
  my $scores = uc ($config->Value('scores')||'scrabble');

  for my $dp (@_) {
    my $distrib = $scores eq 'scrabble' 
      ? $dp->RatingSystem() =~ /^absp$/i ? 'absp' : 'nsa'
      : $scores;
    $tournament->TellUser('irandok', $dp->Name());
    my $round1 = $dp->LeastScores() + 1;
    for my $p ($dp->Players()) {
      my $nscores = $p->CountScores();
      next if $nscores >= $round1;
      next if $nscores >= $p->CountOpponents();
      if ($p->{'pairings'}[$nscores]) {
	my $rating = $p->Rating();
	if ($distrib eq 'WL') {
	  my $opp = $p->Opponent($nscores);
	  my $rdiff = $rating - $opp->Rating();
	  my $pwin = rand(1) < ($rdiff/400 + 0.5);
	  $p->Score($nscores, $pwin ? 2 : 0);
	  $opp->Score($nscores, $pwin ? 0 : 2);
  	  }
	else {
	  push(@{$p->{'scores'}}, 
	    # based on the NSA Club #3 2004-2005 season
	    $distrib eq 'nsa' ? int(130*log(($rating||1000)/75) +rand(200)-100)
	    # based on nothing
	    : int(250+rand(50+($rating>200?200:$rating))));
	  }
	if ($track_firsts) {
	  my $r0 = $#{$p->{'scores'}};
	  my $first = int(rand(2));
	  my $p12p = $p->{'etc'}{'p12'};
	  if ($p12p->[$r0] != 1 && $p12p->[$r0] != 2) {
	    $p12p->[$r0] = 1 + $first;
	    }
	  $p12p = $p->Opponent($r0)->{'etc'}{'p12'};
	  if ($p12p->[$r0] != 1 && $p12p->[$r0] != 2) {
	    $p12p->[$r0] = 2 - $first;
	    }
	  }
	}
      else {
	push(@{$p->{'scores'}}, $config->Value('bye_spread')||0);
        }
      printf "%s: @{$p->{'scores'}}\n", (TSH::Utility::TaggedName $p);
      }
    if (my $cmds = $config->Value('hook_division_complete')->{$dp->Name()}) {
      $this->Processor()->Process($cmds, 'nohistory');
      }
    $dp->Dirty(1);
    $this->Processor()->Flush();
    }
  $tournament->TellUser('idone');
  }

=back

=cut

1;
